// Trendyol Kategori Tarayıcı — Background Service Worker

// ═══════════════════════════════════════════════════════════════════
//  WEB PANEL SYNC — Chrome extension'dan web panele veri gönderme
//  socialProof (ORDER_COUNT, BASKET_COUNT) verisi SSR'da mevcut değildir.
//  Sadece client-side'da (extension) erişilebilir.
//  Bu fonksiyon extension'dan toplanan verileri web panele POST eder.
// ═══════════════════════════════════════════════════════════════════

const DEFAULT_PANEL_URL = "https://app.tpro360.com";

// ═══════════════════════════════════════════════════════════════════
//  PANEL AUTH STATE — Extension'dan panel'e login/logout
//  chrome.storage.local'da panelAuth objesi tutulur:
//  { token, userId, email, name, plan }
// ═══════════════════════════════════════════════════════════════════

async function getPanelAuth() {
    const data = await chrome.storage.local.get("panelAuth");
    return data.panelAuth || null;
}

async function setPanelAuth(authData) {
    await chrome.storage.local.set({ panelAuth: authData });
}

async function clearPanelAuth() {
    await chrome.storage.local.remove("panelAuth");
    // Panel tracked ID cache'ini de temizle
    panelTrackedCache.ids = new Set();
    panelTrackedCache.lastFetch = 0;
}

// Panel tracked IDs cache — 60sn TTL
const panelTrackedCache = {
    ids: new Set(),
    lastFetch: 0,
    TTL: 60000, // 60sn
};

async function refreshPanelTrackedIds() {
    const auth = await getPanelAuth();
    if (!auth || !auth.token) {
        panelTrackedCache.ids = new Set();
        return panelTrackedCache.ids;
    }

    // TTL kontrolu — cache tazeyse tekrar cekme
    if (Date.now() - panelTrackedCache.lastFetch < panelTrackedCache.TTL) {
        return panelTrackedCache.ids;
    }

    try {
        const panelUrl = await getPanelUrl();
        const resp = await fetch(`${panelUrl}/api/extension/products`, {
            headers: {
                "Authorization": `Bearer ${auth.token}`,
                "Content-Type": "application/json",
            },
            signal: AbortSignal.timeout(5000),
        });
        if (resp.ok) {
            const data = await resp.json();
            panelTrackedCache.ids = new Set((data.products || []).map(p => String(p.id)));
            panelTrackedCache.lastFetch = Date.now();
            console.log(`[TK-PANEL] Tracked IDs refreshed: ${panelTrackedCache.ids.size} urun`);
        } else if (resp.status === 401) {
            // Token suresi dolmus — auth temizle
            console.warn("[TK-PANEL] Token suresi dolmus, auth temizleniyor");
            await clearPanelAuth();
        }
    } catch (e) {
        console.log(`[TK-PANEL] Tracked IDs refresh hatasi: ${e.message}`);
    }
    return panelTrackedCache.ids;
}

// Panel'den kullanıcının ürünlerini çek ve stockTracking'e kaydet
// Login sonrası veya extension yeniden kurulduktan sonra çağrılır
async function syncProductsFromPanel() {
    const auth = await getPanelAuth();
    if (!auth || !auth.token) return;

    try {
        const panelUrl = await getPanelUrl();
        const resp = await fetch(`${panelUrl}/api/extension/products`, {
            headers: {
                "Authorization": `Bearer ${auth.token}`,
                "Content-Type": "application/json",
            },
            signal: AbortSignal.timeout(10000),
        });
        if (!resp.ok) {
            if (resp.status === 401) {
                console.warn("[TK-PANEL] Token süresi dolmuş, auth temizleniyor");
                await clearPanelAuth();
            }
            return;
        }

        const data = await resp.json();
        const panelProducts = data.products || [];
        if (panelProducts.length === 0) return;

        // Mevcut stockTracking'i al
        const stData = await chrome.storage.local.get("stockTracking");
        const tracking = stData.stockTracking || { products: {}, settings: {} };
        if (!tracking.products) tracking.products = {};

        let added = 0;
        for (const pp of panelProducts) {
            const pid = String(pp.id);
            // Zaten varsa atla (mevcut tracking verisini korumak için)
            if (tracking.products[pid]) {
                // Sadece panelEstimate ve rank güncelle
                if (pp.panelEstimate) {
                    tracking.products[pid].panelEstimate = pp.panelEstimate;
                }
                tracking.products[pid].dataQualityRank = pp.dataQualityRank || 1;
                continue;
            }

            // Yeni skeleton ürün ekle — extension stok okumaya başlayacak
            tracking.products[pid] = {
                id: pid,
                name: pp.name || "",
                slug: pp.slug || "",
                brand: pp.brand || "",
                price: pp.currentPrice || 0,
                imageUrl: pp.imageUrl || null,
                lastStock: pp.currentStock || 0,
                addedAt: new Date().toISOString(),
                lastFetchAt: null,
                status: "active",
                readings: [],
                events: [],
                dailySummaries: {},
                merchants: [],
                estimates: {},
                confidence: {
                    level: pp.confidenceLevel || 0,
                    label: pp.confidenceLabel || "Panel",
                    daysTracked: pp.daysTracked || 0,
                    readings: 0,
                    saleEvents: 0,
                },
                emaDailySales: pp.emaDailySales || 0,
                stockTrackingMode: pp.stockTrackingMode || "delta",
                panelEstimate: pp.panelEstimate || null,
                dataQualityRank: pp.dataQualityRank || 1,
            };
            added++;
        }

        if (added > 0 || panelProducts.length > 0) {
            await chrome.storage.local.set({ stockTracking: tracking });
            console.log(`[TK-PANEL] Panel sync: ${added} yeni ürün eklendi, ${panelProducts.length} toplam`);
        }

        // Yeni eklenen ürünler için hemen stok çekmeyi başlat
        if (added > 0) {
            fetchStockForAllTrackedProducts().catch(e => {
                console.warn("[TK-PANEL] İlk stok çekme hatası:", e.message);
            });
        }
    } catch (e) {
        console.warn("[TK-PANEL] Panel sync hatası:", e.message);
    }
}

// Panel'e urun track/untrack gonder
async function panelTrackProduct(productInfo) {
    const auth = await getPanelAuth();
    if (!auth || !auth.token) return null;

    try {
        const panelUrl = await getPanelUrl();
        const resp = await fetch(`${panelUrl}/api/extension/track`, {
            method: "POST",
            headers: {
                "Authorization": `Bearer ${auth.token}`,
                "Content-Type": "application/json",
            },
            body: JSON.stringify({
                action: "track",
                productId: productInfo.id,
                slug: productInfo.slug,
                name: productInfo.name || "",
                brand: productInfo.brand || "",
                imageUrl: productInfo.imageUrl || "",
                categoryPath: productInfo.categoryPath || "",
                price: productInfo.price || 0,
                stock: productInfo.stock || productInfo.lastStock || 0,
                merchantId: productInfo.merchantId || productInfo.lastBuyboxMerchantId || undefined,
                otherMerchantIds: productInfo.otherMerchantIds || undefined,
                merchantName: productInfo.merchantName || undefined,
                otherMerchantNames: productInfo.otherMerchantNames || undefined,
                reviewCount: productInfo.reviewCount || undefined,
                favoriteCount: productInfo.favoriteCount || undefined,
            }),
            signal: AbortSignal.timeout(8000),
        });
        if (resp.ok) {
            const result = await resp.json();
            console.log(`[TK-PANEL] Track OK: ${productInfo.id} — ${result.status}`);
            // Cache guncelle
            panelTrackedCache.ids.add(String(productInfo.id));
            return result;
        } else {
            const err = await resp.json().catch(() => ({}));
            console.warn(`[TK-PANEL] Track HTTP ${resp.status}: ${err.error || ""}`);
            if (resp.status === 401) {
                console.warn("[TK-PANEL] Token süresi dolmuş, auth temizleniyor");
                await clearPanelAuth();
                return null;
            }
            // 403 = plan limiti aşıldı
            if (resp.status === 403 && err.limitReached) {
                return { limitReached: true, error: err.error || "Ürün takip limitine ulaştınız" };
            }
            return null;
        }
    } catch (e) {
        console.log(`[TK-PANEL] Track hatasi: ${e.message}`);
        return null;
    }
}

async function panelUntrackProduct(productId) {
    const auth = await getPanelAuth();
    if (!auth || !auth.token) return null;

    try {
        const panelUrl = await getPanelUrl();
        const resp = await fetch(`${panelUrl}/api/extension/track`, {
            method: "POST",
            headers: {
                "Authorization": `Bearer ${auth.token}`,
                "Content-Type": "application/json",
            },
            body: JSON.stringify({
                action: "untrack",
                productId: String(productId),
            }),
            signal: AbortSignal.timeout(5000),
        });
        if (resp.ok) {
            console.log(`[TK-PANEL] Untrack OK: ${productId}`);
            panelTrackedCache.ids.delete(String(productId));
        }
    } catch (e) {
        console.log(`[TK-PANEL] Untrack hatasi: ${e.message}`);
    }
}

// Panel URL — kodda tanımlı, değiştirmek için DEFAULT_PANEL_URL sabitini güncelle
function getPanelUrl() {
    return DEFAULT_PANEL_URL;
}

// ═══════════════════════════════════════════════════════════════════
//  SHIPPING CONFIG SYNC — Panel'den güncel kargo fiyatları + desi defaults
//  /api/extension/config endpoint'inden alınan veriler chrome.storage.local'a yazılır.
//  Content script init sırasında bu verileri okuyup calculator'a aktarır.
//  Periyodik güncelleme: auth sonrası + 6 saatte bir (chrome.alarms)
// ═══════════════════════════════════════════════════════════════════

async function fetchShippingConfig() {
    const auth = await getPanelAuth();
    if (!auth || !auth.token) {
        console.log("[TK-CONFIG] Auth yok — config fetch atlanıyor");
        return null;
    }

    try {
        const panelUrl = await getPanelUrl();
        const resp = await fetch(`${panelUrl}/api/extension/config`, {
            headers: {
                "Authorization": `Bearer ${auth.token}`,
                "Content-Type": "application/json",
            },
            signal: AbortSignal.timeout(8000),
        });

        if (!resp.ok) {
            console.log("[TK-CONFIG] Config fetch başarısız:", resp.status);
            return null;
        }

        const config = await resp.json();

        // chrome.storage.local'a kaydet — content script'ler buradan okur
        await chrome.storage.local.set({
            shippingConfig: {
                shipping: config.shipping,
                desiDefaults: config.desiDefaults,
                kdvRate: config.kdvRate,
                defaultCarrier: config.defaultCarrier,
                fetchedAt: Date.now(),
            }
        });

        console.log("[TK-CONFIG] Shipping config güncellendi:",
            config.shipping?.barem?.length, "barem tier,",
            config.shipping?.weight?.length, "weight tier,",
            config.desiDefaults?.length, "desi pattern");

        return config;
    } catch (e) {
        console.log("[TK-CONFIG] Config fetch hatası:", e.message);
        return null;
    }
}

// Periyodik config güncelleme (6 saatte bir)
chrome.alarms.create("shipping-config-refresh", { periodInMinutes: 360 });

chrome.alarms.onAlarm.addListener((alarm) => {
    if (alarm.name === "shipping-config-refresh") {
        fetchShippingConfig().catch(e =>
            console.log("[TK-CONFIG] Alarm refresh hatası:", e.message)
        );
    }
});

// ─── Hafif SocialProof Cache Sync ───
// Listing sayfasindaki TUM urunlerin socialProof verilerini panele gonder.
// Full sync'ten (/api/extension/sync) cok daha hafif — sadece cache yazma.
// Research route bu cache'i kullanarak satis tahminlerini iyilestirir.
async function syncSocialProofCache(items) {
    if (!items || items.length === 0) return;
    try {
        const panelUrl = await getPanelUrl();
        await fetch(`${panelUrl}/api/extension/socialproof`, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ products: items }),
            signal: AbortSignal.timeout(5000),
        });
    } catch (e) {
        // Silent — cache hatasi kritik degil
        console.log("[TK-SYNC] socialproof cache error:", e.message);
    }
}

// Extension'dan toplanan socialProof verisini web panele gönder
// products: [{productId, socialProof, stock, price, reviewCount, ...}]
async function syncToPanel(products) {
    if (!products || products.length === 0) return;
    try {
        const panelUrl = await getPanelUrl();
        const auth = await getPanelAuth();
        const headers = { "Content-Type": "application/json" };
        // Auth varsa header ekle
        if (auth && auth.token) {
            headers["Authorization"] = `Bearer ${auth.token}`;
        }
        const url = `${panelUrl}/api/extension/sync`;
        const resp = await fetch(url, {
            method: "POST",
            headers,
            body: JSON.stringify({ products })
        });
        if (resp.ok) {
            const result = await resp.json();
            console.log(`[TK-SYNC] Panel sync OK: ${result.updated} güncellendi, ${result.skipped} atlandı`);

            // Panel'den dönen panelEstimate'leri stockTracking'e kaydet
            if (result.results && result.results.length > 0) {
                const hasEstimates = result.results.some(r => r.panelEstimate);
                if (hasEstimates) {
                    try {
                        const stData = await chrome.storage.local.get("stockTracking");
                        const tracking = stData.stockTracking;
                        if (tracking && tracking.products) {
                            let changed = false;
                            for (const r of result.results) {
                                if (tracking.products[r.id]) {
                                    if (r.panelEstimate) {
                                        tracking.products[r.id].panelEstimate = r.panelEstimate;
                                        changed = true;
                                    }
                                    // Veri kalitesi rank (1-10)
                                    if (r.dataQualityRank) {
                                        tracking.products[r.id].dataQualityRank = r.dataQualityRank;
                                        changed = true;
                                    }
                                    // Panel'deki socialProof'u kaydet (RECENT_SALES_COUNT, ORDER_COUNT vb.)
                                    // Extension calculator T1 hesabı için bu veriyi kullanacak
                                    if (r.socialProof) {
                                        tracking.products[r.id].panelSocialProof = r.socialProof;
                                        changed = true;
                                    }
                                }
                            }
                            if (changed) {
                                await chrome.storage.local.set({ stockTracking: tracking });
                                console.log("[TK-SYNC] panelEstimate kaydedildi");
                            }
                        }
                    } catch (e) {
                        console.warn("[TK-SYNC] panelEstimate kaydetme hatası:", e.message);
                    }
                }
            }

            // Kalibrasyon verilerini kaydet — extension calculator'ın sepet dönüşüm oranı için
            if (result.calibration) {
                try {
                    await chrome.storage.local.set({ panelCalibration: result.calibration });
                    console.log("[TK-SYNC] Kalibrasyon kaydedildi:", JSON.stringify(result.calibration));
                } catch (e) {
                    console.warn("[TK-SYNC] Kalibrasyon kaydetme hatası:", e.message);
                }
            }

            // ─── SCRAPE QUEUE: Panel'in worker-blocked ürünlerini extension'da tara ───
            if (result.scrapeQueue && Array.isArray(result.scrapeQueue) && result.scrapeQueue.length > 0) {
                console.log(`[TK-SYNC] Scrape queue alındı: ${result.scrapeQueue.length} ürün — ${result.scrapeQueue.map(q => q.productId).join(', ')}`);
                try {
                    await chrome.storage.local.set({ pendingScrapeQueue: result.scrapeQueue });
                } catch (e) {
                    console.warn("[TK-SYNC] Scrape queue kaydetme hatası:", e.message);
                }
                // Arka planda işlemeye başla (mevcut sync'i bloklamaz)
                processScrapeQueue(result.scrapeQueue).catch(e =>
                    console.warn("[TK-SYNC] Scrape queue işleme hatası:", e.message));
            }
        } else if (resp.status === 401) {
            console.warn("[TK-SYNC] Token süresi dolmuş, auth temizleniyor");
            await clearPanelAuth();
        } else if (resp.status === 429) {
            const errData = await resp.json().catch(() => ({}));
            console.warn(`[TK-SYNC] Günlük sync limiti aşıldı: ${errData.usage || '?'}/${errData.limit || '?'}`);
        } else {
            console.warn(`[TK-SYNC] Panel sync HTTP ${resp.status}`);
        }
    } catch (e) {
        // Panel kapalıysa veya bağlantı yoksa sessizce atla
        console.log(`[TK-SYNC] Panel bağlantısı yok (${e.message}) — atlanıyor`);
    }
}

// ═══════════════════════════════════════════════════════════════════
//  PROXY SCRAPING — Panel'in worker-blocked ürünlerini extension'da tara
//  Panel sync response'undaki scrapeQueue'dan gelen ürünleri,
//  fetchTopMerchantsStock ile çekip sonuçları panel'e geri gönderir.
//  Cloudflare 403 alan ürünler tarayıcı session'ı ile erişilebilir.
// ═══════════════════════════════════════════════════════════════════

let _scrapeQueueProcessing = false;

async function processScrapeQueue(queue) {
    if (_scrapeQueueProcessing) {
        console.log("[TK-PROXY] Scrape queue zaten işleniyor, atlanıyor");
        return;
    }
    if (!queue || queue.length === 0) return;

    _scrapeQueueProcessing = true;
    console.log(`[TK-PROXY] Scrape queue başlıyor: ${queue.length} ürün`);

    try {
        const auth = await getPanelAuth();
        if (!auth || !auth.token) {
            console.log("[TK-PROXY] Auth yok, scrape queue atlanıyor");
            return;
        }

        const results = [];
        const BATCH_SIZE = 2;     // 2 paralel fetch (Trendyol rate limit)
        const DELAY_MS = 1500;    // Batch arası 1.5s bekleme

        for (let i = 0; i < queue.length; i += BATCH_SIZE) {
            const batch = queue.slice(i, i + BATCH_SIZE);

            const promises = batch.map(async (item) => {
                try {
                    const product = {
                        id: item.productId,
                        slug: item.slug,
                        name: item.name || '',
                        price: 0,
                        merchants: [],
                    };

                    const stockResult = await fetchTopMerchantsStock(product);
                    if (!stockResult) {
                        console.warn(`[TK-PROXY] ❌ ${item.productId}: fetch başarısız`);
                        return;
                    }

                    console.log(`[TK-PROXY] ✅ ${item.productId}: stok=${stockResult.totalStock}, fiyat=${stockResult.price}`);

                    // Sync payload oluştur (mevcut syncToPanel formatında)
                    const syncItem = {
                        productId: String(item.productId),
                        slug: item.slug,
                        stock: stockResult.totalStock,
                        price: stockResult.price,
                        merchantId: stockResult.buyboxMerchantId || undefined,
                        otherMerchantStocks: {},
                    };

                    // merchantStocks'tan buybox haricindekileri ayır
                    if (stockResult.merchantStocks) {
                        for (const [mid, mstock] of Object.entries(stockResult.merchantStocks)) {
                            if (mid !== String(stockResult.buyboxMerchantId) && !mid.startsWith('_')) {
                                syncItem.otherMerchantStocks[mid] = mstock;
                            }
                        }
                    }

                    // SocialProof enrichment (apigw API — extension fetch ile)
                    try {
                        const sp = await fetchSocialProofFromApi(item.productId);
                        if (sp && Object.keys(sp).length > 0) {
                            syncItem.socialProof = sp;
                            console.log(`[TK-PROXY] 📊 ${item.productId}: socialProof keys=${Object.keys(sp).join(',')}`);
                        }
                    } catch (e) {
                        // socialProof hatası scrape'i etkilemesin
                    }

                    results.push(syncItem);
                } catch (e) {
                    console.warn(`[TK-PROXY] ❌ ${item.productId} hata: ${e.message}`);
                }
            });

            await Promise.all(promises);

            // Batch arası bekleme (rate limiting)
            if (i + BATCH_SIZE < queue.length) {
                await new Promise(resolve => setTimeout(resolve, DELAY_MS));
            }
        }

        // Sonuçları panel'e sync et (mevcut endpoint üzerinden)
        if (results.length > 0) {
            console.log(`[TK-PROXY] ${results.length}/${queue.length} başarılı, panel'e sync ediliyor`);
            await syncToPanel(results);
        } else {
            console.log("[TK-PROXY] Hiçbir ürün başarılı olmadı");
        }

        // Queue'yu temizle
        await chrome.storage.local.remove("pendingScrapeQueue");

    } catch (e) {
        console.error("[TK-PROXY] Kritik hata:", e.message);
    } finally {
        _scrapeQueueProcessing = false;
        console.log("[TK-PROXY] Scrape queue tamamlandı");
    }
}

// Trendyol kategori URL'leri (en çok satanlar sıralaması)
const CATEGORY_URLS = [
    { name: "Kadın Giyim", url: "https://www.trendyol.com/sr?q=&qt=kadın+giyim&st=kadın+giyim&os=1&sst=BEST_SELLER" },
    { name: "Erkek Giyim", url: "https://www.trendyol.com/sr?q=&qt=erkek+giyim&st=erkek+giyim&os=1&sst=BEST_SELLER" },
    { name: "Kadın Ayakkabı", url: "https://www.trendyol.com/sr?q=&qt=kadın+ayakkabı&st=kadın+ayakkabı&os=1&sst=BEST_SELLER" },
    { name: "Erkek Ayakkabı", url: "https://www.trendyol.com/sr?q=&qt=erkek+ayakkabı&st=erkek+ayakkabı&os=1&sst=BEST_SELLER" },
    { name: "Kadın Çanta", url: "https://www.trendyol.com/sr?q=&qt=kadın+çanta&st=kadın+çanta&os=1&sst=BEST_SELLER" },
    { name: "Elektronik", url: "https://www.trendyol.com/sr?q=&qt=elektronik&st=elektronik&os=1&sst=BEST_SELLER" },
    { name: "Ev & Mobilya", url: "https://www.trendyol.com/sr?q=&qt=ev+dekorasyon&st=ev+dekorasyon&os=1&sst=BEST_SELLER" },
    { name: "Kozmetik", url: "https://www.trendyol.com/sr?q=&qt=kozmetik&st=kozmetik&os=1&sst=BEST_SELLER" },
    { name: "Çocuk & Bebek", url: "https://www.trendyol.com/sr?q=&qt=çocuk+bebek&st=çocuk+bebek&os=1&sst=BEST_SELLER" },
    { name: "Spor & Outdoor", url: "https://www.trendyol.com/sr?q=&qt=spor+outdoor&st=spor+outdoor&os=1&sst=BEST_SELLER" },
    { name: "Süpermarket", url: "https://www.trendyol.com/sr?q=&qt=süpermarket&st=süpermarket&os=1&sst=BEST_SELLER" },
    { name: "Aksesuar & Saat", url: "https://www.trendyol.com/sr?q=&qt=aksesuar+saat&st=aksesuar+saat&os=1&sst=BEST_SELLER" },
    { name: "İç Giyim", url: "https://www.trendyol.com/sr?q=&qt=iç+giyim&st=iç+giyim&os=1&sst=BEST_SELLER" },
    { name: "Yapı Market & Bahçe", url: "https://www.trendyol.com/sr?q=&qt=yapı+market&st=yapı+market&os=1&sst=BEST_SELLER" },
    { name: "Kitap & Hobi", url: "https://www.trendyol.com/sr?q=&qt=kitap+hobi&st=kitap+hobi&os=1&sst=BEST_SELLER" },
    { name: "Otomotiv", url: "https://www.trendyol.com/sr?q=&qt=otomotiv&st=otomotiv&os=1&sst=BEST_SELLER" },
];

let scanState = {
    isScanning: false,
    currentIndex: 0,
    totalCategories: CATEGORY_URLS.length,
    allProducts: [],
    originTabId: null,
    scanTabId: null,
};

// ═══════════════════════════════════════════════════════════════════
//  ÜRÜN DETAY VERİSİ ÇEKME (T1 + Stok)
//  Content script'ten gelen ürün ID'leri için Trendyol arama API'sinden
//  socialProof (orderCount) ve detay sayfasından stok bilgisi çeker.
// ═══════════════════════════════════════════════════════════════════

// Trendyol SSR arama sayfasından __single-search-result__PROPS JSON'ını çıkar
function extractSearchPropsFromHTML(html) {
    const marker = "__single-search-result__PROPS";
    const idx = html.indexOf(marker);
    if (idx === -1) return null;

    // Script tag'ın başını ve sonunu bul
    const scriptStart = html.lastIndexOf("<script", idx);
    const scriptEnd = html.indexOf("</script>", idx);
    if (scriptStart === -1 || scriptEnd === -1) return null;

    const scriptContent = html.substring(scriptStart, scriptEnd);

    // window['__single-search-result__PROPS'] = { ... }; pattern'ını bul
    const assignIdx = scriptContent.indexOf(marker);
    const eqIdx = scriptContent.indexOf("=", assignIdx);
    if (eqIdx === -1) return null;

    let jsonStr = scriptContent.substring(eqIdx + 1).trim();
    // Sondaki ; karakterini kaldır
    if (jsonStr.endsWith(";")) jsonStr = jsonStr.slice(0, -1);

    try {
        return JSON.parse(jsonStr);
    } catch (e) {
        console.warn("[TK-BG] SSR props parse hatası:", e.message);
        return null;
    }
}

// Ürün URL slug'ından arama terimi oluştur
// Örn: "smarter/tuy-toplayici-12-li-yedek-rulo-aparat-p-7035867" → "smarter tuy toplayici 12 li"
function buildSearchQueryFromSlug(slug) {
    if (!slug) return null;
    // URL'den slug'ı ayıkla: /marka/urun-adi-p-ID formatı
    let parts = slug.split("/").filter(Boolean);
    // Son parça genellikle ürün adı + p-ID
    let productPart = parts[parts.length - 1] || "";
    // p-ID'yi kaldır
    productPart = productPart.replace(/-p-\d+.*$/, "");
    // Marka adını da ekle
    const brand = parts.length > 1 ? parts[parts.length - 2] : "";
    // Tire'leri boşlukla değiştir, ilk 5-6 kelimeyi al (çok uzun sorgu sonuç vermez)
    const words = (brand + " " + productPart).replace(/-/g, " ").split(/\s+/).filter(Boolean);
    return words.slice(0, 6).join(" ");
}

// SSR ürün detay sayfasından stok bilgisi çek
// slug: "/smarter/tuy-toplayici-12-li-yedek-rulo-aparat-p-7035867"
async function fetchStockFromDetailPage(slug) {
    try {
        // Cache-buster ekle — CDN cache'ini bypass et
        const cacheBuster = `_t=${Date.now()}`;
        const separator = slug.includes("?") ? "&" : "?";
        const url = `https://www.trendyol.com${slug}${separator}${cacheBuster}`;
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 8000);
        const resp = await fetch(url, {
            headers: {
                "Accept": "text/html",
                "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36"
            },
            cache: "no-store",
            signal: controller.signal
        });
        clearTimeout(timeoutId);
        if (!resp.ok) return 0;
        const html = await resp.text();

        const marker = "__envoy_product-detail__PROPS";
        const idx = html.indexOf(marker);
        if (idx === -1) return 0;

        const scriptStart = html.lastIndexOf("<script", idx);
        const scriptEnd = html.indexOf("</script>", idx);
        if (scriptStart === -1 || scriptEnd === -1) return 0;

        const scriptContent = html.substring(scriptStart, scriptEnd);
        const eqIdx = scriptContent.indexOf("=", scriptContent.indexOf(marker));
        if (eqIdx === -1) return 0;

        let jsonStr = scriptContent.substring(eqIdx + 1).trim();
        if (jsonStr.endsWith(";")) jsonStr = jsonStr.slice(0, -1);

        const data = JSON.parse(jsonStr);
        const p = data && data.product;
        if (p && p.merchantListing && p.merchantListing.winnerVariant) {
            return p.merchantListing.winnerVariant.quantity || 0;
        }
    } catch (e) {
        // silent — stok bilgisi alınamazsa 0 döner
    }
    return 0;
}

// ─── apigw API'den socialProof çek (hafif, alarm cycle için) ───
// Trendyol public API: basketCount, orderCount, favoriteCount döndürür
// camelCase key'ler döner → SCREAMING_SNAKE_CASE'e çevirilir
async function fetchSocialProofFromApi(productId) {
    try {
        const url = `https://apigw.trendyol.com/discovery-web-productgw-service/api/productDetail/${productId}`;
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 6000);
        const resp = await fetch(url, {
            headers: {
                "Accept": "application/json",
                "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36"
            },
            cache: "no-store",
            signal: controller.signal
        });
        clearTimeout(timeoutId);

        if (!resp.ok) return null;
        const data = await resp.json();

        // API response: { result: { socialProof: { basketCount: "504", orderCount: "1000+", ... } } }
        // Panel scraper (scraper.js:294) de ayni path'i kullanir: data?.result?.socialProof
        const socialProof = data?.result?.socialProof;

        // camelCase → SCREAMING_SNAKE_CASE
        const sp = {};
        if (socialProof && typeof socialProof === 'object') {
            // socialProof nested format olabilir: { basketCount: { count: "504" } } veya flat: { basketCount: "504" }
            const extract = (key) => {
                const val = socialProof[key];
                if (!val) return null;
                if (typeof val === 'object' && val.count !== undefined) return String(val.count);
                return String(val);
            };
            const basket = extract('basketCount');
            const order = extract('orderCount');
            const fav = extract('favoriteCount');
            const pageView = extract('pageViewCount');
            if (basket) sp.BASKET_COUNT = basket;
            if (order) sp.ORDER_COUNT = order;
            if (fav) sp.FAVORITE_COUNT = fav;
            if (pageView) sp.PAGE_VIEW_COUNT = pageView;
        } else {
            // Fallback: eski format (top-level) — bazı endpoint'ler farklı yapıda dönebilir
            if (data.basketCount) sp.BASKET_COUNT = String(data.basketCount);
            if (data.orderCount) sp.ORDER_COUNT = String(data.orderCount);
            if (data.favoriteCount) sp.FAVORITE_COUNT = String(data.favoriteCount);
            if (data.pageViewCount) sp.PAGE_VIEW_COUNT = String(data.pageViewCount);
        }

        return Object.keys(sp).length > 0 ? sp : null;
    } catch (e) {
        console.warn(`[TK-BG] apigw socialProof hata (${productId}):`, e.message);
        return null;
    }
}

// ═══════════════════════════════════════════════════════════════════
//  BULK SOCIAL PROOF FETCH — apigw API üzerinden toplu socialProof
//  Panel research sayfası için: sunucu IP 403 alıyor, extension bypass eder.
//  productIds: ["7035867", "12345678", ...]
//  Return: { "7035867": { ORDER_COUNT: "1000+", BASKET_COUNT: "500", ... }, ... }
// ═══════════════════════════════════════════════════════════════════
async function bulkFetchSocialProofFromApigw(productIds) {
    const results = {};
    const batchSize = 5;
    const delayMs = 300;
    let consecutive429 = 0;

    for (let i = 0; i < productIds.length; i += batchSize) {
        if (consecutive429 >= 3) {
            console.log(`[TK-SP] 429 limiti — kalan ${productIds.length - i} urun atlanıyor`);
            break;
        }

        const batch = productIds.slice(i, i + batchSize);
        let batchHas429 = false;

        const promises = batch.map(async (pid) => {
            try {
                const url = `https://apigw.trendyol.com/discovery-web-productgw-service/api/productDetail/${pid}`;
                const controller = new AbortController();
                const timer = setTimeout(() => controller.abort(), 8000);
                const resp = await fetch(url, {
                    headers: { 'Accept': 'application/json' },
                    signal: controller.signal
                });
                clearTimeout(timer);

                if (resp.status === 429) { batchHas429 = true; return; }
                if (!resp.ok) return;

                const data = await resp.json();
                const sp = data?.result?.socialProof;
                if (!sp) return;

                // socialProof normalize: camelCase → SCREAMING_SNAKE_CASE
                const normalized = {};
                if (sp.basketCount) normalized.BASKET_COUNT = String(sp.basketCount);
                if (sp.orderCount) normalized.ORDER_COUNT = String(sp.orderCount);
                if (sp.favoriteCount) normalized.FAVORITE_COUNT = String(sp.favoriteCount);
                if (sp.pageViewCount) normalized.PAGE_VIEW_COUNT = String(sp.pageViewCount);
                // Bazı ürünlerde zaten SCREAMING_SNAKE_CASE gelebilir
                if (sp.BASKET_COUNT) normalized.BASKET_COUNT = String(sp.BASKET_COUNT);
                if (sp.ORDER_COUNT) normalized.ORDER_COUNT = String(sp.ORDER_COUNT);
                if (sp.FAVORITE_COUNT) normalized.FAVORITE_COUNT = String(sp.FAVORITE_COUNT);

                if (Object.keys(normalized).length > 0) {
                    results[String(pid)] = normalized;
                }
            } catch { /* silent */ }
        });

        await Promise.allSettled(promises);

        if (batchHas429) {
            consecutive429++;
            const wait = 2000 * Math.pow(2, consecutive429 - 1);
            await new Promise(r => setTimeout(r, wait));
        } else {
            consecutive429 = 0;
        }

        if (i + batchSize < productIds.length) {
            await new Promise(r => setTimeout(r, delayMs));
        }
    }

    console.log(`[TK-SP] Bulk socialProof: ${Object.keys(results).length}/${productIds.length} urun basarili`);
    return results;
}

// Tek bir ürün için socialProof + stok verisi çek
// productId: "7035867"
// slug: "/smarter/tuy-toplayici-12-li-yedek-rulo-aparat-p-7035867"
// fetchStock: true ise detay sayfasından stok da çeker
async function fetchProductDetail(productId, slug, fetchStock = false) {
    let result = null;
    try {
        // Arama terimi oluştur
        const query = slug ? buildSearchQueryFromSlug(slug) : null;
        if (!query) {
            console.warn(`[TK-BG] fetchProductDetail: slug yok, atlanıyor (${productId})`);
            return null;
        }

        // Trendyol arama sayfasını SSR olarak fetch et → socialProof
        const searchUrl = `https://www.trendyol.com/sr?q=${encodeURIComponent(query)}`;
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 8000); // 8sn timeout
        const resp = await fetch(searchUrl, {
            headers: {
                "Accept": "text/html",
                "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36"
            },
            cache: "no-store",
            signal: controller.signal
        });
        clearTimeout(timeoutId);

        if (!resp.ok) return null;

        const html = await resp.text();
        const props = extractSearchPropsFromHTML(html);

        if (props && props.data && props.data.products) {
            const product = props.data.products.find(p => String(p.id) === String(productId));
            if (product) {
                const spObj = {};
                if (product.socialProof && Array.isArray(product.socialProof)) {
                    product.socialProof.forEach(sp => {
                        spObj[sp.key] = sp.value;
                    });
                }
                result = {
                    id: productId,
                    socialProof: spObj,
                    ratingScore: product.ratingScore || null,
                    price: product.price
                        ? product.price.sellingPrice || product.price.discountedPrice
                        : null,
                    stockQuantity: 0,
                };
            }
        }
    } catch (e) {
        console.warn(`[TK-BG] fetchProductDetail socialProof hata (${productId}):`, e.message);
    }

    // Stok bilgisi çek (ayrı fetch — detay sayfasından)
    // result null olsa bile stok çekilebilmeli (arama sonucu bulunamasa da stok var)
    if (fetchStock && slug) {
        try {
            const stock = await fetchStockFromDetailPage(slug);
            if (result) {
                result.stockQuantity = stock;
            } else {
                // Arama sonucunda bulunamadı ama stok çekilebildi
                result = {
                    id: productId,
                    socialProof: {},
                    ratingScore: null,
                    price: null,
                    stockQuantity: stock
                };
            }
        } catch (e) {
            // silent
        }
    }

    return result;
}

// Toplu ürün detay çekme (rate limit'e dikkat)
// products: [{id: "7035867", slug: "/smarter/tuy-toplayici-...-p-7035867"}, ...]
// fetchStock: true ise her ürün için detay sayfasından stok da çeker
async function fetchMultipleProductDetails(products, fetchStock = false) {
    const results = {};

    // ── Adım 1: SocialProof çek (arama sayfası SSR fetch) ──
    const spBatchSize = 3;
    const spDelayMs = 400;
    for (let i = 0; i < products.length; i += spBatchSize) {
        const batch = products.slice(i, i + spBatchSize);
        const promises = batch.map(p => fetchProductDetail(p.id, p.slug, false));
        const batchResults = await Promise.all(promises);
        batchResults.forEach(r => {
            if (r) results[String(r.id)] = r;
        });
        if (i + spBatchSize < products.length) {
            await new Promise(resolve => setTimeout(resolve, spDelayMs));
        }
    }

    // ── Adım 2: Stok çek (detay sayfası SSR fetch — paralel batch) ──
    if (fetchStock) {
        const stockBatchSize = 5; // Stok fetch daha hafif, batch büyütebiliriz
        const stockDelayMs = 300;
        for (let i = 0; i < products.length; i += stockBatchSize) {
            const batch = products.slice(i, i + stockBatchSize);
            const promises = batch.map(async (p) => {
                if (!p.slug) return;
                try {
                    const stock = await fetchStockFromDetailPage(p.slug);
                    const pid = String(p.id);
                    if (results[pid]) {
                        results[pid].stockQuantity = stock;
                    } else {
                        results[pid] = {
                            id: p.id,
                            socialProof: {},
                            ratingScore: null,
                            price: null,
                            stockQuantity: stock
                        };
                    }
                } catch (e) {
                    // silent
                }
            });
            await Promise.all(promises);
            if (i + stockBatchSize < products.length) {
                await new Promise(resolve => setTimeout(resolve, stockDelayMs));
            }
        }
    }

    return results;
}

// Duplicate mesaj koruması — aynı action+id 1sn içinde tekrar gelirse atla
const _processingActions = new Map();

// Mesaj dinleyici
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    // ─── Proaktif sync: Trendyol sayfası açıkken arka planda güncelleme ───
    if (msg.action === "TRENDYOL_PAGE_ACTIVE") {
        proactiveSyncTrackedProducts().catch(e => {
            console.warn("[TK-PROACTIVE] Tetikleme hatası:", e.message);
        });
        sendResponse({ status: "ok" });
        return;
    }

    if (msg.action === "START_CATEGORY_SCAN") {
        if (scanState.isScanning) {
            sendResponse({ status: "already_scanning" });
            return;
        }
        startCategoryScan(msg.originTabId);
        sendResponse({ status: "started", total: CATEGORY_URLS.length });
        return true;
    }

    if (msg.action === "SCRAPE_RESULT") {
        handleScrapeResult(msg.products, msg.category);
        sendResponse({ status: "ok" });
        return;
    }

    if (msg.action === "GET_SCAN_STATUS") {
        sendResponse({
            isScanning: scanState.isScanning,
            currentIndex: scanState.currentIndex,
            totalCategories: scanState.totalCategories,
            currentCategory: scanState.currentIndex < CATEGORY_URLS.length
                ? CATEGORY_URLS[scanState.currentIndex].name : "",
            productCount: scanState.allProducts.length,
        });
        return;
    }

    if (msg.action === "GET_SCAN_RESULTS") {
        sendResponse({ products: scanState.allProducts });
        return;
    }

    // ─── Panel Research: Toplu socialProof çek (apigw API) ───
    // Research sayfası extension üzerinden Trendyol apigw'den socialProof çeker.
    // Sunucu IP 403 alıyor, kullanıcı tarayıcısından sorunsuz çalışır.
    if (msg.action === "BULK_FETCH_SOCIALPROOF") {
        const productIds = msg.productIds || [];
        if (productIds.length === 0) {
            sendResponse({ results: {} });
            return;
        }
        bulkFetchSocialProofFromApigw(productIds).then(results => {
            sendResponse({ results });
        }).catch(err => {
            console.warn("[TK-BG] BULK_FETCH_SOCIALPROOF hata:", err.message);
            sendResponse({ results: {} });
        });
        return true;
    }

    // ─── Yeni: Toplu ürün detay verisi çek (T1 + Stok) ───
    if (msg.action === "FETCH_PRODUCT_DETAILS") {
        // products: [{id, slug}, ...] formatında
        const products = msg.products || [];
        if (products.length === 0) {
            sendResponse({ results: {} });
            return;
        }
        const fetchStock = msg.fetchStock || false;
        fetchMultipleProductDetails(products, fetchStock).then(results => {
            sendResponse({ results: results });
        }).catch(err => {
            console.warn("[TK-BG] FETCH_PRODUCT_DETAILS hata:", err.message);
            sendResponse({ results: {} });
        });
        return true;
    }

    // ─── Yeni: Tek ürün detay çek (detay sayfası için) ───
    if (msg.action === "FETCH_SINGLE_PRODUCT") {
        const productId = msg.productId;
        const slug = msg.slug;
        if (!productId || !slug) {
            sendResponse({ result: null });
            return;
        }
        fetchProductDetail(productId, slug).then(result => {
            sendResponse({ result: result });
        }).catch(err => {
            sendResponse({ result: null });
        });
        return true;
    }

    // ═══════════════════════════════════════════════════════════════════
    //  STOK DELTA TAKİP SİSTEMİ — Mesaj Handler'ları
    // ═══════════════════════════════════════════════════════════════════

    if (msg.action === "TRACK_PRODUCT") {
        const pid = msg.product && msg.product.id;
        const dedupKey = `TRACK_${pid}`;
        const now = Date.now();
        if (pid && _processingActions.has(dedupKey) && now - _processingActions.get(dedupKey) < 2000) {
            sendResponse({ status: "processing", message: "İstek zaten işleniyor" });
            return true;
        }
        _processingActions.set(dedupKey, now);
        trackProduct(msg.product).then(async (result) => {
            _processingActions.delete(dedupKey);
            // Panel'e de ekle — sonucu kontrol et (limit aşımı vs.)
            const panelResult = await panelTrackProduct(msg.product).catch(() => null);
            if (panelResult && panelResult.limitReached) {
                sendResponse({ status: "limit_reached", error: panelResult.error || "Ürün takip limitine ulaştınız" });
            } else {
                sendResponse(result);
            }
        }).catch(err => {
            _processingActions.delete(dedupKey);
            sendResponse({ status: "error", error: err.message });
        });
        return true;
    }

    if (msg.action === "UNTRACK_PRODUCT") {
        untrackProduct(msg.productId).then(result => {
            // Panel'den de untrack et (arka planda)
            panelUntrackProduct(msg.productId).catch(() => {});
            sendResponse(result);
        });
        return true;
    }

    if (msg.action === "GET_TRACKED_PRODUCTS") {
        getTrackedProducts().then(result => sendResponse(result));
        return true;
    }

    if (msg.action === "GET_PRODUCT_TRACKING_DATA") {
        getProductTrackingData(msg.productId).then(result => sendResponse(result));
        return true;
    }

    // ─── Web Panel Sync: Content script'ten gelen socialProof verisi ───
    // 2 yollu sync: hafif cache (tum listing urunleri) + full sync (stok/fiyat olan urunler)
    if (msg.action === "SYNC_SOCIAL_PROOF") {
        const products = msg.products || [];
        if (products.length > 0) {
            // 1) Hafif socialProof cache: TUM urunlerin socialProof'unu panele gonder
            //    (processStockReading yok, sadece cache yazma — hizli)
            const spCacheItems = products
                .filter(p => p.productId && p.socialProof)
                .map(p => ({ productId: String(p.productId), socialProof: p.socialProof }));
            if (spCacheItems.length > 0) {
                syncSocialProofCache(spCacheItems).catch(e =>
                    console.log("[TK-SYNC] socialproof cache hata:", e.message)
                );
            }

            // 2) Full sync: stok/fiyat bilgisi olan urunler (mevcut davranis)
            const fullSyncItems = products.filter(p =>
                p.stock !== undefined || p.price !== undefined || p.reviewCount !== undefined
            );
            if (fullSyncItems.length > 0) {
                syncToPanel(fullSyncItems).then(() => {
                    sendResponse({ status: "ok" });
                }).catch(() => {
                    sendResponse({ status: "error" });
                });
            } else {
                sendResponse({ status: "ok", cacheOnly: true });
            }
        } else {
            sendResponse({ status: "empty" });
        }
        return true;
    }

    // ═══ Panel Auth Handler'ları ═══
    if (msg.action === "PANEL_LOGIN") {
        (async () => {
            try {
                // Onceki kullaniciyi hatirla (kullanici degisimini tespit etmek icin)
                const prevAuth = await getPanelAuth();
                const prevUserId = prevAuth ? prevAuth.userId : null;

                const panelUrl = await getPanelUrl();
                const resp = await fetch(`${panelUrl}/api/extension/auth`, {
                    method: "POST",
                    headers: { "Content-Type": "application/json" },
                    body: JSON.stringify({ email: msg.email, password: msg.password }),
                    signal: AbortSignal.timeout(8000),
                });
                const data = await resp.json();
                if (resp.ok && data.token) {
                    const newUserId = data.user.id;

                    // Kullanici degistiyse local stockTracking'i temizle
                    // (eski kullanicinin takip verileri yeni kullaniciya gozukmemeli)
                    if (prevUserId && prevUserId !== newUserId) {
                        console.log(`[TK-PANEL] Kullanici degisti: ${prevUserId} → ${newUserId}, local tracking temizleniyor`);
                        await chrome.storage.local.remove("stockTracking");
                    }

                    await setPanelAuth({
                        token: data.token,
                        userId: data.user.id,
                        email: data.user.email,
                        name: data.user.name,
                        plan: data.user.plan,
                    });
                    // Tracked IDs cache'ini hemen doldur
                    panelTrackedCache.lastFetch = 0;
                    await refreshPanelTrackedIds();
                    // Panel'deki ürünleri extension'a sync et
                    await syncProductsFromPanel();
                    // Kargo config'ini çek (arka planda — login'i bloklamaz)
                    fetchShippingConfig().catch(e =>
                        console.log("[TK-CONFIG] Login sonrası config hatası:", e.message)
                    );
                    sendResponse({ status: "ok", user: data.user, limits: data.limits });
                } else {
                    sendResponse({ status: "error", error: data.error || "Giris basarisiz" });
                }
            } catch (e) {
                sendResponse({ status: "error", error: e.message || "Baglanti hatasi" });
            }
        })();
        return true;
    }

    if (msg.action === "PANEL_LOGOUT") {
        clearPanelAuth().then(() => {
            sendResponse({ status: "ok" });
        });
        return true;
    }

    if (msg.action === "GET_PANEL_AUTH") {
        getPanelAuth().then(auth => {
            sendResponse({ auth });
        });
        return true;
    }

    if (msg.action === "FETCH_SHIPPING_CONFIG") {
        // Content script'ten çağrılır — güncel config'i döndür
        (async () => {
            try {
                // Önce storage'dan bak — taze mi?
                const stored = await chrome.storage.local.get("shippingConfig");
                const config = stored.shippingConfig;
                const isFresh = config && config.fetchedAt && (Date.now() - config.fetchedAt < 6 * 60 * 60 * 1000);

                if (isFresh) {
                    sendResponse({ status: "ok", config, cached: true });
                } else {
                    // Taze değil — panel'den çek
                    const freshConfig = await fetchShippingConfig();
                    if (freshConfig) {
                        sendResponse({ status: "ok", config: { ...freshConfig, fetchedAt: Date.now() }, cached: false });
                    } else {
                        // Fetch başarısız — eski config varsa onu dön
                        sendResponse({ status: config ? "ok" : "error", config: config || null, cached: true });
                    }
                }
            } catch (e) {
                sendResponse({ status: "error", error: e.message });
            }
        })();
        return true;
    }

    // ─── Extension View Check: Gunluk goruntuleme limiti kontrol ───
    if (msg.action === "CHECK_VIEW_LIMIT") {
        (async () => {
            try {
                const auth = await getPanelAuth();
                if (!auth || !auth.token) {
                    // Auth yoksa limit yok
                    sendResponse({ allowed: true });
                    return;
                }
                const panelUrl = await getPanelUrl();
                const resp = await fetch(`${panelUrl}/api/extension/view`, {
                    method: "POST",
                    headers: {
                        "Authorization": `Bearer ${auth.token}`,
                        "Content-Type": "application/json",
                    },
                    signal: AbortSignal.timeout(5000),
                });
                if (resp.ok) {
                    const data = await resp.json();
                    sendResponse(data);
                } else {
                    // Hata durumunda engelleme
                    sendResponse({ allowed: true });
                }
            } catch (e) {
                // Baglanti hatasi — engelleme
                sendResponse({ allowed: true });
            }
        })();
        return true;
    }

    if (msg.action === "GET_PANEL_TRACKED_IDS") {
        refreshPanelTrackedIds().then(ids => {
            sendResponse({ ids: [...ids] });
        });
        return true;
    }

    // ─── Panel Estimate Lookup: Untracked ürünler için panel DB'den tahmin çek ───
    // Kullanıcı takip etmemiş ama başka biri track etmiş bir ürünü ziyaret ediyorsa,
    // panel'deki mevcut veriyi gösterir (X kullanıcı, Y kullanıcının verisini görür)
    if (msg.action === "FETCH_PANEL_ESTIMATE") {
        (async () => {
            try {
                const panelUrl = await getPanelUrl();
                const ids = Array.isArray(msg.productIds) ? msg.productIds.join(",") : String(msg.productIds);
                const resp = await fetch(`${panelUrl}/api/extension/sync?ids=${ids}`, {
                    signal: AbortSignal.timeout(5000),
                });
                if (resp.ok) {
                    const data = await resp.json();
                    sendResponse({ status: "ok", estimates: data.estimates || {} });
                } else {
                    sendResponse({ status: "error", estimates: {} });
                }
            } catch (e) {
                sendResponse({ status: "error", estimates: {} });
            }
        })();
        return true;
    }

    // ─── Panel Estimate (socialProof ile): Takip edilmeyen ürünler için panel hesaplama ───
    // Extension'dan socialProof verileri gönderilir, panel computePanelEstimate ile hesaplar.
    // Research/scanner ile aynı hesaplama zinciri — tutarlı sonuç.
    if (msg.action === "BULK_ESTIMATE_SALES") {
        (async () => {
            try {
                const panelUrl = await getPanelUrl();
                const products = msg.products || [];
                if (products.length === 0) {
                    sendResponse({ status: "ok", estimates: {} });
                    return;
                }
                const resp = await fetch(`${panelUrl}/api/extension/estimate`, {
                    method: "POST",
                    headers: { "Content-Type": "application/json" },
                    body: JSON.stringify({ products }),
                    signal: AbortSignal.timeout(10000),
                });
                if (resp.ok) {
                    const data = await resp.json();
                    sendResponse({ status: "ok", estimates: data.estimates || {} });
                } else {
                    sendResponse({ status: "error", estimates: {} });
                }
            } catch (e) {
                console.warn("[TK-BG] BULK_ESTIMATE_SALES hata:", e.message);
                sendResponse({ status: "error", estimates: {} });
            }
        })();
        return true;
    }

    // ─── Fiyat Geçmişi: Panel'den ürünün fiyat geçmişini çek ───
    if (msg.action === "GET_PRICE_HISTORY") {
        (async () => {
            try {
                const auth = await getPanelAuth();
                if (!auth || !auth.token) {
                    sendResponse({ status: "no_auth", data: [] });
                    return;
                }
                const panelUrl = await getPanelUrl();
                const pid = msg.productId;
                const days = msg.days || 90;
                const resp = await fetch(`${panelUrl}/api/extension/price-history?productId=${pid}&days=${days}`, {
                    headers: { "Authorization": `Bearer ${auth.token}` },
                    signal: AbortSignal.timeout(8000),
                });
                if (resp.ok) {
                    const result = await resp.json();
                    sendResponse({ status: "ok", ...result });
                } else {
                    sendResponse({ status: "not_found", data: [] });
                }
            } catch (e) {
                sendResponse({ status: "error", data: [] });
            }
        })();
        return true;
    }

    if (msg.action === "FORCE_STOCK_CHECK") {
        chrome.storage.local.get("stockTracking").then(data => {
            const tracking = data.stockTracking;
            const count = tracking && tracking.products ? Object.keys(tracking.products).length : 0;
            // Hemen yanıt ver — stok çekme arka planda devam eder
            sendResponse({ status: "started", count: count });
            // Arka planda stok çek
            fetchStockForAllTrackedProducts().then(() => {
                console.log("[TK-STOCK] FORCE_STOCK_CHECK tamamlandı");
            }).catch(e => {
                console.error("[TK-STOCK] FORCE_STOCK_CHECK hatası:", e.message);
            });
        });
        return true;
    }
});

async function startCategoryScan(originTabId) {
    scanState = {
        isScanning: true,
        currentIndex: 0,
        totalCategories: CATEGORY_URLS.length,
        allProducts: [],
        originTabId: originTabId,
        scanTabId: null,
    };

    // Tarama için yeni tab aç
    const tab = await chrome.tabs.create({ url: CATEGORY_URLS[0].url, active: false });
    scanState.scanTabId = tab.id;

    // Tab yüklenince scrape başlat
    chrome.tabs.onUpdated.addListener(onTabUpdated);
}

function onTabUpdated(tabId, changeInfo, tab) {
    if (tabId !== scanState.scanTabId) return;
    if (changeInfo.status !== "complete") return;
    if (!scanState.isScanning) return;

    // Küçük bir bekleme — sayfa tamamen render olsun
    setTimeout(() => {
        const category = CATEGORY_URLS[scanState.currentIndex];
        chrome.tabs.sendMessage(scanState.scanTabId, {
            action: "SCRAPE_CATEGORY",
            category: category.name,
            minProducts: 50,
        }).catch(err => {
            console.warn("Content script'e mesaj gönderilemedi, atlıyorum:", category.name, err);
            moveToNextCategory();
        });
    }, 2000);
}

function handleScrapeResult(products, category) {
    if (!scanState.isScanning) return;

    // Kategori bilgisi ekle
    const taggedProducts = (products || []).map(p => ({
        ...p,
        scanCategory: category,
    }));

    scanState.allProducts = scanState.allProducts.concat(taggedProducts);

    // İlerleme bilgisi popup'a gönder
    broadcastProgress();

    moveToNextCategory();
}

function moveToNextCategory() {
    scanState.currentIndex++;

    if (scanState.currentIndex >= CATEGORY_URLS.length) {
        // Tarama bitti
        finishScan();
        return;
    }

    // Sonraki kategori URL'ine git
    const nextUrl = CATEGORY_URLS[scanState.currentIndex].url;
    chrome.tabs.update(scanState.scanTabId, { url: nextUrl });
}

async function finishScan() {
    scanState.isScanning = false;

    // Tarama tabını kapat
    if (scanState.scanTabId) {
        try {
            await chrome.tabs.remove(scanState.scanTabId);
        } catch (e) {
            // Tab zaten kapatılmış olabilir
        }
    }

    // Listener'ı temizle
    chrome.tabs.onUpdated.removeListener(onTabUpdated);

    // Sonuçları storage'a kaydet
    await chrome.storage.local.set({
        scanResults: scanState.allProducts,
        scanDate: new Date().toISOString(),
        scanCategoryCount: CATEGORY_URLS.length,
    });

    // Orijinal tab'a dashboard göster mesajı gönder
    if (scanState.originTabId) {
        try {
            await chrome.tabs.sendMessage(scanState.originTabId, {
                action: "SHOW_SCANNER_DASHBOARD",
                products: scanState.allProducts,
                scanDate: new Date().toISOString(),
            });
        } catch (e) {
            // Orijinal tab Trendyol sayfası değilse, yeni bir Trendyol tab'ı aç
            const newTab = await chrome.tabs.create({ url: "https://www.trendyol.com" });
            // Yeni tab yüklenince dashboard gönder
            const waitForLoad = (tId, info) => {
                if (tId === newTab.id && info.status === "complete") {
                    chrome.tabs.onUpdated.removeListener(waitForLoad);
                    setTimeout(() => {
                        chrome.tabs.sendMessage(newTab.id, {
                            action: "SHOW_SCANNER_DASHBOARD",
                            products: scanState.allProducts,
                            scanDate: new Date().toISOString(),
                        });
                    }, 1500);
                }
            };
            chrome.tabs.onUpdated.addListener(waitForLoad);
        }
    }

    broadcastProgress();
}

function broadcastProgress() {
    // Tüm popup bağlantılarına ilerleme gönder
    chrome.runtime.sendMessage({
        action: "SCAN_PROGRESS",
        currentIndex: scanState.currentIndex,
        totalCategories: scanState.totalCategories,
        currentCategory: scanState.currentIndex < CATEGORY_URLS.length
            ? CATEGORY_URLS[scanState.currentIndex].name : "Tamamlandı",
        productCount: scanState.allProducts.length,
        isScanning: scanState.isScanning,
    }).catch(() => {
        // Popup kapalıysa hata yok
    });
}


// ═══════════════════════════════════════════════════════════════════════
//  STOK DELTA TAKİP SİSTEMİ
//  Ürün stoklarını periyodik takip ederek gerçek satış adetini hesaplar.
//  Top 3 satıcı (ana + en yakın 2) stokları toplanarak izlenir.
//  chrome.alarms ile 2 saatte bir otomatik çalışır — Trendyol sekmesi gerekmez.
// ═══════════════════════════════════════════════════════════════════════

const STOCK_ALARM_NAME = "stockTrackingAlarm";
const STOCK_FETCH_INTERVAL = 60;  // dakika (1 saat) — gündüz
const STOCK_NIGHT_START = 0;      // Gece modu başlangıç saati (00:00)
const STOCK_NIGHT_END = 8;        // Gece modu bitiş saati (08:00)
const STOCK_NIGHT_CHECKS = 2;     // Gece boyunca toplam ölçüm sayısı (00:00'da + ~04:00'da)
const STOCK_MAX_PRODUCTS = 20;
const STOCK_EMA_ALPHA = 0.3;
const STOCK_SUPPLY_THRESHOLD_ABS = 5;     // ≤5 birim artış = iade
const STOCK_SUPPLY_THRESHOLD_PCT = 10;    // >%10 artış = tedarik
const STOCK_ANOMALY_MULTIPLIER = 5;       // Beklenen satış hızının 5 katından fazla düşüş = anomali
const STOCK_ANOMALY_MIN_DROP_PCT = 30;    // Stokun %30'undan fazla düşüş (referans yoksa) = anomali
const STOCK_ANOMALY_CONFIRM_READINGS = 3; // Anomali tespitinden sonra doğrulama için beklenecek ölçüm sayısı
// NOT: Statik minimum eşik (eski STOCK_ANOMALY_MIN_ABS=500) kaldırıldı.
// Anomali tespiti artık tamamen ürünün kendi satış hızına göre dinamik çalışır.
// Ayrıca "pending_verification" sistemi ile yanlış pozitifler sonraki ölçümlerle düzeltilir.
// Influencer paylaşımı, flash kampanya vb. gerçek ani satış artışları böylece korunur.
const STOCK_PRUNE_DAYS = 30;
const BUYBOX_SWITCH_MIN_JUMP = 500;   // Buybox switch tespiti — minimum stok farkı
const BUYBOX_SWITCH_PCT = 50;         // Buybox switch tespiti — minimum % değişim

// ─── Service worker başlangıcında panel sync (login durumu varsa) ───
// Extension restart/update sonrası stockTracking boş olabilir — panel'den doltur
(async () => {
    try {
        const auth = await getPanelAuth();
        if (auth && auth.token) {
            await syncProductsFromPanel();
        }
    } catch (e) {
        console.warn("[TK-PANEL] Startup sync hatası:", e.message);
    }
})();

// ─── Alarm kurulumu (service worker başlangıcında) ───
chrome.alarms.get(STOCK_ALARM_NAME, (alarm) => {
    if (!alarm) {
        chrome.storage.local.get("stockTracking", (data) => {
            const tracking = data.stockTracking;
            if (tracking && tracking.products && Object.keys(tracking.products).length > 0) {
                chrome.alarms.create(STOCK_ALARM_NAME, { periodInMinutes: STOCK_FETCH_INTERVAL });
                console.log("[TK-STOCK] Alarm oluşturuldu (mevcut ürünler var)");
            }
        });
    }
});

chrome.alarms.onAlarm.addListener((alarm) => {
    if (alarm.name === STOCK_ALARM_NAME) {
        const hour = new Date().getHours();
        const isNight = hour >= STOCK_NIGHT_START && hour < STOCK_NIGHT_END;

        if (isNight) {
            if (hour !== 0 && hour !== 4) {
                console.log(`[TK-STOCK] Gece modu — saat ${hour}:00, ölçüm atlandı`);
                return;
            }
            console.log(`[TK-STOCK] Gece ölçümü — saat ${hour}:00`);
        } else {
            console.log("[TK-STOCK] Alarm tetiklendi — stok kontrolü başlıyor");
        }
        // Async IIFE ile service worker'ı canlı tut (await olmadan SW kapanabilir)
        (async () => {
            try {
                await fetchStockForAllTrackedProducts();

                // Pending scrape queue varsa devam et (önceki sync'ten kalan veya yarıda kalan)
                try {
                    const sqData = await chrome.storage.local.get("pendingScrapeQueue");
                    if (sqData.pendingScrapeQueue && sqData.pendingScrapeQueue.length > 0) {
                        console.log("[TK-STOCK] Pending scrape queue bulundu, işleniyor...");
                        await processScrapeQueue(sqData.pendingScrapeQueue);
                    }
                } catch (sqErr) {
                    console.warn("[TK-STOCK] Pending queue hatası:", sqErr.message);
                }
            } catch (e) {
                console.error("[TK-STOCK] Alarm handler hatası:", e.message);
            }
        })();
    }
});

// ─── Detay sayfasından genişletilmiş veri çekme ───
// slug: "/marka/urun-adi-p-12345" veya "?merchantId=XXX" ile
// Geri döner: { stock, price, merchantId, merchantName, otherMerchants: [{id, name}] }
async function fetchStockFromDetailPageFull(slug) {
    try {
        // Cache-buster ekle — CDN cache'ini bypass et
        const cacheBuster = `_t=${Date.now()}`;
        const separator = slug.includes("?") ? "&" : "?";
        const url = `https://www.trendyol.com${slug}${separator}${cacheBuster}`;
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 12000);
        const resp = await fetch(url, {
            headers: {
                "Accept": "text/html",
                "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36"
            },
            cache: "no-store",
            signal: controller.signal
        });
        clearTimeout(timeoutId);
        if (!resp.ok) {
            console.warn(`[TK-STOCK] fetchStockFromDetailPageFull HTTP ${resp.status}: ${slug}`);
            return null;
        }
        const html = await resp.text();

        const marker = "__envoy_product-detail__PROPS";
        const idx = html.indexOf(marker);
        if (idx === -1) return null;

        const scriptStart = html.lastIndexOf("<script", idx);
        const scriptEnd = html.indexOf("</script>", idx);
        if (scriptStart === -1 || scriptEnd === -1) return null;

        const scriptContent = html.substring(scriptStart, scriptEnd);
        const eqIdx = scriptContent.indexOf("=", scriptContent.indexOf(marker));
        if (eqIdx === -1) return null;

        let jsonStr = scriptContent.substring(eqIdx + 1).trim();
        if (jsonStr.endsWith(";")) jsonStr = jsonStr.slice(0, -1);

        const data = JSON.parse(jsonStr);
        const p = data && data.product;
        if (!p || !p.merchantListing) return null;

        const ml = p.merchantListing;
        const wv = ml.winnerVariant;
        const merchant = ml.merchant;
        const price = wv && wv.price
            ? (wv.price.discountedPrice && wv.price.discountedPrice.value) || (wv.price.sellingPrice && wv.price.sellingPrice.value) || 0
            : 0;

        const otherMerchants = [];
        if (p.otherMerchantListings && Array.isArray(p.otherMerchantListings)) {
            p.otherMerchantListings.forEach(oml => {
                if (oml.merchant && oml.merchant.id) {
                    otherMerchants.push({
                        id: oml.merchant.id,
                        name: oml.merchant.name || "",
                        price: oml.winnerVariant && oml.winnerVariant.price
                            ? (oml.winnerVariant.price.discountedPrice && oml.winnerVariant.price.discountedPrice.value) || (oml.winnerVariant.price.sellingPrice && oml.winnerVariant.price.sellingPrice.value) || 0
                            : 0,
                        stock: (oml.winnerVariant && oml.winnerVariant.quantity) || 0
                    });
                }
            });
        }

        return {
            stock: (wv && wv.quantity) || 0,
            price: price,
            merchantId: merchant ? merchant.id : 0,
            merchantName: merchant ? (merchant.name || "") : "",
            otherMerchants: otherMerchants
        };
    } catch (e) {
        console.warn("[TK-STOCK] fetchStockFromDetailPageFull hata:", e.message);
        return null;
    }
}

// ─── Belirli bir merchantId ile stok çekme ───
async function fetchStockForMerchant(slug, merchantId) {
    try {
        // Cache-buster ekle — CDN cache'ini bypass et
        const baseSlug = slug.split("?")[0];
        const url = `https://www.trendyol.com${baseSlug}?merchantId=${merchantId}&_t=${Date.now()}`;
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 10000);
        const resp = await fetch(url, {
            headers: {
                "Accept": "text/html",
                "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36"
            },
            cache: "no-store",
            signal: controller.signal
        });
        clearTimeout(timeoutId);
        if (!resp.ok) return 0;
        const html = await resp.text();

        const marker = "__envoy_product-detail__PROPS";
        const idx = html.indexOf(marker);
        if (idx === -1) return 0;

        const scriptStart = html.lastIndexOf("<script", idx);
        const scriptEnd = html.indexOf("</script>", idx);
        if (scriptStart === -1 || scriptEnd === -1) return 0;

        const scriptContent = html.substring(scriptStart, scriptEnd);
        const eqIdx = scriptContent.indexOf("=", scriptContent.indexOf(marker));
        if (eqIdx === -1) return 0;

        let jsonStr = scriptContent.substring(eqIdx + 1).trim();
        if (jsonStr.endsWith(";")) jsonStr = jsonStr.slice(0, -1);

        const data = JSON.parse(jsonStr);
        const p = data && data.product;
        if (p && p.merchantListing && p.merchantListing.winnerVariant) {
            return p.merchantListing.winnerVariant.quantity || 0;
        }
    } catch (e) {
        // silent
    }
    return 0;
}

// ─── Buybox + en yakın 2 satıcının stoğunu çek ───
// Her ölçümde satıcı listesi güncellenir (buybox kazananı değişebilir)
// Satıcı bazlı stok takibi ile bir satıcının tedariki diğerinin satışını maskelemez
async function fetchTopMerchantsStock(product) {
    const slug = product.slug;
    const merchantStocks = {};
    let totalStock = 0;
    let currentPrice = product.price;

    // 1) Buybox kazananı (ana satıcı — detaylı fetch, fiyat + satıcı listesi gelir)
    console.log(`[TK-STOCK] fetchTopMerchantsStock başlıyor: ${product.id} — slug: ${slug}`);
    const mainData = await fetchStockFromDetailPageFull(slug);
    if (!mainData) {
        console.warn(`[TK-STOCK] Buybox verisi alınamadı: ${product.id} — slug: ${slug}`);
        return null;
    }

    const buyboxId = mainData.merchantId || "main";
    merchantStocks[buyboxId] = mainData.stock;
    totalStock += mainData.stock;
    currentPrice = mainData.price || product.price;

    // 2) Satıcı listesini HER ÖLÇÜMDE güncelle (buybox kazananı, satıcılar değişebilir)
    //    Fiyata en yakın 2 satıcıyı seç (buybox hariç)
    if (mainData.otherMerchants.length > 0) {
        const sorted = mainData.otherMerchants
            .filter(m => m.price > 0 && String(m.id) !== String(buyboxId))
            .sort((a, b) => Math.abs(a.price - currentPrice) - Math.abs(b.price - currentPrice));
        const newMerchants = sorted.slice(0, 2).map(m => ({ id: m.id, name: m.name }));

        // Buybox kazananı değişti mi?
        const prevBuybox = product.buyboxMerchantId;
        if (prevBuybox && String(prevBuybox) !== String(buyboxId)) {
            console.log(`[TK-STOCK] 🔄 Buybox değişti: ${product.id} — ${prevBuybox} → ${buyboxId}`);
        }
        product.buyboxMerchantId = buyboxId;
        product.buyboxMerchantName = mainData.merchantName || "";

        // Daha önce takip edilen satıcılar yeni listede yoksa, stokları 0'a düşmüş olabilir
        // Onları da okumaya devam et (kaybolmasın)
        const newMerchantIds = newMerchants.map(m => String(m.id));
        const keptMerchants = [...newMerchants];

        // Eski satıcılardan yeni listede olmayanları koru (max 2 ek)
        for (const pm of (product.merchants || [])) {
            if (!newMerchantIds.includes(String(pm.id)) && String(pm.id) !== String(buyboxId)) {
                if (keptMerchants.length < 4) {
                    keptMerchants.push(pm);
                }
            }
        }

        product.merchants = keptMerchants;

        // otherMerchantIds/Names'i sync payload icin sakla
        product.otherMerchantIds = keptMerchants.map(m => String(m.id));
        const nameMap = {};
        keptMerchants.forEach(m => { if (m.name) nameMap[String(m.id)] = m.name; });
        if (Object.keys(nameMap).length > 0) product.otherMerchantNames = nameMap;
    }

    console.log(`[TK-STOCK] Buybox: ${buyboxId} stok=${mainData.stock}, diğer satıcı: ${(product.merchants || []).length}`);

    // 3) Diğer satıcıların stoğunu çek (paralel)
    if (product.merchants && product.merchants.length > 0) {
        const otherPromises = product.merchants.map(m =>
            fetchStockForMerchant(slug, m.id).then(qty => {
                merchantStocks[m.id] = qty;
                totalStock += qty;
            })
        );
        await Promise.all(otherPromises);
    }

    return {
        totalStock: totalStock,
        buyboxStock: mainData.stock, // SADECE buybox satıcının stoku (panel sync için)
        price: currentPrice,
        merchantStocks: merchantStocks,
        buyboxMerchantId: buyboxId
    };
}

// ─── Ürün takip başlat ───
async function trackProduct(productInfo) {
    const data = await chrome.storage.local.get("stockTracking");
    const tracking = data.stockTracking || {
        products: {},
        settings: {
            maxProducts: STOCK_MAX_PRODUCTS,
            fetchIntervalMinutes: STOCK_FETCH_INTERVAL,
            emaAlpha: STOCK_EMA_ALPHA
        },
        meta: { lastAlarmFire: null, totalFetches: 0, version: 1 }
    };

    const productCount = Object.keys(tracking.products).length;

    // Zaten takipte mi?
    if (tracking.products[productInfo.id]) {
        return { status: "already_tracking" };
    }

    // Limit yok — panel planı belirler, extension her zaman ekleyebilir

    // Yeni ürün kaydı oluştur
    tracking.products[productInfo.id] = {
        id: productInfo.id,
        slug: productInfo.slug,
        name: productInfo.name || "",
        brand: productInfo.brand || "",
        price: productInfo.price || 0,
        imageUrl: productInfo.imageUrl || "",
        addedAt: new Date().toISOString(),
        lastFetchAt: null,
        lastStock: 0,
        status: "active",
        merchants: [],          // İlk fetch'te doldurulacak
        readings: [],
        events: [],
        dailySummaries: {},
        weeklySummaries: {},
        emaDailySales: null,
        confidence: { level: 0.20, label: "Cok Dusuk", daysTracked: 0 }
    };

    await chrome.storage.local.set({ stockTracking: tracking });
    console.log(`[TK-STOCK] Ürün takibe alındı: ${productInfo.name} (${productInfo.id})`);

    // Alarm yoksa oluştur
    const alarm = await chrome.alarms.get(STOCK_ALARM_NAME);
    if (!alarm) {
        chrome.alarms.create(STOCK_ALARM_NAME, { periodInMinutes: STOCK_FETCH_INTERVAL });
        console.log("[TK-STOCK] Alarm oluşturuldu");
    }

    // İlk stok okumasını hemen yap
    try {
        await fetchStockForSingleProduct(productInfo.id);
    } catch (e) {
        console.warn("[TK-STOCK] İlk stok okuma hatası:", e.message);
    }

    return { status: "ok", productCount: productCount + 1 };
}

// ─── Ürün takip bırak ───
async function untrackProduct(productId) {
    const data = await chrome.storage.local.get("stockTracking");
    const tracking = data.stockTracking;
    if (!tracking || !tracking.products || !tracking.products[productId]) {
        return { status: "not_found" };
    }

    delete tracking.products[productId];
    await chrome.storage.local.set({ stockTracking: tracking });

    // Son ürün de silindiyse alarmı kaldır
    if (Object.keys(tracking.products).length === 0) {
        await chrome.alarms.clear(STOCK_ALARM_NAME);
        console.log("[TK-STOCK] Son ürün silindi, alarm kaldırıldı");
    }

    return { status: "ok", productCount: Object.keys(tracking.products).length };
}

// ─── Tüm takip edilen ürünlerin özetini döndür ───
async function getTrackedProducts() {
    const data = await chrome.storage.local.get("stockTracking");
    const tracking = data.stockTracking;
    if (!tracking || !tracking.products) {
        return { products: [], count: 0, maxProducts: STOCK_MAX_PRODUCTS };
    }

    // NOT: Burada repair yapmıyoruz — storage'a yazma race condition oluşturur.
    // Repair sadece fetchStockForAllTrackedProducts içinde (yazma öncesi) yapılır.

    // Özet bilgi döndür — dizi olarak (popup UI array bekliyor)
    const productsArray = [];
    for (const [id, p] of Object.entries(tracking.products)) {
        const conf = p.confidence || {};
        const est = p.estimates || {};
        const readings = p.readings || [];

        // Önceki stok değeri ve delta hesapla
        let prevStock = null;
        let stockDelta = null;
        let lastDeltaTime = null;
        if (readings.length >= 2) {
            const lastReading = readings[readings.length - 1];
            const prevReading = readings[readings.length - 2];
            prevStock = prevReading.totalStock;
            stockDelta = lastReading.totalStock - prevReading.totalStock;
            lastDeltaTime = lastReading.ts;
        }

        // Son satış event'ini bul
        const events = p.events || [];
        let lastSaleEvent = null;
        for (let i = events.length - 1; i >= 0; i--) {
            if (events[i].type === "sale") {
                lastSaleEvent = events[i];
                break;
            }
        }

        // İlk okumadaki stok (başlangıç referansı)
        const firstStock = readings.length > 0 ? readings[0].totalStock : null;

        productsArray.push({
            id: p.id,
            name: p.name,
            brand: p.brand,
            slug: p.slug,
            price: p.price,
            imageUrl: p.imageUrl,
            addedAt: p.addedAt,
            lastFetchAt: p.lastFetchAt,
            lastStock: p.lastStock,
            status: p.status,
            // Stok delta bilgisi
            prevStock: prevStock,
            stockDelta: stockDelta,
            lastDeltaTime: lastDeltaTime,
            firstStock: firstStock,
            // Son satış event bilgisi
            lastSaleEvent: lastSaleEvent ? {
                sold: lastSaleEvent.sold,
                hourlyRate: lastSaleEvent.hourlyRate,
                prevStock: lastSaleEvent.prevStock,
                newStock: lastSaleEvent.newStock,
                ts: lastSaleEvent.ts
            } : null,
            // Satış tahminleri — HER ZAMAN ham T0 (extension kendi hesabı)
            // Panel blend panelEstimate ayrı field olarak geçer
            hourlyRate: est.hourlyRate || 0,
            dailyEstimate: est.dailyEstimate || 0,
            weeklyEstimate: est.weeklyEstimate || 0,
            monthlyEstimate: est.monthlyEstimate || 0,
            totalRealSold: est.totalRealSold || 0,
            estimateSource: est.source || "none",
            emaDailySales: p.emaDailySales || 0,
            panelEstimate: p.panelEstimate || null,
            dataQualityRank: p.dataQualityRank || 1,
            // Güvenilirlik
            confidenceLevel: conf.level || 0,
            confidenceLabel: conf.label || "Başlangıç",
            daysTracked: conf.daysTracked || 0,
            readingCount: readings.length,
            saleEventCount: conf.saleEvents || 0,
            pendingVerifications: (p.events || []).filter(e => e.type === "pending_verification").length,
            merchants: p.merchants || [],
            buyboxMerchantId: p.buyboxMerchantId || null,
            buyboxMerchantName: p.buyboxMerchantName || "",
            merchantCount: (p.merchants || []).length + 1, // buybox + diğerleri
            // Son okumanın satıcı bazlı stokları
            lastMerchantStocks: readings.length > 0 ? (readings[readings.length - 1].merchants || {}) : {}
        });
    }

    // En son eklenen en üstte
    productsArray.sort((a, b) => new Date(b.addedAt || 0).getTime() - new Date(a.addedAt || 0).getTime());

    return { products: productsArray, count: productsArray.length, maxProducts: STOCK_MAX_PRODUCTS };
}

// ─── Tek ürünün detaylı takip verisini döndür ───
async function getProductTrackingData(productId) {
    const data = await chrome.storage.local.get("stockTracking");
    const tracking = data.stockTracking;
    if (!tracking || !tracking.products || !tracking.products[productId]) {
        return { data: null };
    }

    const p = tracking.products[productId];
    const daysTracked = Math.max(0, Math.floor((Date.now() - new Date(p.addedAt).getTime()) / 86400000));

    // Son fiyat değişikliği bul
    let lastPriceChange = null;
    if (p.events && p.events.length > 0) {
        for (let i = p.events.length - 1; i >= 0; i--) {
            if (p.events[i].type === "price_change") {
                lastPriceChange = p.events[i];
                break;
            }
        }
    }

    // Son 7 günlük satış trendi
    const last7Days = [];
    const now = new Date();
    for (let d = 6; d >= 0; d--) {
        const dateKey = new Date(now.getTime() - d * 86400000).toISOString().split("T")[0];
        const summary = p.dailySummaries && p.dailySummaries[dateKey];
        last7Days.push({
            date: dateKey,
            sales: summary ? summary.totalSold : 0,
            readings: summary ? summary.readingCount : 0
        });
    }

    // Son 30 günlük fiyat geçmişi (dailySummaries'den)
    const priceHistory = [];
    const summaries = p.dailySummaries || {};
    const sortedDates = Object.keys(summaries).sort();
    const last30 = sortedDates.slice(-30);
    for (const dateKey of last30) {
        const s = summaries[dateKey];
        if (s && s.priceMin > 0) {
            priceHistory.push({
                date: dateKey,
                min: s.priceMin,
                max: s.priceMax,
                avg: Math.round((s.priceMin + s.priceMax) / 2 * 100) / 100
            });
        }
    }

    const est = p.estimates || {};
    const conf = p.confidence || {};

    const pe = p.panelEstimate || null;

    return {
        data: {
            id: p.id,
            slug: p.slug,
            name: p.name,
            price: p.price,
            lastStock: p.lastStock,
            status: p.status,
            // Satış tahminleri — HER ZAMAN extension'ın kendi ham T0 hesabı
            // Panel blend ayrı field (panelEstimate) olarak geçer
            // Content.js calculator T0+T1+T3 blend'i kendisi yapacak
            hourlyRate: est.hourlyRate || 0,
            dailyEstimate: est.dailyEstimate || 0,
            weeklyEstimate: est.weeklyEstimate || 0,
            monthlyEstimate: est.monthlyEstimate || 0,
            estimatedMonthlySales: est.monthlyEstimate || 0,
            totalRealSold: est.totalRealSold || 0,
            estimateSource: est.source || "none",
            emaDailySales: p.emaDailySales || 0,
            panelEstimate: pe,
            dataQualityRank: p.dataQualityRank || 1,
            // Güvenilirlik
            confidence: conf.level || 0,
            confidenceLabel: conf.label || "Başlangıç",
            daysTracked: conf.daysTracked || 0,
            readings: conf.readings || 0,
            saleEvents: conf.saleEvents || 0,
            // Stok takip modu
            stockTrackingMode: p.stockTrackingMode || 'delta',
            // Ek
            lastFetchAt: p.lastFetchAt,
            lastPriceChange: lastPriceChange,
            last7Days: last7Days,
            priceHistory: priceHistory,
            totalReadings: (p.readings || []).length,
            totalEvents: (p.events || []).length,
            merchants: p.merchants || [],
            // Panel'den gelen socialProof (RECENT_SALES_COUNT, ORDER_COUNT vb.)
            // Content.js calculator'a T1 olarak inject edecek
            panelSocialProof: p.panelSocialProof || null
        }
    };
}

// ─── Tek ürün için stok çek ve işle ───
// İlk takipte veya tek ürün güncelleme için kullanılır
// Hızlı olması için sadece ana satıcıdan stok çeker (diğer satıcılar alarm'da çekilir)
async function fetchStockForSingleProduct(productId) {
    const data = await chrome.storage.local.get("stockTracking");
    const tracking = data.stockTracking;
    if (!tracking || !tracking.products || !tracking.products[productId]) {
        console.warn(`[TK-STOCK] fetchStockForSingleProduct: ürün bulunamadı — ${productId}`);
        return;
    }

    const product = tracking.products[productId];
    console.log(`[TK-STOCK] fetchStockForSingleProduct başlıyor: ${productId} — slug: ${product.slug}`);

    // Hızlı tek fetch — sadece ana satıcı (fetchStockFromDetailPageFull)
    const mainData = await fetchStockFromDetailPageFull(product.slug);
    if (!mainData) {
        console.warn(`[TK-STOCK] İlk fetch BAŞARISIZ: ${productId} — slug: ${product.slug}`);
        // Slug sorunlu olabilir — en azından lastFetchAt güncelle ki popup "bekleniyor" göstermesin
        product.lastFetchAt = Date.now();
        await chrome.storage.local.set({ stockTracking: tracking });
        return;
    }

    const totalStock = mainData.stock;
    const price = mainData.price || product.price;
    const merchantStocks = { [mainData.merchantId || "main"]: mainData.stock };

    // Diğer satıcıları kaydet (alarm'da kullanılacak)
    if (mainData.otherMerchants.length > 0 && (!product.merchants || product.merchants.length === 0)) {
        const sorted = mainData.otherMerchants
            .filter(m => m.price > 0)
            .sort((a, b) => Math.abs(a.price - price) - Math.abs(b.price - price));
        product.merchants = sorted.slice(0, 2).map(m => ({ id: m.id, name: m.name }));
    }

    console.log(`[TK-STOCK] İlk fetch OK: ${productId} — stok=${totalStock}, fiyat=${price}, readings önce=${(product.readings||[]).length}`);

    processStockReading(product, totalStock, price, merchantStocks, mainData.merchantId);
    product.lastFetchAt = Date.now();
    product.lastStock = totalStock;
    product.price = price;
    product.lastBuyboxMerchantId = mainData.merchantId || null;

    try {
        await chrome.storage.local.set({ stockTracking: tracking });
        console.log(`[TK-STOCK] Storage güncellendi: ${productId} — readings=${(product.readings||[]).length}, events=${(product.events||[]).length}`);
    } catch (e) {
        console.error(`[TK-STOCK] Storage set hatası (single):`, e.message);
    }
}

// ─── Tüm takip edilen ürünler için stok çek (alarm handler) ───
async function fetchStockForAllTrackedProducts() {
    const data = await chrome.storage.local.get("stockTracking");
    const tracking = data.stockTracking;
    if (!tracking || !tracking.products) return;

    const productIds = Object.keys(tracking.products);
    if (productIds.length === 0) return;

    console.log(`[TK-STOCK] ${productIds.length} ürün için stok kontrolü başlıyor`);
    tracking.meta = tracking.meta || {};
    tracking.meta.lastAlarmFire = new Date().toISOString();
    tracking.meta.totalFetches = (tracking.meta.totalFetches || 0) + productIds.length;

    // Mevcut verilerde eksik sale event varsa bellekte onar
    // (ayrıca storage'a yazma — en sonda tek seferde yazılacak)
    for (const pid of productIds) {
        const product = tracking.products[pid];
        if (product) repairProductData(product);
    }

    // Batch: 3 ürün paralel, araya 500ms bekleme
    const batchSize = 3;
    const delayMs = 500;

    for (let i = 0; i < productIds.length; i += batchSize) {
        const batch = productIds.slice(i, i + batchSize);

        const promises = batch.map(async (pid) => {
            const product = tracking.products[pid];
            if (!product || product.status === "paused") return;

            try {
                const result = await fetchTopMerchantsStock(product);
                if (result) {
                    console.log(`[TK-STOCK] ✅ ${pid}: stok=${result.totalStock}, fiyat=${result.price}, readings önce=${(product.readings||[]).length}`);
                    processStockReading(product, result.totalStock, result.price, result.merchantStocks, result.buyboxMerchantId);
                    product.lastFetchAt = Date.now();
                    product.lastStock = result.totalStock;
                    product.lastBuyboxStock = result.buyboxStock ?? result.totalStock; // Buybox satıcının stoku (sync için)
                    product.price = result.price;
                    product.lastBuyboxMerchantId = result.buyboxMerchantId || null;
                    // merchantStocks'ı kaydet — panel sync'te diğer satıcıların stokunu da göndermek için
                    product.lastMerchantStocks = result.merchantStocks || {};
                    console.log(`[TK-STOCK] ✅ ${pid}: readings sonra=${(product.readings||[]).length}, events=${(product.events||[]).length}`);
                } else {
                    console.warn(`[TK-STOCK] ❌ ${pid}: fetchTopMerchantsStock null döndü — slug: ${product.slug}`);
                }
            } catch (e) {
                console.warn(`[TK-STOCK] ❌ Stok okuma hatası (${pid}):`, e.message);
            }

            // ─── apigw API'den taze socialProof çek ───
            // Content script'e ihtiyaç duymadan arka planda socialProof güncelle
            try {
                const freshSP = await fetchSocialProofFromApi(pid);
                if (freshSP) {
                    if (!product.panelSocialProof) product.panelSocialProof = {};
                    // Taze veriyi kaydet (mevcut content.js verisini ezmemek için sadece yoksa veya eskiyse)
                    if (freshSP.BASKET_COUNT) product.panelSocialProof.BASKET_COUNT = freshSP.BASKET_COUNT;
                    if (freshSP.ORDER_COUNT) product.panelSocialProof.ORDER_COUNT = freshSP.ORDER_COUNT;
                    if (freshSP.FAVORITE_COUNT) product.panelSocialProof.FAVORITE_COUNT = freshSP.FAVORITE_COUNT;
                    if (freshSP.PAGE_VIEW_COUNT) product.panelSocialProof.PAGE_VIEW_COUNT = freshSP.PAGE_VIEW_COUNT;
                    product.panelSocialProof._apigwFetchedAt = Date.now();
                    console.log(`[TK-STOCK] 📊 ${pid}: apigw socialProof güncellendi — B:${freshSP.BASKET_COUNT || '-'} O:${freshSP.ORDER_COUNT || '-'} F:${freshSP.FAVORITE_COUNT || '-'}`);
                }
            } catch (e) {
                // socialProof fetch hatası stok kontrolünü etkilemesin
            }
        });

        await Promise.all(promises);

        // Batch arası bekleme
        if (i + batchSize < productIds.length) {
            await new Promise(resolve => setTimeout(resolve, delayMs));
        }
    }

    // Eski readings'leri temizle
    for (const pid of productIds) {
        if (tracking.products[pid]) {
            pruneOldReadings(tracking.products[pid]);
        }
    }

    // Tek seferde storage'a yaz
    try {
        await chrome.storage.local.set({ stockTracking: tracking });
        if (chrome.runtime.lastError) {
            console.error(`[TK-STOCK] Storage yazma hatası (batch): ${chrome.runtime.lastError.message}`);
        }
    } catch (e) {
        console.error(`[TK-STOCK] Storage set hatası (batch):`, e.message);
    }

    // Storage boyutunu logla (debug)
    chrome.storage.local.getBytesInUse("stockTracking", (bytes) => {
        console.log(`[TK-STOCK] Stok kontrolü tamamlandı — ${productIds.length} ürün güncellendi — Storage: ${(bytes / 1024).toFixed(1)} KB`);
    });

    // ─── Web panele stok + T0 delta + socialProof verisini sync et ───
    try {
        const syncProducts = productIds.map(pid => {
            const p = tracking.products[pid];
            if (!p) return null;
            // stock: buybox satıcının stoku (totalStock DEĞİL!)
            // Panel processStockReading buyboxStock olarak kullanıyor.
            // totalStock göndermek → panel tek satıcıya totalStock atıyor → sahte delta
            const buyboxStock = p.lastBuyboxStock ?? p.lastStock;
            const syncItem = {
                productId: p.id,
                stock: buyboxStock,
                price: p.price,
                merchantId: p.lastBuyboxMerchantId || undefined,
                merchantName: p.buyboxMerchantName || undefined,
                otherMerchantIds: p.otherMerchantIds || undefined,
                otherMerchantNames: p.otherMerchantNames || undefined,
            };
            // Diğer satıcıların stoklarını da gönder — panel per-merchant delta için gerekli
            if (p.lastMerchantStocks && typeof p.lastMerchantStocks === 'object') {
                const otherStocks = {};
                for (const [mid, mstock] of Object.entries(p.lastMerchantStocks)) {
                    if (mid !== String(p.lastBuyboxMerchantId) && !mid.startsWith('_')) {
                        otherStocks[mid] = mstock;
                    }
                }
                if (Object.keys(otherStocks).length > 0) {
                    syncItem.otherMerchantStocks = otherStocks;
                }
            }

            // Extension'ın kendi T0 delta hesabını panele aktar
            const est = p.estimates || {};
            if (est.monthlyEstimate > 0) {
                syncItem.stockDeltaMonthlySales = est.monthlyEstimate;
                syncItem.stockDeltaDaysTracked = (p.confidence && p.confidence.daysTracked) || 0;
            }

            // Favori sayısı (extension storage'da varsa)
            if (p.favoriteCount > 0) syncItem.favoriteCount = p.favoriteCount;

            // socialProof gönder — apigw'den taze çekilen + content.js'den toplanan veriler
            if (p.panelSocialProof && typeof p.panelSocialProof === 'object') {
                const sp = {};
                if (p.panelSocialProof.ORDER_COUNT) sp.ORDER_COUNT = p.panelSocialProof.ORDER_COUNT;
                if (p.panelSocialProof.BASKET_COUNT) sp.BASKET_COUNT = p.panelSocialProof.BASKET_COUNT;
                if (p.panelSocialProof.RECENT_SALES_COUNT) {
                    sp.RECENT_SALES_COUNT = p.panelSocialProof.RECENT_SALES_COUNT;
                    sp.RECENT_SALES_DAYS = p.panelSocialProof.RECENT_SALES_DAYS || 3;
                }
                if (p.panelSocialProof.FAVORITE_COUNT) sp.FAVORITE_COUNT = p.panelSocialProof.FAVORITE_COUNT;
                if (Object.keys(sp).length > 0) syncItem.socialProof = sp;
            }

            // ─── extensionEstimate: Panel T0 bazlı tahmine uyumlu ───
            // Panel zaten T0+T1+T3 ensemble hesaplıyor (sales-utils.js).
            // Extension kendi ham T0'sunu göndermek yerine:
            //   1) panelEstimate varsa → onu gönder (panel'in kendi T0 bazlı ensemble'ı)
            //   2) panelEstimate yoksa → kendi ham T0'sunu gönder (yeni ürün vb.)
            // Böylece panel-extension döngüsü kırılır, tutarlı sonuç.
            const pe = p.panelEstimate;
            if (pe && pe.monthly > 0 && pe.tier) {
                // Panel'in T0 bazlı hesabını kullan — en doğru sonuç
                syncItem.extensionEstimate = {
                    monthly: pe.monthly,
                    daily: pe.daily || Math.round(pe.monthly / 30),
                    tier: pe.tier,
                    min: pe.min || Math.round(pe.monthly * 0.8),
                    max: pe.max || Math.round(pe.monthly * 1.2),
                };
            } else if (est.monthlyEstimate > 0) {
                // Panel tahmini yok → kendi ham T0'muzu gönder
                const monthly = est.monthlyEstimate;
                const daily = Math.round(monthly / 30);
                const spread = est.source === 'ema' ? 0.30 : 0.25;
                const min = Math.round(monthly / (1 + spread));
                const max = Math.round(monthly * (1 + spread));
                syncItem.extensionEstimate = {
                    monthly: monthly,
                    daily: daily,
                    tier: 'T0',
                    min: min,
                    max: max,
                };
            }

            return syncItem;
        }).filter(Boolean);
        if (syncProducts.length > 0) {
            syncToPanel(syncProducts);
        }
    } catch (e) {
        // Sync hatası stok kontrolünü etkilemesin
    }
}

// ═══════════════════════════════════════════════════════════════════
//  PROAKTİF SYNC — Trendyol açıkken arka planda stok + socialProof güncelleme
//  ─────────────────────────────────────────────────────────────────
//  Alarm (60dk) arasında panel verisini güncel tutmak için:
//  Trendyol herhangi bir sayfası açıkken content script sinyal gönderir,
//  bu fonksiyon 15dk aralıkla:
//    1) apigw socialProof çeker (tüm ürünler, hafif)
//    2) SSR detay sayfasından stok çeker (buybox stok + fiyat + diğer satıcılar)
//  Tam per-merchant stok fetch YAPMAZ (sadece buybox — alarm'da tam fetch yapılır)
// ═══════════════════════════════════════════════════════════════════

const PROACTIVE_SYNC_INTERVAL = 15 * 60 * 1000; // 15 dakika (ms)
const PROACTIVE_MAX_PRODUCTS = 100; // Tek seferde max ürün (rate limit koruması)
const PROACTIVE_STOCK_BATCH_SIZE = 2; // Stok fetch batch boyutu (SSR ağır — 2'şer)
const PROACTIVE_STOCK_DELAY = 600; // Stok fetch arası bekleme (ms)
let _lastProactiveSyncTime = 0;
let _proactiveSyncRunning = false;

// Panel'den TÜM aktif ürünlerin listesini çek (global — tüm kullanıcıların ürünleri)
async function fetchGlobalProductList() {
    const auth = await getPanelAuth();
    if (!auth || !auth.token) return null;

    try {
        const panelUrl = await getPanelUrl();
        const resp = await fetch(`${panelUrl}/api/extension/global-products`, {
            headers: {
                "Authorization": `Bearer ${auth.token}`,
                "Content-Type": "application/json",
            },
            signal: AbortSignal.timeout(8000),
        });
        if (resp.ok) {
            const data = await resp.json();
            console.log(`[TK-PROACTIVE] Global ürün listesi: ${data.count} ürün`);
            return data.products || [];
        } else if (resp.status === 401) {
            console.warn("[TK-PROACTIVE] Token süresi dolmuş");
            return null;
        }
    } catch (e) {
        console.log(`[TK-PROACTIVE] Global ürün listesi çekme hatası: ${e.message}`);
    }
    return null;
}

async function proactiveSyncTrackedProducts() {
    if (_proactiveSyncRunning) return;

    const now = Date.now();
    if (now - _lastProactiveSyncTime < PROACTIVE_SYNC_INTERVAL) {
        console.log("[TK-PROACTIVE] Son sync'ten bu yana 15dk geçmedi, atlanıyor");
        return;
    }

    const auth = await getPanelAuth();
    if (!auth || !auth.token) return;

    _proactiveSyncRunning = true;
    _lastProactiveSyncTime = now;

    try {
        // 1) Panel'den GLOBAL ürün listesini çek (tüm kullanıcıların ürünleri)
        const globalProducts = await fetchGlobalProductList();
        if (!globalProducts || globalProducts.length === 0) {
            console.log("[TK-PROACTIVE] Global ürün listesi boş veya alınamadı");
            return;
        }

        // 2) Kendi stockTracking'imizi de al (varsa mevcut stok/socialProof verisi kullanılır)
        const stData = await chrome.storage.local.get("stockTracking");
        const tracking = stData.stockTracking || { products: {} };
        const myProducts = tracking.products || {};

        // 3) Max ürün sınırı uygula (rate limit koruması)
        const productsToSync = globalProducts.slice(0, PROACTIVE_MAX_PRODUCTS);

        console.log(`[TK-PROACTIVE] ${productsToSync.length} ürün için sync başlıyor (toplam global: ${globalProducts.length})`);

        // ─── ADIM A: Tüm ürünler için apigw socialProof çek (hafif, 3'erli batch) ───
        const spBatchSize = 3;
        const spDelayMs = 300;
        let spUpdated = 0;
        const socialProofMap = {}; // pid → freshSP

        for (let i = 0; i < productsToSync.length; i += spBatchSize) {
            const batch = productsToSync.slice(i, i + spBatchSize);
            const promises = batch.map(async (gp) => {
                const pid = String(gp.id);
                try {
                    const freshSP = await fetchSocialProofFromApi(pid);
                    if (freshSP) {
                        socialProofMap[pid] = freshSP;
                        spUpdated++;
                    }
                } catch (e) { /* silent */ }
            });
            await Promise.all(promises);
            if (i + spBatchSize < productsToSync.length) {
                await new Promise(resolve => setTimeout(resolve, spDelayMs));
            }
        }

        console.log(`[TK-PROACTIVE] SocialProof: ${spUpdated}/${productsToSync.length} güncellendi`);

        // ─── ADIM B: Tüm ürünler için SSR'dan stok + fiyat çek (2'şerli batch, daha yavaş) ───
        let stockUpdated = 0;
        const stockDataMap = {}; // pid → { stock, price, merchantId, merchantName, otherMerchants }

        for (let i = 0; i < productsToSync.length; i += PROACTIVE_STOCK_BATCH_SIZE) {
            const batch = productsToSync.slice(i, i + PROACTIVE_STOCK_BATCH_SIZE);
            const promises = batch.map(async (gp) => {
                const pid = String(gp.id);
                const slug = gp.slug;
                if (!slug) return;

                try {
                    const mainData = await fetchStockFromDetailPageFull(slug);
                    if (mainData && mainData.stock >= 0) {
                        stockDataMap[pid] = mainData;
                        stockUpdated++;
                    }
                } catch (e) {
                    // SSR fetch hatası — sessiz devam
                }
            });
            await Promise.all(promises);
            if (i + PROACTIVE_STOCK_BATCH_SIZE < productsToSync.length) {
                await new Promise(resolve => setTimeout(resolve, PROACTIVE_STOCK_DELAY));
            }
        }

        console.log(`[TK-PROACTIVE] Stok: ${stockUpdated}/${productsToSync.length} ürün için taze stok çekildi`);

        // ─── ADIM C: SyncItem'lar oluştur + panel'e gönder ───
        const syncResults = [];

        for (const gp of productsToSync) {
            const pid = String(gp.id);
            const myProduct = myProducts[pid];
            const freshSP = socialProofMap[pid] || null;
            const freshStock = stockDataMap[pid] || null;

            const syncItem = { productId: pid };

            // Stok + fiyat: taze SSR verisi varsa kullan, yoksa mevcut tracking'den
            if (freshStock) {
                syncItem.stock = freshStock.stock;
                syncItem.price = freshStock.price || (myProduct ? myProduct.price : undefined);
                syncItem.merchantId = freshStock.merchantId || undefined;
                syncItem.merchantName = freshStock.merchantName || undefined;

                // Diğer satıcı bilgileri (ID + isim + stok)
                if (freshStock.otherMerchants && freshStock.otherMerchants.length > 0) {
                    syncItem.otherMerchantIds = freshStock.otherMerchants.map(m => String(m.id));
                    const nameMap = {};
                    const stockMap = {};
                    freshStock.otherMerchants.forEach(m => {
                        if (m.name) nameMap[String(m.id)] = m.name;
                        if (m.stock > 0) stockMap[String(m.id)] = m.stock;
                    });
                    if (Object.keys(nameMap).length > 0) syncItem.otherMerchantNames = nameMap;
                    if (Object.keys(stockMap).length > 0) syncItem.otherMerchantStocks = stockMap;
                }

                // Kendi tracking'imizi de güncelle (lastStock, price)
                if (myProduct) {
                    myProduct.lastStock = freshStock.stock;
                    myProduct.price = freshStock.price || myProduct.price;
                    myProduct.lastBuyboxMerchantId = freshStock.merchantId || myProduct.lastBuyboxMerchantId;
                    myProduct.buyboxMerchantName = freshStock.merchantName || myProduct.buyboxMerchantName;
                    myProduct.lastFetchAt = now;
                    // Diğer satıcı bilgilerini güncelle
                    if (freshStock.otherMerchants && freshStock.otherMerchants.length > 0) {
                        myProduct.otherMerchantIds = freshStock.otherMerchants.map(m => String(m.id));
                        const nameMap = {};
                        const merchantStocks = {};
                        freshStock.otherMerchants.forEach(m => {
                            if (m.name) nameMap[String(m.id)] = m.name;
                            if (m.stock > 0) merchantStocks[String(m.id)] = m.stock;
                        });
                        if (Object.keys(nameMap).length > 0) myProduct.otherMerchantNames = nameMap;
                        // merchantStocks cache: buybox + diğerleri (SSR fail durumunda fallback)
                        if (freshStock.merchantId && freshStock.stock > 0) {
                            merchantStocks[String(freshStock.merchantId)] = freshStock.stock;
                        }
                        if (Object.keys(merchantStocks).length > 0) {
                            myProduct.lastMerchantStocks = merchantStocks;
                        }
                    }
                }
            } else if (myProduct) {
                // SSR fetch başarısız — mevcut tracking verisini gönder
                // lastBuyboxStock: buybox satıcının stoku (lastStock = totalStock, panele buybox gönder)
                syncItem.stock = myProduct.lastBuyboxStock ?? myProduct.lastStock;
                syncItem.price = myProduct.price;
                syncItem.merchantId = myProduct.lastBuyboxMerchantId || undefined;
                syncItem.merchantName = myProduct.buyboxMerchantName || undefined;
                syncItem.otherMerchantIds = myProduct.otherMerchantIds || undefined;
                syncItem.otherMerchantNames = myProduct.otherMerchantNames || undefined;
                // Mevcut merchantStocks varsa diğer satıcıları da gönder
                if (myProduct.lastMerchantStocks && typeof myProduct.lastMerchantStocks === 'object') {
                    const otherStocks = {};
                    for (const [mid, mstock] of Object.entries(myProduct.lastMerchantStocks)) {
                        if (mid !== String(myProduct.lastBuyboxMerchantId) && !mid.startsWith('_')) {
                            otherStocks[mid] = mstock;
                        }
                    }
                    if (Object.keys(otherStocks).length > 0) {
                        syncItem.otherMerchantStocks = otherStocks;
                    }
                }
            }

            // SocialProof: taze apigw verisi + tracking'deki mevcut veri
            if (myProduct) {
                if (freshSP) {
                    if (!myProduct.panelSocialProof) myProduct.panelSocialProof = {};
                    if (freshSP.BASKET_COUNT) myProduct.panelSocialProof.BASKET_COUNT = freshSP.BASKET_COUNT;
                    if (freshSP.ORDER_COUNT) myProduct.panelSocialProof.ORDER_COUNT = freshSP.ORDER_COUNT;
                    if (freshSP.FAVORITE_COUNT) myProduct.panelSocialProof.FAVORITE_COUNT = freshSP.FAVORITE_COUNT;
                    if (freshSP.PAGE_VIEW_COUNT) myProduct.panelSocialProof.PAGE_VIEW_COUNT = freshSP.PAGE_VIEW_COUNT;
                    myProduct.panelSocialProof._proactiveFetchedAt = now;
                }

                // Mevcut socialProof'u sync payload'a ekle
                if (myProduct.panelSocialProof && typeof myProduct.panelSocialProof === "object") {
                    const sp = {};
                    if (myProduct.panelSocialProof.ORDER_COUNT) sp.ORDER_COUNT = myProduct.panelSocialProof.ORDER_COUNT;
                    if (myProduct.panelSocialProof.BASKET_COUNT) sp.BASKET_COUNT = myProduct.panelSocialProof.BASKET_COUNT;
                    if (myProduct.panelSocialProof.RECENT_SALES_COUNT) {
                        sp.RECENT_SALES_COUNT = myProduct.panelSocialProof.RECENT_SALES_COUNT;
                        sp.RECENT_SALES_DAYS = myProduct.panelSocialProof.RECENT_SALES_DAYS || 3;
                    }
                    if (myProduct.panelSocialProof.FAVORITE_COUNT) sp.FAVORITE_COUNT = myProduct.panelSocialProof.FAVORITE_COUNT;
                    if (Object.keys(sp).length > 0) syncItem.socialProof = sp;
                }

                // panelEstimate passthrough
                const pe = myProduct.panelEstimate;
                if (pe && pe.monthly > 0 && pe.tier) {
                    syncItem.extensionEstimate = {
                        monthly: pe.monthly,
                        daily: pe.daily || Math.round(pe.monthly / 30),
                        tier: pe.tier,
                        min: pe.min || Math.round(pe.monthly * 0.8),
                        max: pe.max || Math.round(pe.monthly * 1.2),
                    };
                } else {
                    const est = myProduct.estimates || {};
                    if (est.monthlyEstimate > 0) {
                        const monthly = est.monthlyEstimate;
                        const daily = Math.round(monthly / 30);
                        const spread = est.source === "ema" ? 0.30 : 0.25;
                        syncItem.extensionEstimate = {
                            monthly, daily, tier: "T0",
                            min: Math.round(monthly / (1 + spread)),
                            max: Math.round(monthly * (1 + spread)),
                        };
                    }
                }
                if (myProduct.favoriteCount > 0) syncItem.favoriteCount = myProduct.favoriteCount;

            } else {
                // Kendi tracking'imizde YOK — sadece taze socialProof gönder
                if (freshSP) {
                    syncItem.socialProof = freshSP;
                }
            }

            // En az bir veri varsa sync listesine ekle
            if (syncItem.socialProof || syncItem.stock != null || syncItem.price) {
                syncResults.push(syncItem);
            }
        }

        // 4) Kendi stockTracking'i güncelle (socialProof + stok cache)
        if ((spUpdated > 0 || stockUpdated > 0) && Object.keys(myProducts).length > 0) {
            await chrome.storage.local.set({ stockTracking: tracking });
        }

        // 5) Panel'e toplu sync et (50'şerli batch — panel limiti)
        if (syncResults.length > 0) {
            const SYNC_BATCH_SIZE = 50;
            for (let i = 0; i < syncResults.length; i += SYNC_BATCH_SIZE) {
                const batch = syncResults.slice(i, i + SYNC_BATCH_SIZE);
                await syncToPanel(batch);
                if (i + SYNC_BATCH_SIZE < syncResults.length) {
                    await new Promise(resolve => setTimeout(resolve, 500));
                }
            }
        }

        console.log(`[TK-PROACTIVE] Tamamlandı: SP=${spUpdated}, Stok=${stockUpdated}, Sync=${syncResults.length} ürün panel'e gönderildi`);

    } catch (e) {
        console.warn("[TK-PROACTIVE] Hata:", e.message);
    } finally {
        _proactiveSyncRunning = false;
    }
}

// ═══════════════════════════════════════════════════════════════════
//  BEKLEYEN DOĞRULAMA SİSTEMİ (Pending Verification)
//  ─────────────────────────────────────────────────────────────────
//  Şüpheli stok düşüşlerini hemen anomali ilan etmek yerine,
//  sonraki ölçümlere bakarak doğrulama yapar.
//
//  Senaryo A — Influencer/kampanya (gerçek satış):
//    Ölçüm 5: stok 1000 → 400 (Δ=-600, şüpheli) → pending_verification
//    Ölçüm 6: stok 400 → 350 (Δ=-50, hız hâlâ yüksek) → doğrulama devam
//    Ölçüm 7: stok 350 → 290 (Δ=-60, yüksek hız sürdü) → ONAYLA: gerçek satış!
//    → pending_verification → sale olarak çevrilir, 600 adet satış sayılır
//
//  Senaryo B — Satıcı stok düşürdü:
//    Ölçüm 5: stok 1000 → 400 (Δ=-600, şüpheli) → pending_verification
//    Ölçüm 6: stok 400 → 398 (Δ=-2, normal hız) → doğrulama devam
//    Ölçüm 7: stok 398 → 396 (Δ=-2, normal) → KESİNLEŞTİR: stok ayarlaması!
//    → pending_verification → stock_adjustment olarak çevrilir, 600 sayılmaz
// ═══════════════════════════════════════════════════════════════════

function resolvePendingVerifications(product, currentStock, now) {
    if (!product.events || product.events.length === 0) return;

    // Bekleyen doğrulamaları bul
    for (let i = 0; i < product.events.length; i++) {
        const ev = product.events[i];
        if (ev.type !== "pending_verification") continue;

        // Bu pending_verification'dan sonra kaç okuma yapıldı?
        const readingsAfter = product.readings.length - (ev.readingsAtDetection || 0);

        if (readingsAfter < (ev.verificationsNeeded || STOCK_ANOMALY_CONFIRM_READINGS)) {
            // Henüz yeterli doğrulama ölçümü yok — bekle
            continue;
        }

        // ─── DOĞRULAMA: Sonraki ölçümlerdeki satış hızını karşılaştır ───
        // Pending event'ten sonraki ölçümlerdeki ortalama saatlik düşüş hızı
        const pendingReadingIdx = (ev.readingsAtDetection || 2) - 1; // pending olduğu andaki son reading index'i
        const afterReadings = product.readings.slice(pendingReadingIdx);

        let totalDropAfter = 0;
        let totalHoursAfter = 0;
        let dropSegments = 0;

        for (let r = 1; r < afterReadings.length; r++) {
            const segDelta = afterReadings[r].totalStock - afterReadings[r - 1].totalStock;
            const segHours = Math.max((afterReadings[r].ts - afterReadings[r - 1].ts) / 3600000, 0.1);
            totalHoursAfter += segHours;
            if (segDelta < 0) {
                totalDropAfter += Math.abs(segDelta);
                dropSegments++;
            }
        }

        const afterHourlyRate = totalHoursAfter > 0 ? totalDropAfter / totalHoursAfter : 0;

        // Pending event'teki düşüş hızı
        const pendingHourlyRate = ev.hourlyRate || 0;

        // Karar: Sonraki satış hızı, pending düşüşün hızının en az %20'si kadarsa
        // → yüksek tempo devam etti → gerçek satış (influencer/kampanya)
        // Aksi halde → satıcı stok düşürmüş
        //
        // Ek kontrol: Sonraki ölçümlerde hiç stok düşüşü yoksa (dropSegments=0)
        // → kesinlikle satıcı düşürmüş
        const emaAtDetection = ev.emaRateAtDetection || 0;
        // Referans hız: pending öncesi bilinen EMA (yoksa pending hızının %20'si)
        const referenceRate = emaAtDetection > 0 ? emaAtDetection : pendingHourlyRate * 0.2;
        // Eğer sonraki hız referansın en az %50'si kadarsa → trend gerçek, satış sayılır
        const isRealSale = dropSegments >= 1 && afterHourlyRate >= referenceRate * 0.5;

        if (isRealSale) {
            // ✅ Gerçek satış — influencer/kampanya etkisi
            // pending_verification → sale olarak dönüştür
            product.events[i] = {
                ts: ev.ts, type: "sale", delta: ev.delta, sold: ev.drop,
                prevStock: ev.prevStock, newStock: ev.newStock,
                elapsedHours: ev.elapsedHours,
                hourlyRate: ev.hourlyRate,
                verifiedAt: now,
                verificationResult: "confirmed_sale",
                afterHourlyRate: Math.round(afterHourlyRate * 1000) / 1000
            };
            console.log(`[TK-STOCK] ✅ Doğrulandı — GERÇEK SATIŞ: ${product.id} — ${ev.drop} adet, sonraki hız=${afterHourlyRate.toFixed(2)}/saat (referans: ${referenceRate.toFixed(2)})`);

            // Sale event olduğu güne ait dailySummary'ye ekle
            const dateKey = new Date(ev.ts).toISOString().split("T")[0];
            if (product.dailySummaries && product.dailySummaries[dateKey]) {
                product.dailySummaries[dateKey].totalSold += ev.drop;
                product.dailySummaries[dateKey].hourlyRates.push(ev.hourlyRate);
            }
        } else {
            // ❌ Satıcı stok düşürmüş — satış sayılmaz
            product.events[i] = {
                ts: ev.ts, type: "stock_adjustment", delta: ev.delta, drop: ev.drop,
                prevStock: ev.prevStock, newStock: ev.newStock,
                elapsedHours: ev.elapsedHours,
                verifiedAt: now,
                verificationResult: "confirmed_adjustment",
                afterHourlyRate: Math.round(afterHourlyRate * 1000) / 1000,
                reason: "anomaly_verified"
            };
            console.log(`[TK-STOCK] ❌ Doğrulandı — STOK AYARLAMASI: ${product.id} — ${ev.drop} adet, sonraki hız=${afterHourlyRate.toFixed(2)}/saat (referans: ${referenceRate.toFixed(2)})`);
        }

        // Tahminleri yeniden hesapla (event tipi değişti)
        computeSalesEstimates(product);
        computeConfidence(product);
    }
}

// ═══════════════════════════════════════════════════════════════════
//  BUYBOX + VIRTUAL STOK YARDIMCI FONKSİYONLARI
// ═══════════════════════════════════════════════════════════════════

/**
 * Buybox switch tespiti — _buybox meta key üzerinden karşılaştırma
 * @returns {boolean} true ise buybox switch gerçekleşmiş
 */
function detectBuyboxSwitch(prevMerchants, currMerchants, prevStock, newTotalStock) {
    const prevBuyboxId = prevMerchants._buybox || Object.keys(prevMerchants).find(k => k !== '_buybox') || null;
    const currBuyboxId = currMerchants._buybox || Object.keys(currMerchants).find(k => k !== '_buybox') || null;

    const prevIds = Object.keys(prevMerchants).filter(k => k !== '_buybox');
    const currIds = Object.keys(currMerchants).filter(k => k !== '_buybox');

    if (prevIds.length === 0 || currIds.length === 0) return false;

    const buyboxChanged = prevBuyboxId && currBuyboxId && prevBuyboxId !== currBuyboxId;

    if (buyboxChanged) {
        const stockDiff = Math.abs(newTotalStock - prevStock);
        const pctChange = prevStock > 0 ? (stockDiff / prevStock) * 100 : 100;

        if (stockDiff >= BUYBOX_SWITCH_MIN_JUMP || pctChange >= BUYBOX_SWITCH_PCT) {
            console.log(`[TK-STOCK] BUYBOX SWITCH tespit: satici ${prevBuyboxId} → ${currBuyboxId}, stok ${prevStock} → ${newTotalStock} (fark: ${newTotalStock - prevStock})`);
            return true;
        }
    }

    // Ek kontrol: Satıcı ID'leri tamamen farklı mı
    const commonIds = prevIds.filter(id => currIds.includes(id));
    if (commonIds.length === 0 && prevIds.length > 0 && currIds.length > 0) {
        console.log(`[TK-STOCK] BUYBOX SWITCH tespit: Tum satici ID'leri degisti — onceki: [${prevIds.join(',')}] → simdi: [${currIds.join(',')}]`);
        return true;
    }

    // Ek kontrol: Buybox değiştiyse ve eski/yeni satıcı birbirinin listesinde yoksa
    if (buyboxChanged) {
        const oldBuyboxNowStock = currMerchants[prevBuyboxId] || 0;
        const newBuyboxPrevStock = prevMerchants[currBuyboxId] || 0;

        if (oldBuyboxNowStock === 0 && newBuyboxPrevStock === 0) {
            const stockJump = Math.abs(newTotalStock - prevStock);
            if (stockJump > 100) {
                console.log(`[TK-STOCK] BUYBOX SWITCH tespit: Yeni satici daha once izlenmiyordu, stok atladi ${stockJump}`);
                return true;
            }
        }
    }

    return false;
}

/**
 * Buybox switch sırasında sadece ortak satıcıların delta'larını hesapla
 * _buybox meta key'i atlar
 */
function calculateSalesAcrossBuyboxSwitch(prevMerchants, currMerchants) {
    let commonDrop = 0;
    let commonSupply = 0;
    const commonDeltas = {};

    for (const mid of Object.keys(prevMerchants)) {
        if (mid === '_buybox') continue;
        if (mid in currMerchants) {
            const delta = Number(currMerchants[mid]) - Number(prevMerchants[mid]);
            if (delta < 0) {
                commonDrop += Math.abs(delta);
                commonDeltas[mid] = { prev: Number(prevMerchants[mid]), curr: Number(currMerchants[mid]), delta };
            } else if (delta > 0) {
                commonSupply += delta;
                commonDeltas[mid] = { prev: Number(prevMerchants[mid]), curr: Number(currMerchants[mid]), delta };
            }
        }
    }

    return { commonDrop, commonSupply, commonDeltas };
}

/**
 * Buybox flip-flop tespiti — 6 saat içinde A→B→A paterni
 * >=2 flip varsa flip-flop → sahte event üretilmez
 */
function detectBuyboxFlipFlop(readings, currBuyboxId, nowTs) {
    const sixHoursAgo = nowTs - 6 * 3600000;
    const recentReadings = readings.filter(r => r.ts >= sixHoursAgo);
    if (recentReadings.length < 4) return false;

    const buyboxSequence = recentReadings.map(r => {
        const ms = r.merchants || {};
        return ms._buybox || Object.keys(ms).find(k => k !== '_buybox') || null;
    }).filter(Boolean);

    if (buyboxSequence.length < 4) return false;

    let flipCount = 0;
    for (let i = 2; i < buyboxSequence.length; i++) {
        if (buyboxSequence[i] === buyboxSequence[i - 2] && buyboxSequence[i] !== buyboxSequence[i - 1]) {
            flipCount++;
        }
    }

    return flipCount >= 2;
}

/**
 * Virtual (sanal) stok tespiti — 3 kriter
 * Returns: 'virtual' veya 'delta'
 */
function detectVirtualStock(readings) {
    if (!readings || readings.length < 12) return null;

    const nowTs = Date.now();
    const oneDayAgo = nowTs - 24 * 3600000;
    const recentReadings = readings.filter(r => r.ts >= oneDayAgo);

    if (recentReadings.length < 8) return null;

    const stockValues = recentReadings.map(r => r.totalStock);
    const uniqueStocks = [...new Set(stockValues)];

    const firstStock = stockValues[0];
    const lastStock = stockValues[stockValues.length - 1];
    const minStock = Math.min(...stockValues);
    const maxStock = Math.max(...stockValues);

    // Kriter 1: Stok hiç düşmemiş + >5000 + <=4 unique değer
    if (lastStock >= firstStock && maxStock > 5000 && uniqueStocks.length <= 4) {
        return 'virtual';
    }

    // Kriter 2: Min stok >10000 + spread <%1 + 10+ reading
    const spread = maxStock - minStock;
    const spreadPct = minStock > 0 ? (spread / minStock) * 100 : 0;
    if (minStock > 10000 && spreadPct < 1 && recentReadings.length >= 10) {
        return 'virtual';
    }

    // Kriter 3: Hiçbir anlamlı düşüş yok (>%0.5 veya >10 birim) + >5000
    let hasAnyDrop = false;
    for (let i = 1; i < recentReadings.length; i++) {
        if (recentReadings[i].totalStock < recentReadings[i - 1].totalStock) {
            const drop = recentReadings[i - 1].totalStock - recentReadings[i].totalStock;
            const dropPct = recentReadings[i - 1].totalStock > 0 ? (drop / recentReadings[i - 1].totalStock) * 100 : 0;
            if (dropPct >= 0.5 || drop >= 10) {
                hasAnyDrop = true;
                break;
            }
        }
    }
    if (!hasAnyDrop && maxStock > 5000) {
        return 'virtual';
    }

    return 'delta';
}

// ═══════════════════════════════════════════════════════════════════
//  STOK DELTA HESAPLAMA MOTORU v2
//  ─────────────────────────────────────────────────────────────────
//  Model: Her iki okuma arasında satış hızı (hourlyRate) hesapla.
//  İlk okumadan itibaren tahmin üret — 1 gün beklemeye gerek yok.
//
//  Örnek:
//    Okuma 1 → stok: 100  (t=0)
//    Okuma 2 → stok: 98   (t=2h) → Δ=-2, 2h'de 2 satış → 1/saat → 24/gün
//    Okuma 3 → stok: 97   (t=4h) → Δ=-1, 2h'de 1 satış → 0.5/saat
//    hourlyRate EMA = 0.5×0.3 + 1.0×0.7 = 0.85 → 20.4/gün
//
//  Kademeli ilerleme:
//    < 24 saat  → saatlik ölçümlerden günlük tahmin (projectedDaily)
//    1-7 gün    → günlük gerçek satış ortalaması
//    7-30 gün   → haftalık ortalama günlük satış
//    30+ gün    → aylık ortalama (en güvenilir)
// ═══════════════════════════════════════════════════════════════════

function processStockReading(product, newTotalStock, newPrice, merchantStocks, buyboxMerchantId) {
    const now = Date.now();

    // ─── _buybox meta key enrichment ───
    // Her reading'e buybox merchant ID'yi meta olarak kaydet
    const enrichedMerchantStocks = { ...(merchantStocks || {}) };
    if (buyboxMerchantId) {
        enrichedMerchantStocks._buybox = String(buyboxMerchantId);
    } else if (Object.keys(enrichedMerchantStocks).length > 0) {
        const firstKey = Object.keys(enrichedMerchantStocks).find(k => k !== '_buybox');
        if (firstKey) enrichedMerchantStocks._buybox = firstKey;
    }

    // ─── 20dk reading dedup ───
    if (!product.readings) product.readings = [];
    if (product.readings.length > 0) {
        const lastReading = product.readings[product.readings.length - 1];
        const timeDiff = now - lastReading.ts;
        const DEDUP_WINDOW_MS = 20 * 60 * 1000;

        if (timeDiff < DEDUP_WINDOW_MS) {
            if (lastReading.totalStock === newTotalStock && lastReading.price === newPrice) {
                const prevM = lastReading.merchants || {};
                const currM = enrichedMerchantStocks;
                const prevKeys = Object.keys(prevM).filter(k => k !== '_buybox').sort();
                const currKeys = Object.keys(currM).filter(k => k !== '_buybox').sort();
                let sameStocks = prevKeys.length === currKeys.length && prevKeys.join(',') === currKeys.join(',');
                if (sameStocks) {
                    for (const k of prevKeys) {
                        if (Number(prevM[k]) !== Number(currM[k])) { sameStocks = false; break; }
                    }
                }
                if (sameStocks) {
                    console.log(`[TK-STOCK] Dedup: ${product.id} — ayni veri 20dk icinde, atlaniyor`);
                    return;
                }
            }
        }
    }

    // 1) Okuma kaydet
    const reading = {
        ts: now,
        totalStock: newTotalStock,
        price: newPrice,
        merchants: enrichedMerchantStocks
    };
    product.readings.push(reading);

    // 2) Bekleyen doğrulamaları kontrol et (pending_verification)
    if (!product.events) product.events = [];
    resolvePendingVerifications(product, newTotalStock, now);

    // 3) Olay sınıflandırma + event kaydetme
    //    ─── BUYBOX-AWARE SATICI BAZLI DELTA HESAPLAMA ───
    if (product.readings.length >= 2) {
        const prevReading = product.readings[product.readings.length - 2];
        const elapsedMs = now - prevReading.ts;
        const elapsedHours = Math.max(elapsedMs / 3600000, 0.1);

        const prevMerchants = prevReading.merchants || {};
        const currMerchants = enrichedMerchantStocks;

        // ─── Buybox switch tespiti ───
        const isBuyboxSwitch = detectBuyboxSwitch(prevMerchants, currMerchants, prevReading.totalStock, newTotalStock);

        if (isBuyboxSwitch) {
            const currBuyboxId = currMerchants._buybox || Object.keys(currMerchants).find(k => k !== '_buybox') || null;
            const isFlipFlop = detectBuyboxFlipFlop(product.readings, currBuyboxId, now);

            if (isFlipFlop) {
                // Flip-flop: sahte sale/supply uretme, sadece bilgi event'i logla
                product.events.push({
                    ts: now, type: "buybox_switch",
                    prevBuybox: prevMerchants._buybox || null,
                    newBuybox: currBuyboxId,
                    prevStock: prevReading.totalStock,
                    newStock: newTotalStock,
                    flipFlop: true
                });
                console.log(`[TK-STOCK] 🔄 Buybox flip-flop: ${product.id} — sahte event uretilmedi`);
            } else {
                // Normal buybox switch — sadece common merchant delta'larini say
                const { commonDrop, commonSupply, commonDeltas } = calculateSalesAcrossBuyboxSwitch(prevMerchants, currMerchants);

                product.events.push({
                    ts: now, type: "buybox_switch",
                    prevBuybox: prevMerchants._buybox || null,
                    newBuybox: currBuyboxId,
                    prevStock: prevReading.totalStock,
                    newStock: newTotalStock,
                    commonDrop, commonSupply
                });

                if (commonDrop > 0) {
                    const hourlyRate = commonDrop / elapsedHours;
                    product.events.push({
                        ts: now, type: "sale", delta: -commonDrop, sold: commonDrop,
                        prevStock: prevReading.totalStock, newStock: newTotalStock,
                        elapsedHours: Math.round(elapsedHours * 100) / 100,
                        hourlyRate: Math.round(hourlyRate * 1000) / 1000,
                        merchantDeltas: commonDeltas,
                        source: "buybox_common"
                    });
                }
                console.log(`[TK-STOCK] 🔀 Buybox switch: ${product.id} — commonDrop=${commonDrop}`);
            }
        } else {
            // Normal delta hesaplama (_buybox key'i filtrele)
            const allMerchantIds = new Set([
                ...Object.keys(prevMerchants).filter(k => k !== '_buybox'),
                ...Object.keys(currMerchants).filter(k => k !== '_buybox')
            ]);

            let totalMerchantDrop = 0;
            let totalMerchantSupply = 0;
            const merchantDeltas = {};

            for (const mid of allMerchantIds) {
                const prevQty = Number(prevMerchants[mid]) || 0;
                const currQty = Number(currMerchants[mid]) || 0;
                const mDelta = currQty - prevQty;

                if (mDelta !== 0) {
                    merchantDeltas[mid] = { prev: prevQty, curr: currQty, delta: mDelta };
                }

                if (mDelta < 0) {
                    totalMerchantDrop += Math.abs(mDelta);
                } else if (mDelta > 0) {
                    totalMerchantSupply += mDelta;
                }
            }

            const hasMerchantData = allMerchantIds.size > 0 && (Object.keys(prevMerchants).filter(k => k !== '_buybox').length > 0 || Object.keys(currMerchants).filter(k => k !== '_buybox').length > 0);

            let effectiveDrop, effectiveSupply, effectiveDelta;

            if (hasMerchantData && allMerchantIds.size > 0) {
                effectiveDrop = totalMerchantDrop;
                effectiveSupply = totalMerchantSupply;
                effectiveDelta = newTotalStock - prevReading.totalStock;

                if (Object.keys(merchantDeltas).length > 0) {
                    console.log(`[TK-STOCK] 📊 Satıcı bazlı delta: ${product.id}`, JSON.stringify(merchantDeltas));
                }
            } else {
                const simpleDelta = newTotalStock - prevReading.totalStock;
                effectiveDelta = simpleDelta;
                effectiveDrop = simpleDelta < 0 ? Math.abs(simpleDelta) : 0;
                effectiveSupply = simpleDelta > 0 ? simpleDelta : 0;
            }

            // ─── Tedarik / İade event'i ───
            if (effectiveSupply > 0) {
                const prevStock = prevReading.totalStock;
                const pctInc = prevStock > 0 ? (effectiveSupply / prevStock) * 100 : 100;
                if (prevStock === 0 || (effectiveSupply > STOCK_SUPPLY_THRESHOLD_ABS && pctInc > STOCK_SUPPLY_THRESHOLD_PCT)) {
                    product.events.push({ ts: now, type: "supply", delta: effectiveSupply, prevStock, newStock: newTotalStock, merchantDeltas });
                } else {
                    product.events.push({ ts: now, type: "return", delta: effectiveSupply, prevStock, newStock: newTotalStock });
                }
            }

            // ─── Satış / Anomali tespiti ───
            if (effectiveDrop > 0) {
                const hourlyRate = effectiveDrop / elapsedHours;

                const prevSaleEvents = (product.events || []).filter(e => e.type === "sale");
                let isSuspicious = false;
                let emaRate = 0;

                if (prevSaleEvents.length >= 1) {
                    emaRate = prevSaleEvents[0].hourlyRate;
                    for (let si = 1; si < prevSaleEvents.length; si++) {
                        emaRate = prevSaleEvents[si].hourlyRate * STOCK_EMA_ALPHA + emaRate * (1 - STOCK_EMA_ALPHA);
                    }
                    const expectedDrop = emaRate * elapsedHours * STOCK_ANOMALY_MULTIPLIER;
                    const effectiveMax = Math.max(expectedDrop, 3);
                    if (effectiveDrop > effectiveMax && emaRate > 0) {
                        isSuspicious = true;
                    }
                } else {
                    if (prevReading.totalStock > 0) {
                        const dropPct = (effectiveDrop / prevReading.totalStock) * 100;
                        if (dropPct >= STOCK_ANOMALY_MIN_DROP_PCT) {
                            isSuspicious = true;
                        }
                    }
                }

                if (isSuspicious) {
                    product.events.push({
                        ts: now, type: "pending_verification", delta: -effectiveDrop, drop: effectiveDrop,
                        prevStock: prevReading.totalStock, newStock: newTotalStock,
                        elapsedHours: Math.round(elapsedHours * 100) / 100,
                        hourlyRate: Math.round(hourlyRate * 1000) / 1000,
                        emaRateAtDetection: Math.round(emaRate * 1000) / 1000,
                        readingsAtDetection: product.readings.length,
                        verificationsNeeded: STOCK_ANOMALY_CONFIRM_READINGS,
                        merchantDeltas
                    });
                    console.log(`[TK-STOCK] ⏳ Şüpheli düşüş: ${product.id} — düşüş=${effectiveDrop}, EMA=${emaRate.toFixed(2)}/saat → doğrulama bekleniyor`);
                } else {
                    product.events.push({
                        ts: now, type: "sale", delta: -effectiveDrop, sold: effectiveDrop,
                        prevStock: prevReading.totalStock, newStock: newTotalStock,
                        elapsedHours: Math.round(elapsedHours * 100) / 100,
                        hourlyRate: Math.round(hourlyRate * 1000) / 1000,
                        merchantDeltas
                    });
                }
            }
        }

        // ─── Durum güncelle ───
        if (newTotalStock === 0) {
            product.status = "sold_out";
        } else if (product.status === "sold_out") {
            product.status = "active";
        }

    } else {
        // İlk okuma — henüz delta yok
        product.lastStock = newTotalStock;
    }

    // 4) Günlük özet güncelle
    updateDailySummary(product, newTotalStock, newPrice, now);

    // 5) Fiyat değişikliği (günlük özet SONRASINDA ekle — sale event'i gölgelemesin)
    if (product.readings.length >= 2) {
        const prevReading = product.readings[product.readings.length - 2];
        if (newPrice > 0 && prevReading.price > 0 && newPrice !== prevReading.price) {
            product.events.push({ ts: now, type: "price_change", oldPrice: prevReading.price, newPrice });
        }
    }

    // 6) Satış tahminlerini hesapla (asıl motor)
    computeSalesEstimates(product);

    // 7) Güvenilirlik hesapla
    computeConfidence(product);

    // 8) Virtual stok tespiti
    const trackingMode = detectVirtualStock(product.readings);
    if (trackingMode) {
        product.stockTrackingMode = trackingMode;
        if (trackingMode === 'virtual' && product.estimates && product.estimates.dailyEstimate === 0) {
            // Virtual stokta delta 0 ise socialProof ORDER_COUNT fallback
            console.log(`[TK-STOCK] 🔮 Virtual stok: ${product.id} — socialProof fallback`);
        }
    }
}

// ─── Günlük özet güncelleme ───
function updateDailySummary(product, newTotalStock, newPrice, now) {
    const dateKey = new Date(now).toISOString().split("T")[0];
    if (!product.dailySummaries) product.dailySummaries = {};

    if (!product.dailySummaries[dateKey]) {
        product.dailySummaries[dateKey] = {
            totalSold: 0,
            supplyAmount: 0,
            readingCount: 0,
            firstReading: now,
            lastReading: now,
            firstStock: newTotalStock,
            lastStock: newTotalStock,
            minStock: newTotalStock,
            avgStock: newTotalStock,
            priceMin: newPrice,
            priceMax: newPrice,
            hourlyRates: []  // Her ölçümdeki saatlik satış hızı
        };
    }

    const s = product.dailySummaries[dateKey];
    s.readingCount++;
    s.lastReading = now;
    s.lastStock = newTotalStock;
    s.minStock = Math.min(s.minStock, newTotalStock);
    s.avgStock = Math.round((s.avgStock * (s.readingCount - 1) + newTotalStock) / s.readingCount);
    if (newPrice > 0) {
        s.priceMin = Math.min(s.priceMin, newPrice);
        s.priceMax = Math.max(s.priceMax, newPrice);
    }

    // Bu okumada oluşan tüm event'leri günlük özete yansıt
    // NOT: Aynı anda hem sale hem price_change olabilir — hepsini işle
    if (product.events && product.events.length > 0) {
        for (let i = product.events.length - 1; i >= 0; i--) {
            const ev = product.events[i];
            if (ev.ts !== now) break; // Sadece bu okumadaki event'ler
            if (ev.type === "sale") {
                s.totalSold += ev.sold;
                s.hourlyRates.push(ev.hourlyRate);
            } else if (ev.type === "supply") {
                s.supplyAmount += ev.delta;
            }
        }
    }
}

// ═══════════════════════════════════════════════════════════════════
//  SATIŞ TAHMİN MOTORU — Kademeli ilerleme
// ═══════════════════════════════════════════════════════════════════
function computeSalesEstimates(product) {
    if (!product.readings || product.readings.length < 2) return;

    // Takip süresi (saat cinsinden)
    const trackingMs = Date.now() - new Date(product.addedAt).getTime();
    const trackingHours = trackingMs / 3600000;
    const trackingDays = trackingHours / 24;

    // Tahminleri saklayacak obje
    if (!product.estimates) product.estimates = {};

    // ─── MİNİMUM İZLEME SÜRESİ: 3 SAAT ───
    // İlk 3 saat içinde projeksiyon yapmak güvenilir değil
    if (trackingHours < 3) {
        product.estimates.hourlyRate = 0;
        product.estimates.dailyEstimate = 0;
        product.estimates.weeklyEstimate = 0;
        product.estimates.monthlyEstimate = 0;
        product.estimates.totalRealSold = 0;
        product.estimates.source = "insufficient_data";
        product.emaDailySales = 0;
        return;
    }

    // ─── SAATLIK SATIŞ HIZI (tüm aşamalarda temel metrik) ───
    // Son sale event'lerinden kümülatif ağırlıklı ortalama
    const saleEvents = (product.events || []).filter(e => e.type === "sale");
    if (saleEvents.length === 0) {
        // Sale event yok — readings'teki net stok düşüşüne bak (fallback)
        const readings = product.readings;
        const firstReading = readings[0];
        const lastReading = readings[readings.length - 1];
        // Net düşüş: min bazlı hesapla (buybox switch'lerdeki geçici artışları yoksay)
        const minStock = Math.min(...readings.map(r => r.totalStock));
        const netDrop = firstReading.totalStock - minStock;
        const elapsedHours = Math.max((lastReading.ts - firstReading.ts) / 3600000, 0.5);

        if (netDrop > 0 && elapsedHours > 0) {
            const hourlyRate = netDrop / elapsedHours;
            let dailyEst = Math.round(hourlyRate * 24);

            // Üst sınır koruması: kısa takipte aşırı projeksiyonu önle
            if (trackingHours < 6) dailyEst = Math.min(dailyEst, netDrop * 3);
            else if (trackingHours < 12) dailyEst = Math.min(dailyEst, netDrop * 5);
            else if (trackingHours < 24) dailyEst = Math.min(dailyEst, netDrop * 10);
            // totalSold * 10 cap
            dailyEst = Math.min(dailyEst, netDrop * 10);

            product.estimates.hourlyRate = Math.round(hourlyRate * 1000) / 1000;
            product.estimates.dailyEstimate = dailyEst;
            product.estimates.weeklyEstimate = Math.round(dailyEst * 7);
            product.estimates.monthlyEstimate = Math.round(dailyEst * 30);
            product.estimates.totalRealSold = netDrop;
            product.estimates.source = "readings_fallback";
            product.emaDailySales = dailyEst;
        } else {
            product.estimates.hourlyRate = 0;
            product.estimates.dailyEstimate = 0;
            product.estimates.weeklyEstimate = 0;
            product.estimates.monthlyEstimate = 0;
            product.estimates.totalRealSold = 0;
            product.estimates.source = "no_sales";
            product.emaDailySales = 0;
        }
        return;
    }

    // EMA saatlik hız — son event'lere daha fazla ağırlık
    let emaHourly = saleEvents[0].hourlyRate;
    for (let i = 1; i < saleEvents.length; i++) {
        emaHourly = saleEvents[i].hourlyRate * STOCK_EMA_ALPHA + emaHourly * (1 - STOCK_EMA_ALPHA);
    }
    product.estimates.hourlyRate = Math.round(emaHourly * 1000) / 1000;

    // ─── KADEMELİ TAHMİN ───
    const dailySummaryKeys = Object.keys(product.dailySummaries || {}).sort();
    const completeDays = dailySummaryKeys.filter(k => {
        const s = product.dailySummaries[k];
        return s.readingCount >= 3; // En az 3 ölçüm olan günler = "anlamlı" gün
    });

    // Toplam gerçek satış (tüm zamanlar) — üst sınır hesabı için önce hesapla
    let totalRealSold = 0;
    Object.values(product.dailySummaries || {}).forEach(s => { totalRealSold += s.totalSold; });
    product.estimates.totalRealSold = totalRealSold;

    if (completeDays.length >= 7) {
        // ─── AŞAMA 3: 7+ gün verisi → haftalık ortalama (en güvenilir) ───
        const last7 = completeDays.slice(-7);
        let totalSold7 = 0;
        last7.forEach(k => { totalSold7 += product.dailySummaries[k].totalSold; });
        const dailyAvg = totalSold7 / 7;
        product.estimates.dailyEstimate = Math.round(dailyAvg);
        product.estimates.source = "weekly_avg";
    } else if (completeDays.length >= 1) {
        // ─── AŞAMA 2: 1-6 gün verisi → günlük satış ortalaması ───
        let totalSold = 0;
        completeDays.forEach(k => { totalSold += product.dailySummaries[k].totalSold; });
        const dailyAvg = totalSold / completeDays.length;

        // Bugünkü projeksiyonla blend et
        const todayKey = new Date().toISOString().split("T")[0];
        const todayS = product.dailySummaries[todayKey];
        if (todayS && todayS.readingCount >= 2 && !completeDays.includes(todayKey)) {
            const todayElapsedH = (Date.now() - todayS.firstReading) / 3600000;
            const todayProjected = todayElapsedH > 0 ? (todayS.totalSold / todayElapsedH) * 24 : 0;
            product.estimates.dailyEstimate = Math.round(dailyAvg * 0.7 + todayProjected * 0.3);
        } else {
            product.estimates.dailyEstimate = Math.round(dailyAvg);
        }
        product.estimates.source = "daily_avg";
    } else {
        // ─── AŞAMA 1: İlk 24 saat → saatlik hızdan projeksiyon ───
        let dailyEst = Math.round(emaHourly * 24);
        // Üst sınır: totalSold * 10 cap (saatlik projeksiyon abartılı olabilir)
        if (totalRealSold > 0) dailyEst = Math.min(dailyEst, totalRealSold * 10);
        // Kısa takip üst sınırları
        if (trackingHours < 6) dailyEst = Math.min(dailyEst, totalRealSold * 3);
        else if (trackingHours < 12) dailyEst = Math.min(dailyEst, totalRealSold * 5);
        else if (trackingHours < 24) dailyEst = Math.min(dailyEst, totalRealSold * 10);
        product.estimates.dailyEstimate = dailyEst;
        product.estimates.source = "hourly_projection";
    }

    // Haftalık ve aylık tahmin
    product.estimates.weeklyEstimate = Math.round(product.estimates.dailyEstimate * 7);
    product.estimates.monthlyEstimate = Math.round(product.estimates.dailyEstimate * 30);

    // emaDailySales güncelle (geriye uyumluluk + T0 tier)
    product.emaDailySales = product.estimates.dailyEstimate;
}

// ─── Güvenilirlik — ölçüm sayısına + süreye göre ───
function computeConfidence(product) {
    const readings = (product.readings || []).length;
    const saleEvents = (product.events || []).filter(e => e.type === "sale").length;
    const trackingMs = Date.now() - new Date(product.addedAt).getTime();
    const trackingHours = trackingMs / 3600000;
    const daysTracked = Math.floor(trackingHours / 24);
    const completeDays = Object.keys(product.dailySummaries || {}).filter(k => {
        const s = product.dailySummaries[k];
        return s && s.readingCount >= 3;
    }).length;

    // Puanlama: ölçüm sayısı + satış event sayısı + tam gün sayısı
    let score = 0;
    score += Math.min(readings * 2, 20);         // max 20 puan (10+ ölçüm)
    score += Math.min(saleEvents * 5, 30);        // max 30 puan (6+ satış event)
    score += Math.min(completeDays * 7, 50);      // max 50 puan (7+ tam gün)

    // score: 0-100 → level: 0.05-0.95
    const level = Math.min(0.95, Math.max(0.05, score / 100));

    let label;
    if (level < 0.15) label = "Başlangıç";
    else if (level < 0.30) label = "Düşük";
    else if (level < 0.50) label = "Gelişiyor";
    else if (level < 0.70) label = "Orta";
    else if (level < 0.85) label = "Yüksek";
    else label = "Çok Yüksek";

    const source = product.estimates ? product.estimates.source : "none";

    product.confidence = { level, label, daysTracked, readings, saleEvents, completeDays, source };
}

// ─── Eski okumaları temizle (30 günden eski) ───
function pruneOldReadings(product) {
    if (!product.readings) return;
    const cutoff = Date.now() - (STOCK_PRUNE_DAYS * 86400000);
    const before = product.readings.length;
    product.readings = product.readings.filter(r => r.ts > cutoff);
    if (product.readings.length < before) {
        console.log(`[TK-STOCK] ${product.id}: ${before - product.readings.length} eski okuma silindi`);
    }
    // Events da temizle (60 günden eski)
    if (product.events) {
        const eventCutoff = Date.now() - (60 * 86400000);
        product.events = product.events.filter(e => e.ts > eventCutoff);
    }
}

// ─── Mevcut ürün verisini onar ───
// Readings varsa ama sale event / dailySummary eksikse geriye dönük oluştur
// Anomali tespiti ile satıcı stok düzeltmelerini filtreler
function repairProductData(product) {
    if (!product.readings || product.readings.length < 2) return false;

    const existingSaleEvents = (product.events || []).filter(e => e.type === "sale");
    let repaired = false;

    // Sale event eksikse — readings'lerden geriye dönük oluştur (anomali tespiti ile)
    if (existingSaleEvents.length === 0) {
        if (!product.events) product.events = [];
        if (!product.dailySummaries) product.dailySummaries = {};

        // Repair sırasında oluşan sale event'lerden EMA hesaplamak için geçici list
        const repairSaleRates = [];

        for (let i = 1; i < product.readings.length; i++) {
            const prev = product.readings[i - 1];
            const curr = product.readings[i];
            const elapsedMs = curr.ts - prev.ts;
            const elapsedHours = Math.max(elapsedMs / 3600000, 0.1);

            // ─── Satıcı bazlı delta hesapla (processStockReading ile aynı mantık) ───
            const prevMerchants = prev.merchants || {};
            const currMerchants = curr.merchants || {};
            const allMids = new Set([
                ...Object.keys(prevMerchants).filter(k => k !== '_buybox'),
                ...Object.keys(currMerchants).filter(k => k !== '_buybox')
            ]);
            const hasMerchantData = allMids.size > 0 && (Object.keys(prevMerchants).filter(k => k !== '_buybox').length > 0);

            let effectiveDrop = 0;
            let effectiveSupply = 0;

            if (hasMerchantData) {
                for (const mid of allMids) {
                    const mDelta = (Number(currMerchants[mid]) || 0) - (Number(prevMerchants[mid]) || 0);
                    if (mDelta < 0) effectiveDrop += Math.abs(mDelta);
                    else if (mDelta > 0) effectiveSupply += mDelta;
                }
            } else {
                // Eski format — merchant verisi yok, totalStock'tan hesapla
                const simpleDelta = curr.totalStock - prev.totalStock;
                effectiveDrop = simpleDelta < 0 ? Math.abs(simpleDelta) : 0;
                effectiveSupply = simpleDelta > 0 ? simpleDelta : 0;
            }

            if (effectiveSupply > 0) {
                product.events.push({ ts: curr.ts, type: "supply", delta: effectiveSupply, prevStock: prev.totalStock, newStock: curr.totalStock });
            }

            if (effectiveDrop > 0) {
                const hourlyRate = effectiveDrop / elapsedHours;

                // Anomali tespiti (dinamik — ürün bazlı + doğrulama penceresi)
                let isSuspicious = false;
                let emaRate = 0;
                if (repairSaleRates.length >= 1) {
                    emaRate = repairSaleRates[0];
                    for (let si = 1; si < repairSaleRates.length; si++) {
                        emaRate = repairSaleRates[si] * STOCK_EMA_ALPHA + emaRate * (1 - STOCK_EMA_ALPHA);
                    }
                    const expectedMax = Math.max(emaRate * elapsedHours * STOCK_ANOMALY_MULTIPLIER, 3);
                    if (effectiveDrop > expectedMax && emaRate > 0) isSuspicious = true;
                } else {
                    const dropPct = prev.totalStock > 0 ? (effectiveDrop / prev.totalStock) * 100 : 100;
                    if (dropPct >= STOCK_ANOMALY_MIN_DROP_PCT) isSuspicious = true;
                }

                let isAnomaly = false;
                if (isSuspicious) {
                    // Geriye dönük doğrulama: sonraki ölçümlere bak
                    const confirmEnd = Math.min(i + STOCK_ANOMALY_CONFIRM_READINGS, product.readings.length - 1);
                    if (confirmEnd > i) {
                        let totalDropAfter = 0;
                        let totalHoursAfter = 0;
                        let dropSegments = 0;
                        for (let r = i; r < confirmEnd; r++) {
                            const rCurr = product.readings[r + 1];
                            const rPrev = product.readings[r];
                            // Doğrulamada da satıcı bazlı hesapla
                            const rpM = rPrev.merchants || {};
                            const rcM = rCurr.merchants || {};
                            const rMids = new Set([
                                ...Object.keys(rpM).filter(k => k !== '_buybox'),
                                ...Object.keys(rcM).filter(k => k !== '_buybox')
                            ]);
                            let segDrop = 0;
                            if (rMids.size > 0 && Object.keys(rpM).filter(k => k !== '_buybox').length > 0) {
                                for (const mid of rMids) {
                                    const md = (Number(rcM[mid]) || 0) - (Number(rpM[mid]) || 0);
                                    if (md < 0) segDrop += Math.abs(md);
                                }
                            } else {
                                const sd = rCurr.totalStock - rPrev.totalStock;
                                segDrop = sd < 0 ? Math.abs(sd) : 0;
                            }
                            const segH = Math.max((rCurr.ts - rPrev.ts) / 3600000, 0.1);
                            totalHoursAfter += segH;
                            if (segDrop > 0) {
                                totalDropAfter += segDrop;
                                dropSegments++;
                            }
                        }
                        const afterRate = totalHoursAfter > 0 ? totalDropAfter / totalHoursAfter : 0;
                        const refRate = emaRate > 0 ? emaRate : hourlyRate * 0.2;
                        if (dropSegments >= 1 && afterRate >= refRate * 0.5) {
                            isAnomaly = false;
                        } else {
                            isAnomaly = true;
                        }
                    } else {
                        isAnomaly = true;
                    }
                }

                if (isAnomaly) {
                    product.events.push({
                        ts: curr.ts, type: "stock_adjustment", delta: -effectiveDrop, drop: effectiveDrop,
                        prevStock: prev.totalStock, newStock: curr.totalStock,
                        elapsedHours: Math.round(elapsedHours * 100) / 100,
                        reason: "anomaly_detected_repair"
                    });
                } else {
                    product.events.push({
                        ts: curr.ts, type: "sale", delta: -effectiveDrop, sold: effectiveDrop,
                        prevStock: prev.totalStock, newStock: curr.totalStock,
                        elapsedHours: Math.round(elapsedHours * 100) / 100,
                        hourlyRate: Math.round(hourlyRate * 1000) / 1000
                    });
                    repairSaleRates.push(hourlyRate);

                    // dailySummary güncelle
                    const dateKey = new Date(curr.ts).toISOString().split("T")[0];
                    if (!product.dailySummaries[dateKey]) {
                        product.dailySummaries[dateKey] = {
                            totalSold: 0, supplyAmount: 0, readingCount: 0,
                            firstReading: curr.ts, lastReading: curr.ts,
                            firstStock: curr.totalStock, lastStock: curr.totalStock,
                            minStock: curr.totalStock, avgStock: curr.totalStock,
                            priceMin: curr.price || 0, priceMax: curr.price || 0,
                            hourlyRates: []
                        };
                    }
                    product.dailySummaries[dateKey].totalSold += effectiveDrop;
                    product.dailySummaries[dateKey].hourlyRates.push(hourlyRate);
                }
                repaired = true;
            }
        }

        if (repaired) {
            product.events.sort((a, b) => a.ts - b.ts);
            computeSalesEstimates(product);
            computeConfidence(product);
            const saleCount = product.events.filter(e => e.type === "sale").length;
            const adjCount = product.events.filter(e => e.type === "stock_adjustment").length;
            console.log(`[TK-REPAIR] ${product.id}: ${saleCount} satış + ${adjCount} stok ayarlaması`);
        }
    }

    // dailySummary readingCount kontrolü — readings'ten doğru sayıyı al
    if (!product.dailySummaries) product.dailySummaries = {};
    for (const r of product.readings) {
        const dk = new Date(r.ts).toISOString().split("T")[0];
        if (!product.dailySummaries[dk]) {
            product.dailySummaries[dk] = {
                totalSold: 0, supplyAmount: 0, readingCount: 0,
                firstReading: r.ts, lastReading: r.ts,
                firstStock: r.totalStock, lastStock: r.totalStock,
                minStock: r.totalStock, avgStock: r.totalStock,
                priceMin: r.price || 0, priceMax: r.price || 0,
                hourlyRates: []
            };
        }
        product.dailySummaries[dk].readingCount++;
        product.dailySummaries[dk].lastReading = r.ts;
        product.dailySummaries[dk].lastStock = r.totalStock;
    }

    return repaired;
}
