// Trendyol Karlılık Hesaplayıcı - Content Script

(function () {
  "use strict";

  // Debug modu (console logları açıp kapar — production'da false)
  const TK_DEBUG = false;
  function tkLog(...args) { if (TK_DEBUG) console.log("[TK]", ...args); }

  // Ayarlar (chrome.storage'dan yüklenir)
  let settings = {
    carrier: "TEX-PTT",
    salesMultiplier: 5,
    kdvRate: 20,
    panelVisible: true
  };

  // ─── Kalibrasyon verileri (panel sync'den gelen) ───
  let panelCalibration = { basket_to_sales_rate: 0.10 };

  // ─── Shipping config uygula (panelden gelen kargo fiyatları + desi defaults) ───
  function _applyShippingConfig(config) {
    if (!config) return;
    // Kargo oranlarını güncelle (data/shipping.js'deki global fonksiyon)
    if (config.shipping && typeof updateShippingRates === 'function') {
      updateShippingRates(config.shipping.barem, config.shipping.weight);
    }
    // Kategori desi varsayılanlarını güncelle
    if (config.desiDefaults && typeof setDesiDefaults === 'function') {
      setDesiDefaults(config.desiDefaults);
    }
    tkLog("Shipping config uygulandı:", config.shipping?.lastUpdated || "bilinmiyor");
  }

  // ─── Stok Takip: Takip edilen ürün ID'leri (Set) ───
  let trackedProductIds = new Set();   // Yerel (extension) takip listesi
  let panelTrackedIds = new Set();     // Panel takip listesi

  // Birlesiik takip kontrolu — yerel VEYA panel'de takipte mi?
  function isProductTracked(pid) {
    return trackedProductIds.has(String(pid)) || panelTrackedIds.has(String(pid));
  }

  async function loadTrackedProductIds() {
    // 1) Yerel takip listesini yukle
    const localPromise = new Promise(resolve => {
      if (!chrome.runtime || !chrome.runtime.sendMessage) { resolve(); return; }
      chrome.runtime.sendMessage({ action: "GET_TRACKED_PRODUCTS" }, (resp) => {
        if (chrome.runtime.lastError) { resolve(); return; }
        if (resp && resp.products) {
          if (Array.isArray(resp.products)) {
            trackedProductIds = new Set(resp.products.map(p => String(p.id)));
          } else {
            trackedProductIds = new Set(Object.keys(resp.products));
          }
        }
        resolve();
      });
    });

    // 2) Panel takip listesini yukle
    const panelPromise = new Promise(resolve => {
      if (!chrome.runtime || !chrome.runtime.sendMessage) { resolve(); return; }
      chrome.runtime.sendMessage({ action: "GET_PANEL_TRACKED_IDS" }, (resp) => {
        if (chrome.runtime.lastError) { resolve(); return; }
        if (resp && resp.ids && Array.isArray(resp.ids)) {
          panelTrackedIds = new Set(resp.ids.map(id => String(id)));
        }
        resolve();
      });
    });

    await Promise.all([localPromise, panelPromise]);
  }

  // Ayarları yükle
  function loadSettings() {
    return new Promise(resolve => {
      if (chrome.storage && chrome.storage.sync) {
        chrome.storage.sync.get(settings, stored => {
          settings = { ...settings, ...stored };
          resolve();
        });
      } else {
        resolve();
      }
    });
  }

  // Sayfa tipini belirle
  function getPageType() {
    const url = window.location.href;
    if (/-p-\d+/.test(url)) return "product";
    if (/\/sr\?/.test(url) || /-c\d+/.test(url) || /-x-/.test(url)) return "listing";
    return "other";
  }

  // ═══════════════════════════════════════════════════════════════════
  //  PAGE DATA READER
  //  inject.js (MAIN world) sayfa JS değişkenlerini okuyup #tk-page-data
  //  DOM elementine yazar. Bu fonksiyon o veriyi content script'ten okur.
  //  inject.js manifest'te world:"MAIN" olarak tanımlıdır → CSP sorunu yok.
  // ═══════════════════════════════════════════════════════════════════
  function readPageData() {
    return new Promise((resolve) => {
      let attempts = 0;
      const maxAttempts = 50; // 50 × 100ms = 5 saniye
      const currentURL = window.location.href;

      function isValidData(data) {
        if (!data) return false;
        // URL eşleşmesi kontrol — stale data kullanma
        if (data._url && data._url !== currentURL) return false;
        // En azından bir veri kaynağı olmalı
        return !!(data.detailProduct || data.listingProducts);
      }

      function tryRead() {
        const dataEl = document.getElementById("tk-page-data");
        if (dataEl && dataEl.textContent && dataEl.textContent.length > 5) {
          try {
            const data = JSON.parse(dataEl.textContent);
            if (isValidData(data)) {
              tkLog("Page data okundu:", data.detailProduct ? "detay" : "listing", JSON.stringify(data).length, "byte");
              resolve(data);
              return;
            }
          } catch (e) {
            // parse hatası, tekrar dene
          }
        }
        attempts++;
        if (attempts < maxAttempts) {
          setTimeout(tryRead, 100);
        } else {
          tkLog("Page data bulunamadı (timeout) — null ile devam");
          // inject.js'ye yeniden okuma isteği gönder
          document.dispatchEvent(new CustomEvent("tk-page-data-refresh"));
          resolve(null);
        }
      }

      // Custom event dinle — inject.js veriyi yazdığında tetiklenir
      const onReady = () => {
        document.removeEventListener("tk-page-data-ready", onReady);
        // Kısa gecikme ile oku (DOM güncellenmesini garantile)
        setTimeout(tryRead, 10);
      };
      document.addEventListener("tk-page-data-ready", onReady);

      // Aynı zamanda polling de yap (event kaçırılırsa diye)
      tryRead();
    });
  }

  // ===== ÜRÜN DETAY SAYFASI =====

  function extractProductData(pageData) {
    const data = {};

    // Fiyat (2025+ yeni yapı dahil)
    const detailPriceSelectors = [
      ".price-value",                // Yaprak fiyat elementi (2025+, en spesifik)
      ".discounted-price",           // İndirimli fiyat (2025+)
      ".sale-price",                 // Satış fiyatı
      ".price-section",              // Ana fiyat (2025+)
      ".single-price",               // Tekil fiyat (2025+)
      ".price-container .discounted", // Eski indirimli
      ".prc-dsc",                    // Eski indirimli kısa
      ".prc-box-dscntd",             // Eski indirimli kutu
      ".prc-box-sllng",              // Eski satış fiyatı kutu
      ".product-price-container span"
    ];
    let priceEl = null;
    for (const sel of detailPriceSelectors) {
      const el = document.querySelector(sel);
      if (el) {
        const parsed = TrendyolCalculator.parsePrice(el.textContent);
        if (parsed > 0) { priceEl = el; break; }
      }
    }
    data.price = priceEl ? TrendyolCalculator.parsePrice(priceEl.textContent) : 0;

    // ═══════════════════════════════════════════════════════════════
    //  Çok Satıcılı Sayfa Desteği
    //  Bazı ürünlerde standart fiyat elementleri yok, bunun yerine
    //  "Ürünün Diğer Satıcıları" listesi gösterilir.
    //  URL'deki merchantId ile eşleşen satıcıyı bul, yoksa ilk satıcıyı al.
    // ═══════════════════════════════════════════════════════════════
    if (!data.price) {
      try {
        const urlMerchantId = new URLSearchParams(window.location.search).get("merchantId");
        const sellerCards = document.querySelectorAll('[data-testid="other-seller-item"]');
        let matchedPrice = 0;
        let firstPrice = 0;
        let matchedMerchant = "";
        let firstMerchant = "";

        sellerCards.forEach((card) => {
          const priceEl = card.querySelector(".price-current-price");
          const midLink = card.querySelector('a[href*="/sr?mid="]');
          const nameEl = card.querySelector('[class*="merchant-name"]');
          const mid = midLink ? (midLink.href.match(/mid=(\d+)/) || [])[1] : null;
          const cardPrice = priceEl ? TrendyolCalculator.parsePrice(priceEl.textContent) : 0;
          const merchantName = nameEl ? nameEl.textContent.trim() : "";

          if (cardPrice > 0 && firstPrice === 0) {
            firstPrice = cardPrice;
            firstMerchant = merchantName;
          }
          if (urlMerchantId && mid === urlMerchantId && cardPrice > 0) {
            matchedPrice = cardPrice;
            matchedMerchant = merchantName;
          }
        });

        // merchantId eşleşen satıcı yoksa ilk satıcıyı kullan
        data.price = matchedPrice > 0 ? matchedPrice : firstPrice;
        data.merchant = matchedMerchant || firstMerchant || "";
        if (data.price > 0) {
          tkLog(`Çok satıcılı sayfa — Fiyat: ${data.price} TL, Satıcı: ${data.merchant}`);
        }
      } catch (e) {
        tkLog("Çok satıcılı fiyat okunamadı:", e.message);
      }
    }

    // Son fallback: Hidden data'dan fiyat (inject.js → sellingPrice/discountedPrice)
    if (!data.price && pageData && pageData.detailProduct) {
      const dp = pageData.detailProduct;
      const hiddenPrice = dp.discountedPrice || dp.sellingPrice || 0;
      if (hiddenPrice > 0) {
        data.price = hiddenPrice;
        tkLog(`Hidden data'dan fiyat alındı: ${hiddenPrice} TL`);
      }
    }

    // Başlık
    const titleEl = document.querySelector("h1.product-title") ||
      document.querySelector("h1[class*='product']") ||
      document.querySelector(".pr-new-br") ||
      document.querySelector("h1"); // Genel fallback (çok satıcılı sayfalar dahil)
    data.title = titleEl ? titleEl.textContent.trim() : "";

    // Breadcrumb / Kategori (birden fazla selector dene)
    const breadcrumbSelectors = [
      ".product-detail-breadcrumbs-item a",
      ".breadcrumb-wrapper a",
      "[class*='breadcrumb'] a",
      "nav[aria-label*='breadcrumb'] a",
      "nav[aria-label*='Breadcrumb'] a",
      "[itemtype*='BreadcrumbList'] a",
      ".product-detail-breadcrumb a",
      "[class*='Breadcrumb'] a",
    ];

    let bcLinks = [];
    for (const sel of breadcrumbSelectors) {
      const found = document.querySelectorAll(sel);
      if (found.length > 0) {
        bcLinks = Array.from(found);
        break;
      }
    }

    data.breadcrumbs = bcLinks
      .map(a => a.textContent.trim())
      .filter(t => t && t !== "Trendyol" && t !== "Ana Sayfa" && !t.includes(data.title) && t.length > 1);

    // Breadcrumb bulunamazsa, URL'den ve sayfa meta bilgisinden çıkarmayı dene
    if (data.breadcrumbs.length === 0) {
      // Schema.org veya meta tag'dan kategori bilgisi
      const metaCategory = document.querySelector('meta[property="product:category"]');
      if (metaCategory && metaCategory.content) {
        data.breadcrumbs = metaCategory.content.split(/[>/]/).map(s => s.trim()).filter(Boolean);
      }
    }

    if (data.breadcrumbs.length === 0) {
      // URL path'inden çıkarmayı dene
      const urlPath = window.location.pathname;
      const pathParts = urlPath.split("/").filter(p => p && !/-p-\d+/.test(p));
      data.breadcrumbs = pathParts.map(p =>
        p.replace(/-x-.*$/, "")
          .replace(/-c\d+$/, "")
          .replace(/-/g, " ")
          .replace(/\b\w/g, l => l.toUpperCase())
      ).filter(t => t.length > 1);
    }

    // Değerlendirme sayısı
    const reviewEl = document.querySelector("a.reviews-summary-reviews-detail") ||
      document.querySelector("[class*='reviews-summary'] a");
    if (reviewEl) {
      const match = reviewEl.textContent.match(/(\d+)/);
      data.reviewCount = match ? parseInt(match[1]) : 0;
    } else {
      data.reviewCount = 0;
    }

    // Rating
    const ratingEl = document.querySelector("span.reviews-summary-average-rating") ||
      document.querySelector("[class*='reviews-summary'] span");
    data.rating = ratingEl ? parseFloat(ratingEl.textContent) : 0;

    // Marka (detay sayfası)
    const brandEl = document.querySelector(".pr-new-br span, .pr-new-br a, .product-brand-name-with-link a, [class*='brand-name'] a, [data-testid='brand-name-wrapper'] span");
    data.brand = brandEl ? brandEl.textContent.trim() : "";

    // Satıcı
    const merchantEl = document.querySelector("[class*='merchant-name']");
    data.merchant = merchantEl ? merchantEl.textContent.trim() : "";

    // Sosyal kanıt (sepet sayısı) — DOM'dan
    const socialEl = document.querySelector(".social-proof-item-focused-text");
    if (socialEl) {
      const match = socialEl.textContent.match(/(\d[\d.]*)/);
      data.socialProof = match ? match[1].replace(/\./g, "") : "";
    }

    // ═══════════════════════════════════════════════════════════════
    //  Trendyol Hidden Data — Detay Sayfası
    // ═══════════════════════════════════════════════════════════════
    //  pageData.detailProduct (inject edilen script'ten gelir):
    //    ratingScore.totalCount (tam sayı)
    //    favoriteCount (tam sayı)
    //    categoryHierarchy (güvenilir breadcrumb)
    //    stockQuantity (stok)
    // ═══════════════════════════════════════════════════════════════
    data.socialProofData = null;
    try {
      const product = pageData && pageData.detailProduct;
      if (product) {
        tkLog("Detay sayfası hidden data bulundu (injected):", product.id);

        // ratingScore'dan review/rating bilgisi al
        // NOT: DOM'daki değer daha güncel olabilir (SSR props cache'li kalır)
        // Bu yüzden sadece DOM değeri yoksa veya SSR daha yüksekse SSR'ı kullan
        if (product.ratingScore) {
          if (product.ratingScore.totalCount > 0) {
            const ssrCount = product.ratingScore.totalCount;
            if (!data.reviewCount || ssrCount > data.reviewCount) {
              data.reviewCount = ssrCount;
              tkLog(`  reviewCount güncellendi: ${data.reviewCount} (hidden data)`);
            } else {
              tkLog(`  reviewCount DOM'dan korundu: ${data.reviewCount} (SSR: ${ssrCount})`);
            }
          }
          if (product.ratingScore.averageRating > 0) {
            data.rating = product.ratingScore.averageRating;
          }
        }

        // favoriteCount
        const favCount = product.favoriteCount || 0;

        // category.hierarchy → güvenilir breadcrumb
        if (product.categoryHierarchy) {
          const hierarchyBreadcrumbs = product.categoryHierarchy
            .split("/")
            .map(s => s.trim())
            .filter(s => s && s !== "Trendyol");
          if (hierarchyBreadcrumbs.length > 0) {
            data.breadcrumbs = hierarchyBreadcrumbs;
            tkLog(`  Breadcrumb güncellendi (hierarchy): ${hierarchyBreadcrumbs.join(" > ")}`);
          }
        }

        // socialProofData oluştur (basketCount ve orderCount DOM'dan çekilecek)
        data.socialProofData = {
          favoriteCount: favCount > 1000 ? Math.round(favCount / 1000) + "K" : String(favCount)
        };

        // Stok bilgisi
        if (product.stockQuantity > 0) {
          data.stockQuantity = product.stockQuantity;
          tkLog(`  Stok: ${data.stockQuantity} adet`);
        }

        // categoryTopRankings → bestSeller badge
        if (product.categoryTopRankings && Array.isArray(product.categoryTopRankings)) {
          product.categoryTopRankings.forEach(r => {
            if (r.name === "bestSeller") {
              data.badges = data.badges || [];
              data.badges.push("En Çok Satan #" + (r.order || ""));
              tkLog(`  bestSeller badge: #${r.order}`);
            }
          });
        }

        // ─── Gerçek KDV Oranı (Trendyol backend) ───
        if (product.taxRate !== undefined && product.taxRate !== null) {
          data.taxRate = product.taxRate / 100; // 10 → 0.10, 20 → 0.20, 1 → 0.01
          tkLog(`  KDV oranı (gerçek): %${product.taxRate}`);
        }

        // ─── Kategori Ağacı (ID + isim) ───
        if (product.categoryTree && product.categoryTree.length > 0) {
          data.categoryTree = product.categoryTree;
          tkLog(`  Kategori ağacı: ${product.categoryTree.map(c => c.name).join(" > ")}`);
        }

        // ─── Satıcı ID (buybox kazanan) ───
        if (product.seller && product.seller.id) {
          data.merchantId = String(product.seller.id);
          data.merchantName = product.seller.name || '';
          tkLog(`  Satıcı ID: ${data.merchantId} (${product.seller.name})`);
        }

        // ─── Diğer Satıcılar (merchantListing.otherMerchants) ───
        if (product.otherMerchants && product.otherMerchants.length > 0) {
          data.otherMerchantIds = product.otherMerchants.map(om => String(om.id)).filter(Boolean);
          data.otherMerchantNames = {};
          product.otherMerchants.forEach(om => {
            if (om.id && om.name) data.otherMerchantNames[String(om.id)] = om.name;
          });
          tkLog(`  Diğer satıcılar: ${data.otherMerchantIds.length} adet`);
        }

        // ─── Kategori Yolu (categoryPath) ───
        if (product.categoryHierarchy) {
          data.categoryPath = product.categoryHierarchy;
        }

        // ─── Favori sayısı (raw — panel DB için) ───
        if (favCount > 0) {
          data.favoriteCount = favCount;
        }

        // ─── Plus Fiyatı ve Orijinal Fiyat ───
        // plusPrice = tyPlusCouponApplicablePrice (inject.js'den)
        // isTyPlusEligible = false ise plusPrice == discountedPrice (indirim yok)
        if (product.plusPrice > 0 && product.isTyPlusEligible && product.plusPrice < (data.price || Infinity)) {
          data.plusPrice = product.plusPrice;
          tkLog(`  Plus fiyatı: ${data.plusPrice} TL (eligible)`);
        }
        if (product.originalPrice > 0) {
          data.originalPrice = product.originalPrice;
          tkLog(`  Orijinal fiyat: ${data.originalPrice} TL`);
        }

      }
    } catch (e) {
      tkLog("Detay sayfası hidden data okunamadı:", e.message);
    }

    // DOM'dan socialProof text'leri — sepet sayısı ve satış verisi
    // NOT: "satıldı" verisi T1 (orderCount) DEĞİLDİR!
    //   Detay sayfasında gerçek orderCount yok. "3 günde 500+ satıldı"
    //   viral pencere verisidir → recentSalesData (T2) olarak işlenir.
    //   Gerçek T1 orderCount sadece listing sayfasında __single-search-result__PROPS'ta bulunur.
    data.recentSalesData = null;
    try {
      const spWrapper = document.querySelector("[data-testid='social-proof'], .social-proof-social-proof-wrapper, .social-proof-social-proof, [class*='social-proof-wrapper']");
      if (spWrapper) {
        // socialProofData yoksa oluştur (inject.js verisi gelmemişse null olabilir)
        if (!data.socialProofData) data.socialProofData = {};

        // Hem class hem data-testid ile item'ları bul (Trendyol class isimlerini değiştiriyor)
        const spItems = spWrapper.querySelectorAll("[data-testid='social-proof-item'], [class*='social-proof-item']");
        tkLog(`Social proof items bulundu: ${spItems.length} adet`);
        spItems.forEach(item => {
          const text = item.textContent.trim();
          // "3,3B kişinin sepetinde" → basketCount (socialProofData)
          if (text.includes("sepetinde")) {
            const focusEl = item.querySelector(".social-proof-item-focused-text, [class*='focused']");
            if (focusEl) {
              // focusEl text: "3,5B kişinin" veya "3,5B" veya "4K kişinin" gibi olabilir
              let bText = focusEl.textContent.trim().replace(/\s+kişinin.*$/, "").replace(/\s+sepetinde.*$/, "");
              // Hala sayı + suffix bırak: "3,5B", "4K", "500"
              data.socialProofData.basketCount = bText;
              tkLog(`  basketCount DOM: "${focusEl.textContent.trim()}" → "${bText}"`);
            }
            // Focused element bulunamadıysa text'ten direkt parse et
            if (!data.socialProofData.basketCount) {
              const bMatch = text.match(/([\d.,]+\s*[BKMbkm]?)\s*(?:\+?\s*)?kişinin/);
              if (bMatch) {
                data.socialProofData.basketCount = bMatch[1].trim();
                tkLog(`  basketCount text fallback: "${bMatch[1].trim()}"`);
              }
            }
          }
          // "3 günde 500+ ürün satıldı" → recentSalesData (T2 viral pencere)
          if (text.includes("satıldı") && !data.recentSalesData) {
            const fullText = text;
            const daysMatch = fullText.match(/(\d+)\s+(gün|saat|dakika)/);
            let salesCount = 0;
            let days = 7;

            // Önce focused element'ten oku
            const focusEl = item.querySelector(".social-proof-item-focused-text, [class*='focused']");
            if (focusEl) {
              const focusText = focusEl.textContent.trim();
              // BUG FIX: focusText "3 günde 5B+" olabilir — ilk sayı gün sayısı!
              // Önce suffix'li sayıyı ara (5B+, 3,5K+, 1,2M+ gibi) — doğru sırayı yakalamak için
              const suffixMatch = focusText.match(/([\d.,]+)\s*([BKMbkm])\+?/);
              if (suffixMatch) {
                const numStr = suffixMatch[1];
                const suffix = suffixMatch[2].toUpperCase();
                if (suffix === "B" || suffix === "K") salesCount = parseFloat(numStr.replace(",", ".")) * 1000;
                else if (suffix === "M") salesCount = parseFloat(numStr.replace(",", ".")) * 1000000;
              } else {
                // Suffix yoksa — düz sayı (500+, 1.000+ gibi)
                const salesMatch = focusText.match(/(\d[\d.,]*)\+?\s*/);
                if (salesMatch) {
                  salesCount = parseFloat(salesMatch[1].replace(/\./g, "").replace(",", "."));
                }
              }
            }

            // Focused element bulunamadıysa veya parse edemediyse → full text'ten parse et
            if (salesCount === 0) {
              // "3 günde 500+ ürün satıldı" veya "3 günde 1,5B+ ürün satıldı"
              const textSalesMatch = fullText.match(/([\d.,]+)\s*([BKMbkm])?\+?\s*ürün/);
              if (textSalesMatch) {
                const numStr = textSalesMatch[1];
                const suffix = (textSalesMatch[2] || "").toUpperCase();
                if (suffix === "B") salesCount = parseFloat(numStr.replace(",", ".")) * 1000;
                else if (suffix === "K") salesCount = parseFloat(numStr.replace(",", ".")) * 1000;
                else if (suffix === "M") salesCount = parseFloat(numStr.replace(",", ".")) * 1000000;
                else salesCount = parseFloat(numStr.replace(/\./g, "").replace(",", "."));
                tkLog(`  satıldı text fallback parse: "${textSalesMatch[0]}" → ${salesCount}`);
              }
            }

            if (daysMatch) {
              days = parseInt(daysMatch[1]);
              if (daysMatch[2].includes("saat")) days = 1;
              if (daysMatch[2].includes("dakika")) days = 1;
            }

            if (salesCount > 0) {
              data.recentSalesData = { count: salesCount, days: days };
              tkLog(`Viral Satış (DOM): ${days} günde ${salesCount} adet`);
            }
          }
        });
        tkLog(`  socialProofData: basketCount=${data.socialProofData?.basketCount || "-"}, recentSales=${data.recentSalesData ? data.recentSalesData.count + "/" + data.recentSalesData.days + "gün" : "-"}`);
      }
    } catch (e) {
      tkLog("DOM socialProof okunamadı:", e.message);
    }

    // Eski selector'larla da recentSalesData yakalamaya çalış (fallback)
    if (!data.recentSalesData) {
      const socialProofEl = document.querySelector(".social-proof-text") ||
        document.querySelector(".pr-r-text");
      if (socialProofEl) {
        const text = socialProofEl.textContent;
        const daysMatch = text.match(/(\d+)\s+(gün|saat|dakika)/);
        const salesMatch = text.match(/(\d+[.,]?\d*)\s*B?\+?\s+ürün/);
        if (daysMatch && salesMatch) {
          let days = parseInt(daysMatch[1]);
          if (daysMatch[2].includes("saat")) days = 1;
          if (daysMatch[2].includes("dakika")) days = 1;
          let salesCount = 0;
          let salesStr = salesMatch[1]; // Grup 1: "1,5" veya "500"
          const fullMatchStr = salesMatch[0]; // Tam eşleşme: "1,5B+ ürün" veya "500+ ürün"
          if (fullMatchStr.includes("B")) {
            // "1,5B" → 1.5 * 1000 = 1500
            salesCount = parseFloat(salesStr.replace(",", ".")) * 1000;
          } else {
            // "500" → 500 (binlik noktayı kaldır, virgülü ondalığa çevir)
            salesCount = parseFloat(salesStr.replace(/\./g, "").replace(",", "."));
          }
          data.recentSalesData = { count: salesCount, days: days };
          tkLog(`Viral Satış (eski selector): ${days} günde ${salesCount} adet`);
        }
      }
    }

    // Ürün görseli — og:image meta tag (en güvenilir kaynak)
    try {
      const ogImage = document.querySelector('meta[property="og:image"]');
      if (ogImage && ogImage.content) {
        data.imageUrl = ogImage.content;
      }
      // Fallback: gallery'deki ilk ürün görseli
      if (!data.imageUrl) {
        const galleryImg = document.querySelector('.gallery-container img, .product-slide img, [class*="product-image"] img, .base-product-image img');
        if (galleryImg && galleryImg.src && !galleryImg.src.includes('placeholder')) {
          data.imageUrl = galleryImg.src;
        }
      }
    } catch (e) {
      // silent
    }

    // Rozetler (Flash, En Çok Satan)
    data.badges = [];
    document.querySelectorAll(".stamps .promotion-stamp, .campaign-name, .stamp-container, .active-campaign-wrapper").forEach(el => {
      data.badges.push(el.textContent.trim());
    });

    // Diğer satıcı ID'leri (DOM'dan)
    try {
      const sellerLinks = document.querySelectorAll("[class*='other-merchant'] a[href*='merchantId='], [class*='other-sellers'] a[href*='merchantId=']");
      const merchantIds = [];
      sellerLinks.forEach(a => {
        const match = a.href.match(/merchantId=(\d+)/);
        if (match) merchantIds.push(parseInt(match[1]));
      });
      data.otherMerchantIds = [...new Set(merchantIds)];
    } catch (e) {
      data.otherMerchantIds = [];
    }

    return data;
  }

  function createPanel(result, productData) {
    // Eski paneli temizle
    const existing = document.querySelector(".tk-panel");
    if (existing) existing.remove();
    const existingBtn = document.querySelector(".tk-toggle-btn");
    if (existingBtn) existingBtn.remove();

    // Toggle butonu
    const toggleBtn = document.createElement("button");
    toggleBtn.className = "tk-toggle-btn tk-hidden";
    toggleBtn.innerHTML = '<svg viewBox="0 0 100 100" width="24" height="24"><circle cx="50" cy="50" r="48" fill="#FF6B00"/><circle cx="50" cy="50" r="34" fill="#0f172a"/><path d="M50 20 A30 30 0 0 1 80 50" stroke="#3b82f6" stroke-width="8" stroke-linecap="round" fill="none"/><path d="M80 50 A30 30 0 0 1 50 80" stroke="#2563eb" stroke-width="8" stroke-linecap="round" fill="none"/><path d="M50 80 A30 30 0 0 1 20 50" stroke="#3b82f6" stroke-width="8" stroke-linecap="round" fill="none"/><path d="M20 50 A30 30 0 0 1 50 20" stroke="#2563eb" stroke-width="8" stroke-linecap="round" fill="none"/><circle cx="50" cy="50" r="8" fill="#3b82f6"/></svg>';
    toggleBtn.title = "TPro360 Satış Analizi";
    toggleBtn.addEventListener("click", () => {
      panel.classList.remove("tk-collapsed");
      toggleBtn.classList.add("tk-hidden");
    });
    document.body.appendChild(toggleBtn);

    // Panel
    const panel = document.createElement("div");
    panel.className = "tk-panel";
    panel.innerHTML = buildPanelHTML(result, productData);

    document.body.appendChild(panel);
    attachPanelEvents(panel, toggleBtn, productData);

    // Gunluk goruntuleme limiti kontrol
    checkViewLimit(panel);
  }

  function checkViewLimit(panel) {
    if (!chrome.runtime || !chrome.runtime.sendMessage) return;
    chrome.runtime.sendMessage({ action: "CHECK_VIEW_LIMIT" }, (resp) => {
      if (chrome.runtime.lastError) return;
      if (resp && resp.allowed === false) {
        applyViewBlur(panel, resp.usage, resp.limit);
      }
    });
  }

  function applyViewBlur(panel, usage, limit) {
    const body = panel.querySelector(".tk-body");
    if (!body) return;
    body.style.filter = "blur(6px)";
    body.style.pointerEvents = "none";
    body.style.userSelect = "none";
    body.style.position = "relative";

    // Overlay mesaji
    const overlay = document.createElement("div");
    overlay.style.cssText = "position:absolute;top:0;left:0;right:0;bottom:0;display:flex;flex-direction:column;align-items:center;justify-content:center;z-index:10;background:rgba(255,255,255,0.7);border-radius:8px;padding:20px;text-align:center;";
    overlay.innerHTML = `
      <div style="font-size:28px;margin-bottom:8px">🔒</div>
      <div style="font-size:13px;font-weight:700;color:#333;margin-bottom:4px">Günlük Limit Aşıldı</div>
      <div style="font-size:11px;color:#666">${usage || '?'}/${limit || '?'} görüntüleme kullanıldı</div>
      <div style="font-size:10px;color:#f27a1a;margin-top:6px;font-weight:600">Planınızı yükselterek limiti artırabilirsiniz</div>
    `;
    body.parentElement.style.position = "relative";
    body.parentElement.appendChild(overlay);
  }

  // ═══════════════════════════════════════════
  //  FIYAT GEÇMİŞİ — Trendyol Sayfasına Enjeksiyon
  // ═══════════════════════════════════════════

  async function injectPriceHistoryChart(productId) {
    if (!productId || !chrome.runtime || !chrome.runtime.sendMessage) return;
    if (document.getElementById("tk-price-history-widget")) return;

    try {
      const resp = await new Promise((resolve) => {
        chrome.runtime.sendMessage(
          { action: "GET_PRICE_HISTORY", productId: productId, days: 90 },
          (r) => resolve(r)
        );
      });

      if (!resp || resp.status !== "ok" || !resp.data || resp.data.length < 2) return;

      const data = resp.data;
      const salesMap = resp.sales || {};
      const prices = data.map(d => d.p);
      const minPrice = Math.min(...prices);
      const maxPrice = Math.max(...prices);
      const priceRange = maxPrice - minPrice || 1;
      const padding = priceRange * 0.1;
      const yMin = minPrice - padding;
      const yMax = maxPrice + padding;
      const totalRange = yMax - yMin;
      const currentPrice = resp.currentPrice || prices[prices.length - 1];
      const firstPrice = prices[0];
      const priceDiff = currentPrice - firstPrice;
      const pricePct = firstPrice > 0 ? ((priceDiff / firstPrice) * 100).toFixed(1) : "0";
      const isDown = priceDiff < 0;
      const isUp = priceDiff > 0;

      // Her veri noktasının tarihine en yakın günlük satışı eşle
      const dataWithSales = data.map(d => {
        const dateKey = d.t ? d.t.slice(0, 10) : '';
        return { ...d, sold: salesMap[dateKey] || 0, dateKey, m: d.m || null };
      });

      // Satış verisi var mı?
      const hasSalesData = dataWithSales.some(d => d.sold > 0);
      const maxSold = hasSalesData ? Math.max(...dataWithSales.map(d => d.sold), 1) : 0;

      // SVG boyutları: fiyat üst %75, satış barları alt %25
      const W = 600, H_TOTAL = hasSalesData ? 220 : 180;
      const H_PRICE = hasSalesData ? 160 : 180;
      const H_SALES = hasSalesData ? 50 : 0;
      const SALES_TOP = H_PRICE + 10; // 10px boşluk

      const pointCoords = data.map((d, i) => {
        const x = (i / Math.max(data.length - 1, 1)) * W;
        const y = H_PRICE - ((d.p - yMin) / totalRange) * H_PRICE;
        return { x, y, d };
      });
      const linePoints = pointCoords.map(p => `${p.x.toFixed(1)},${p.y.toFixed(1)}`).join(' ');
      const areaPoints = `0,${H_PRICE} ${linePoints} ${W},${H_PRICE}`;

      // Orijinal fiyat çizgisi
      const hasOrigPrice = data.some(d => d.op && d.op > d.p);
      let origLine = '';
      if (hasOrigPrice) {
        const origPoints = data.map((d, i) => {
          const x = (i / Math.max(data.length - 1, 1)) * W;
          const pv = (d.op && d.op > d.p) ? d.op : d.p;
          const y = H_PRICE - ((pv - yMin) / totalRange) * H_PRICE;
          return `${x.toFixed(1)},${y.toFixed(1)}`;
        }).join(' ');
        origLine = `<polyline fill="none" stroke="#94a3b8" stroke-width="1" stroke-dasharray="4" points="${origPoints}" />`;
      }

      // Plus fiyat çizgisi
      const hasPlusPrice = data.some(d => d.pp && d.pp > 0 && d.pp < d.p);
      let plusLine = '';
      if (hasPlusPrice) {
        const plusPts = data.filter(d => d.pp && d.pp > 0).map((d) => {
          const idx = data.indexOf(d);
          const x = (idx / Math.max(data.length - 1, 1)) * W;
          const y = H_PRICE - ((d.pp - yMin) / totalRange) * H_PRICE;
          return `${x.toFixed(1)},${y.toFixed(1)}`;
        }).join(' ');
        plusLine = `<polyline fill="none" stroke="#a855f7" stroke-width="1.5" stroke-dasharray="3" points="${plusPts}" />`;
      }

      // Satış barları SVG
      let salesBars = '';
      if (hasSalesData) {
        // Günlük satışları grupla (aynı gün birden fazla reading olabilir)
        const dailySales = {};
        dataWithSales.forEach(d => {
          if (d.sold > 0 && d.dateKey) {
            dailySales[d.dateKey] = d.sold;
          }
        });

        // Ayırıcı çizgi
        salesBars += `<line x1="0" y1="${SALES_TOP - 5}" x2="${W}" y2="${SALES_TOP - 5}" stroke="#e5e7eb" stroke-width="0.5" />`;

        // Her veri noktası için bar
        const barW = Math.max(W / data.length * 0.6, 2);
        const drawnDays = new Set();
        dataWithSales.forEach((d, i) => {
          if (d.sold > 0 && !drawnDays.has(d.dateKey)) {
            drawnDays.add(d.dateKey);
            const x = (i / Math.max(data.length - 1, 1)) * W;
            const barH = (d.sold / maxSold) * H_SALES;
            const barY = SALES_TOP + H_SALES - barH;
            salesBars += `<rect x="${(x - barW/2).toFixed(1)}" y="${barY.toFixed(1)}" width="${barW.toFixed(1)}" height="${barH.toFixed(1)}" fill="#3b82f6" opacity="0.6" rx="1" />`;
          }
        });
      }

      // Tarih aralığı
      const firstDate = data[0].t ? new Date(data[0].t).toLocaleDateString('tr-TR', { day: 'numeric', month: 'short' }) : '';
      const lastDate = data[data.length - 1].t ? new Date(data[data.length - 1].t).toLocaleDateString('tr-TR', { day: 'numeric', month: 'short' }) : '';
      const midIdx = Math.floor(data.length / 2);
      const midDate = data[midIdx]?.t ? new Date(data[midIdx].t).toLocaleDateString('tr-TR', { day: 'numeric', month: 'short' }) : '';

      const trendColor = isDown ? '#22c55e' : isUp ? '#ef4444' : '#94a3b8';
      const trendIcon = isDown ? '↓' : isUp ? '↑' : '→';
      const trendText = `${trendIcon} %${Math.abs(pricePct)}`;

      const widget = document.createElement("div");
      widget.id = "tk-price-history-widget";
      widget.innerHTML = `
        <div class="tk-ph-header">
          <div class="tk-ph-title">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#f97316" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="22 12 18 12 15 21 9 3 6 12 2 12"></polyline></svg>
            <span>Fiyat Geçmişi</span>
            <span class="tk-ph-badge">TPRO360</span>
          </div>
          <div class="tk-ph-stats">
            <span class="tk-ph-current">${currentPrice.toFixed(2)} TL</span>
            <span class="tk-ph-trend" style="color:${trendColor}">${trendText}</span>
          </div>
        </div>
        <div class="tk-ph-chart-container" style="height:${H_TOTAL}px">
          <svg viewBox="0 0 ${W} ${H_TOTAL}" preserveAspectRatio="none" class="tk-ph-svg">
            <defs>
              <linearGradient id="tkPhGrad" x1="0" y1="0" x2="0" y2="1">
                <stop offset="0%" stop-color="#f97316" stop-opacity="0.3" />
                <stop offset="100%" stop-color="#f97316" stop-opacity="0.02" />
              </linearGradient>
            </defs>
            <line x1="0" y1="${H_PRICE*0.25}" x2="${W}" y2="${H_PRICE*0.25}" stroke="#e5e7eb" stroke-width="0.5" stroke-dasharray="3" />
            <line x1="0" y1="${H_PRICE*0.5}" x2="${W}" y2="${H_PRICE*0.5}" stroke="#e5e7eb" stroke-width="0.5" stroke-dasharray="3" />
            <line x1="0" y1="${H_PRICE*0.75}" x2="${W}" y2="${H_PRICE*0.75}" stroke="#e5e7eb" stroke-width="0.5" stroke-dasharray="3" />
            <polygon fill="url(#tkPhGrad)" points="${areaPoints}" />
            <polyline fill="none" stroke="#f97316" stroke-width="2" stroke-linejoin="round" points="${linePoints}" />
            ${origLine}
            ${plusLine}
            ${salesBars}
          </svg>
          <div class="tk-ph-y-labels">
            <span>${maxPrice.toFixed(0)} TL</span>
            <span>${((maxPrice + minPrice) / 2).toFixed(0)} TL</span>
            <span>${minPrice.toFixed(0)} TL</span>
          </div>
          ${hasSalesData ? `<div class="tk-ph-sales-label">${maxSold} adet</div>` : ''}
          <div class="tk-ph-tooltip" id="tk-ph-tooltip" style="display:none"></div>
        </div>
        <div class="tk-ph-x-labels">
          <span>${firstDate}</span>
          ${data.length > 4 ? `<span>${midDate}</span>` : ''}
          <span>${lastDate}</span>
        </div>
        <div class="tk-ph-legend">
          <span><i class="tk-ph-line-orange"></i> Satış Fiyatı</span>
          ${hasOrigPrice ? '<span><i class="tk-ph-line-gray"></i> Liste Fiyatı</span>' : ''}
          ${hasPlusPrice ? '<span><i class="tk-ph-line-purple"></i> Plus Fiyat</span>' : ''}
          ${hasSalesData ? '<span><i class="tk-ph-bar-blue"></i> Günlük Satış</span>' : ''}
        </div>
      `;

      // Hover interaksiyon
      const svgEl = widget.querySelector('.tk-ph-svg');
      const tooltipEl = widget.querySelector('#tk-ph-tooltip');
      if (svgEl && tooltipEl) {
        const chartContainer = widget.querySelector('.tk-ph-chart-container');
        chartContainer.addEventListener('mousemove', (e) => {
          const rect = svgEl.getBoundingClientRect();
          const xPct = (e.clientX - rect.left) / rect.width;
          const idx = Math.min(Math.max(Math.round(xPct * (data.length - 1)), 0), data.length - 1);
          const d = dataWithSales[idx];
          const dateStr = d.t ? new Date(d.t).toLocaleDateString('tr-TR', { day: 'numeric', month: 'short', year: 'numeric' }) : '';
          let html = `<div style="font-size:11px;color:#888;margin-bottom:2px">${dateStr}</div>`;
          html += `<div style="font-weight:700;color:#f97316">${d.p.toFixed(2)} TL</div>`;
          if (d.op && d.op > d.p) html += `<div style="color:#999;font-size:11px">Liste: ${d.op.toFixed(2)} TL</div>`;
          if (d.pp && d.pp > 0 && d.pp < d.p) html += `<div style="color:#a855f7;font-size:11px">Plus: ${d.pp.toFixed(2)} TL</div>`;
          if (d.m) html += `<div style="color:#6366f1;font-size:11px">Satıcı: ${d.m}</div>`;
          if (d.sold > 0) html += `<div style="color:#3b82f6;font-size:11px;font-weight:600">Satış: ${d.sold} adet</div>`;
          tooltipEl.innerHTML = html;
          tooltipEl.style.display = 'block';
          const leftPx = Math.min(Math.max(xPct * 100, 8), 88);
          tooltipEl.style.left = leftPx + '%';
        });
        chartContainer.addEventListener('mouseleave', () => {
          tooltipEl.style.display = 'none';
        });
      }

      // Trendyol sayfasına enjekte et
      const targets = [
        ".content-description-main-container",
        ".content-description-container",
        ".content-description-wrapper",
        "[class*='product-info-main-container']",
        "[class*='product-details-other-details']",
        "[class*='product-details-product-details-container']",
      ];

      let injected = false;
      for (const sel of targets) {
        const target = document.querySelector(sel);
        if (target && target.parentNode) {
          target.parentNode.insertBefore(widget, target);
          injected = true;
          tkLog("Fiyat geçmişi enjekte edildi:", sel);
          break;
        }
      }

      if (!injected) {
        const mainContent = document.querySelector("[class*='product-detail-app'], [class*='product-container'], main, #root > div");
        if (mainContent) {
          mainContent.appendChild(widget);
          injected = true;
          tkLog("Fiyat geçmişi fallback enjekte edildi");
        }
      }

      if (!injected) tkLog("Fiyat geçmişi: enjeksiyon hedefi bulunamadı!");
    } catch (e) {
      tkLog("Fiyat geçmişi enjeksiyon hatası:", e.message);
    }
  }

  // ═══════════════════════════════════════════
  //  SVG CHART BUILDERS
  // ═══════════════════════════════════════════

  /**
   * Fiyat trendi SVG area chart — 270×50px
   * @param {Array} priceHistory - [{date, min, max, avg}]
   */
  function buildPriceChartSVG(priceHistory) {
    if (!priceHistory || priceHistory.length < 2) return '';

    const W = 270, H = 50, PAD_X = 0, PAD_Y = 4;
    const chartW = W - PAD_X * 2;
    const chartH = H - PAD_Y * 2;

    const prices = priceHistory.map(p => p.avg);
    const minP = Math.min(...prices);
    const maxP = Math.max(...prices);
    const range = maxP - minP || 1;

    const firstP = prices[0];
    const lastP = prices[prices.length - 1];
    const pctChange = ((lastP - firstP) / firstP * 100).toFixed(1);
    const trendDir = lastP < firstP ? 'down' : lastP > firstP ? 'up' : 'flat';
    const trendIcon = trendDir === 'down' ? '↓' : trendDir === 'up' ? '↑' : '↔';
    const trendClass = 'tk-trend-' + trendDir;

    // Renk: düşüş yeşil, artış kırmızı
    const strokeColor = trendDir === 'down' ? '#00b365' : trendDir === 'up' ? '#e53e3e' : '#999';
    const fillColor = trendDir === 'down' ? 'rgba(0,179,101,0.12)' : trendDir === 'up' ? 'rgba(229,62,62,0.12)' : 'rgba(153,153,153,0.08)';

    // Nokta koordinatları hesapla
    const points = prices.map((p, i) => {
      const x = PAD_X + (i / (prices.length - 1)) * chartW;
      const y = PAD_Y + chartH - ((p - minP) / range) * chartH;
      return { x: Math.round(x * 10) / 10, y: Math.round(y * 10) / 10 };
    });

    // Polyline string
    const linePoints = points.map(p => p.x + ',' + p.y).join(' ');
    // Polygon (area fill): ilk nokta → tüm noktalar → son noktadan tabana
    const areaPoints = PAD_X + ',' + H + ' ' + linePoints + ' ' + (PAD_X + chartW) + ',' + H;

    // Tarih label'ları
    const firstDate = priceHistory[0].date.slice(5); // MM-DD
    const lastDate = priceHistory[priceHistory.length - 1].date.slice(5);

    const formatP = (v) => v >= 1000 ? Math.round(v).toLocaleString('tr-TR') : v.toFixed(2);

    return `
    <div class="tk-chart-section">
      <div class="tk-chart-header">
        <span class="tk-chart-title">Fiyat Trendi</span>
        <span class="tk-chart-trend ${trendClass}">${trendIcon} %${Math.abs(pctChange)}</span>
      </div>
      <svg class="tk-price-chart-svg" viewBox="0 0 ${W} ${H}" preserveAspectRatio="none">
        <polygon points="${areaPoints}" fill="${fillColor}" />
        <polyline points="${linePoints}" fill="none" stroke="${strokeColor}" stroke-width="1.5" stroke-linejoin="round" stroke-linecap="round" />
        <circle cx="${points[points.length - 1].x}" cy="${points[points.length - 1].y}" r="2.5" fill="${strokeColor}" />
      </svg>
      <div class="tk-chart-range">
        <span>${firstDate} — ${formatP(firstP)} TL</span>
        <span>${lastDate} — ${formatP(lastP)} TL</span>
      </div>
    </div>`;
  }

  /**
   * Satış trendi SVG bar chart + trend line — 270×50px
   * @param {Array} last7Days - [{date, sales, readings}]
   */
  function buildSalesChartSVG(last7Days) {
    if (!last7Days || !last7Days.some(d => d.sales > 0)) return '';

    const W = 270, H = 50, PAD_X = 5, PAD_Y = 4;
    const chartW = W - PAD_X * 2;
    const chartH = H - PAD_Y * 2 - 10; // 10px alt kısım gün label için

    const sales = last7Days.map(d => d.sales);
    const maxSale = Math.max(...sales, 1);
    const totalSales = sales.reduce((a, b) => a + b, 0);

    const barW = Math.floor(chartW / 7) - 4;
    const barGap = (chartW - barW * 7) / 6;

    let barsHtml = '';
    const barCenters = [];

    for (let i = 0; i < 7; i++) {
      const x = PAD_X + i * (barW + barGap);
      const barH = Math.max(2, Math.round((sales[i] / maxSale) * chartH));
      const y = PAD_Y + chartH - barH;
      const opacity = sales[i] > 0 ? 0.8 : 0.2;
      const dayLabel = last7Days[i].date.slice(8); // DD

      barsHtml += '<rect x="' + x + '" y="' + y + '" width="' + barW + '" height="' + barH + '" rx="2" fill="#9c27b0" opacity="' + opacity + '" />';
      barsHtml += '<text x="' + (x + barW / 2) + '" y="' + (H - 1) + '" text-anchor="middle" fill="#999" font-size="7" font-family="sans-serif">' + dayLabel + '</text>';

      barCenters.push({
        x: x + barW / 2,
        y: y + barH / 2
      });
    }

    // Trend line (kesikli çizgi, tepeleri birleştir)
    const trendPoints = [];
    for (let i = 0; i < 7; i++) {
      const x = PAD_X + i * (barW + barGap) + barW / 2;
      const barH = Math.max(2, Math.round((sales[i] / maxSale) * chartH));
      const y = PAD_Y + chartH - barH;
      trendPoints.push(x + ',' + y);
    }
    const trendLine = '<polyline points="' + trendPoints.join(' ') + '" fill="none" stroke="#7b1fa2" stroke-width="1" stroke-dasharray="3,2" stroke-linejoin="round" opacity="0.6" />';

    return `
    <div class="tk-chart-section tk-chart-sales">
      <div class="tk-chart-header">
        <span class="tk-chart-title">Satis Trendi (7 Gun)</span>
        <span class="tk-chart-total">${totalSales} adet</span>
      </div>
      <svg class="tk-sales-chart-svg" viewBox="0 0 ${W} ${H}" preserveAspectRatio="none">
        ${barsHtml}
        ${trendLine}
      </svg>
    </div>`;
  }

  function buildPanelHTML(result, productData) {
    const c = result.commission;
    const s = result.shipping;
    const confidenceClass = c.confidence > 0.6 ? "tk-high" : c.confidence > 0.3 ? "tk-medium" : "tk-low";
    const confidenceLabel = c.confidence > 0.6 ? "Yüksek" : c.confidence > 0.3 ? "Orta" : "Düşük";

    return `
      <div class="tk-header">
        <div class="tk-header-title">
          <svg class="tk-header-logo" viewBox="0 0 100 100" fill="none" width="22" height="22">
            <circle cx="50" cy="50" r="48" fill="#FF6B00"/>
            <circle cx="50" cy="50" r="34" fill="#0f172a"/>
            <path d="M50 20 A30 30 0 0 1 80 50" stroke="#3b82f6" stroke-width="8" stroke-linecap="round" fill="none"/>
            <path d="M80 50 A30 30 0 0 1 50 80" stroke="#2563eb" stroke-width="8" stroke-linecap="round" fill="none"/>
            <path d="M50 80 A30 30 0 0 1 20 50" stroke="#3b82f6" stroke-width="8" stroke-linecap="round" fill="none"/>
            <path d="M20 50 A30 30 0 0 1 50 20" stroke="#2563eb" stroke-width="8" stroke-linecap="round" fill="none"/>
            <circle cx="50" cy="50" r="8" fill="#3b82f6"/>
          </svg>
          <span>TPro360</span> Satış Analizi
        </div>
        <div class="tk-header-actions">
          <button class="tk-btn-icon tk-btn-minimize" title="Küçült">−</button>
          <button class="tk-btn-icon tk-btn-close" title="Gizle">×</button>
        </div>
      </div>
      <div class="tk-body">
        <!-- Satış Fiyatı -->
        <div class="tk-section">
          <div class="tk-section-title">Fiyat Bilgisi</div>
          <div class="tk-row">
            <span class="tk-row-label">Satış Fiyatı (KDV Dahil)</span>
            <span class="tk-row-value tk-highlight">${TrendyolCalculator.formatPrice(result.salePrice)}</span>
          </div>
        </div>

        <!-- Komisyon -->
        <div class="tk-section">
          <div class="tk-section-title">Komisyon</div>
          <div class="tk-row">
            <span class="tk-row-label">Komisyon Oranı</span>
            <span class="tk-row-value">${TrendyolCalculator.formatPercent(c.rate)}</span>
          </div>
          <div class="tk-row">
            <span class="tk-row-label">Komisyon Tutarı</span>
            <span class="tk-row-value tk-negative">${TrendyolCalculator.formatPrice(c.amount)}</span>
          </div>
          <div class="tk-category-match">
            Kategori: ${c.match}
            <span class="tk-confidence ${confidenceClass}">${confidenceLabel}</span>
          </div>
          <div class="tk-vade-info">Ödeme vadesi: ${c.vade} gün</div>
          <div class="tk-input-group" style="margin-top:6px">
            <label>Komisyon Oranını Değiştir (%)</label>
            <input type="number" class="tk-input tk-commission-input"
              value="${c.rate}" step="0.5" min="0" max="50"
              placeholder="Komisyon oranı">
          </div>
        </div>

        <!-- Kargo & Hizmet -->
        <div class="tk-section">
          <div class="tk-section-title">Kargo & Kesintiler</div>
          <div class="tk-row">
            <span class="tk-row-label">Kargo Firması</span>
            <span class="tk-row-value">${s.carrier}</span>
          </div>
          <div class="tk-row">
            <span class="tk-row-label">Kargo Ücreti (KDV Dahil)</span>
            <span class="tk-row-value tk-negative">${TrendyolCalculator.formatPrice(s.total)}</span>
          </div>
          <div class="tk-row">
            <span class="tk-row-label">Hizmet Bedeli (KDV Dahil)</span>
            <span class="tk-row-value tk-negative">${TrendyolCalculator.formatPrice(result.hizmetBedeli)}</span>
          </div>
          <div class="tk-category-match" style="font-size:10px">
            ${s.tier || ""}${s.method === "desi" ? "" : " (barem)"}
          </div>
          <div class="tk-input-group" style="margin-top:6px">
            <label>Kargo Firması</label>
            <select class="tk-select tk-carrier-select">
              <option value="TEX-PTT" ${settings.carrier === "TEX-PTT" ? "selected" : ""}>TEX-PTT (En Ucuz)</option>
              <option value="ARAS" ${settings.carrier === "ARAS" ? "selected" : ""}>Aras Kargo</option>
              <option value="SÜRAT" ${settings.carrier === "SÜRAT" ? "selected" : ""}>Sürat Kargo</option>
              <option value="KOLAY GELSİN" ${settings.carrier === "KOLAY GELSİN" ? "selected" : ""}>Kolay Gelsin</option>
              <option value="DHLeCommerce" ${settings.carrier === "DHLeCommerce" ? "selected" : ""}>DHL eCommerce</option>
              <option value="YK" ${settings.carrier === "YK" ? "selected" : ""}>Yurtiçi Kargo</option>
            </select>
          </div>
          <div class="tk-input-group" style="margin-top:6px">
            <label>Desi (kg)</label>
            <input type="number" class="tk-input tk-desi-input" placeholder="Bos = Barem" value="" step="0.5" min="0.5" max="500" title="Ürüne ait farklı desi girmek isterseniz doldurunuz">
            <div class="tk-category-match" style="font-size:9px;margin-top:3px;color:#888">Ürüne ait farklı desi girmek isterseniz doldurunuz</div>
          </div>
        </div>

        <!-- Satış Tahmini -->
        <div class="tk-section">
          <div class="tk-section-title">Satış Tahmini</div>
          <div class="tk-row">
            <span class="tk-row-label">Yorum Sayısı</span>
            <span class="tk-row-value">${productData.reviewCount || 0}</span>
          </div>
          <div class="tk-row">
            <span class="tk-row-label">Tahmini Aylık Satış</span>
            <span class="tk-row-value">${result.estimatedMin > 0 && result.estimatedMax > 0 ? (result.estimatedMin.toLocaleString("tr-TR") + " - " + result.estimatedMax.toLocaleString("tr-TR")) : result.estimatedMonthlySales} adet</span>
          </div>${productData.recentSalesData ? `
          <div class="tk-row">
            <span class="tk-row-label">Trendyol Satış (son ${productData.recentSalesData.days} gün)</span>
            <span class="tk-row-value">${productData.recentSalesData.count.toLocaleString("tr-TR")}+ adet</span>
          </div>` : (result.rawOrderCount ? `
          <div class="tk-row">
            <span class="tk-row-label">Trendyol Sipariş (son ${result.orderPeriodDays || 7} gün)</span>
            <span class="tk-row-value">${result.rawOrderCount} sipariş</span>
          </div>` : "")}${productData.stockQuantity ? `
          <div class="tk-row">
            <span class="tk-row-label">Satıcı Stok</span>
            <span class="tk-row-value">${productData.stockQuantity.toLocaleString("tr-TR")} adet</span>
          </div>` : ""}
          ${(() => {
            const pid = productData.productId;
            const isTracked = pid && isProductTracked(pid);
            const td = productData.trackingData;
            let trackingHtml = "";

            // Takip Et / Takipte butonu
            if (pid) {
              trackingHtml += `
              <div class="tk-row" style="border-top:1px solid #f0f0f0;padding-top:8px;margin-top:4px">
                <span class="tk-row-label">Stok Takibi</span>
                <button class="tk-track-btn${isTracked ? " tk-tracking" : ""}"
                  data-product-id="${pid}"
                  data-slug="${window.location.pathname}"
                  data-name="${(productData.title || "").replace(/"/g, "&quot;")}"
                  data-price="${productData.price || 0}">
                  ${isTracked ? "&#10003; Takipte" : "+ Takip Et"}
                </button>
              </div>`;
            }

            // insufficient_data: henüz yeterli izleme süresi yok
            if (td && td.estimateSource === "insufficient_data") {
              trackingHtml += `
              <div class="tk-tracking-summary" style="padding:8px;text-align:center;color:#999;font-size:11px">
                ⏳ Yeterli izleme suresi yok (min. 3 saat gerekli)
              </div>`;
            }

            // T0 tracking özeti — yeni motor: günlük/haftalık/aylık
            // TUTARLILIK: result.estimatedMonthlySales (calculator blend) kullan,
            // td.dailyEstimate (ham T0 projeksiyon) yerine — 3 görünüm aynı sayıyı göstersin
            else if (td && (td.dailyEstimate > 0 || td.totalRealSold > 0 || td.totalReadings >= 2)) {
              const confPct = Math.round((td.confidence || 0) * 100);
              const dataRank = td.dataQualityRank || (productData.trackingData && productData.trackingData.dataQualityRank) || 1;

              // Blended estimate: calculator sonucu varsa onu kullan (panel ile tutarlı)
              const blendedMonthly = result.estimatedMonthlySales > 0 ? result.estimatedMonthlySales : (td.monthlyEstimate || 0);
              const blendedDaily = Math.round(blendedMonthly / 30);
              const blendedWeekly = Math.round(blendedMonthly / 30 * 7);

              // Satış aralığı: result'tan veya panelEstimate'den al, yoksa hesapla
              let rangeMin = result.estimatedMin || 0;
              let rangeMax = result.estimatedMax || 0;
              if (rangeMin <= 0 && blendedMonthly > 0) {
                const pe2 = (productData.trackingData && productData.trackingData.panelEstimate) || productData.panelEstimate;
                if (pe2 && pe2.min > 0) {
                  rangeMin = pe2.min;
                  rangeMax = pe2.max || blendedMonthly;
                } else {
                  const range = TrendyolCalculator.computeSalesRange(blendedMonthly, {
                    dataQualityRank: dataRank,
                    tier: result.salesTier || null,
                    daysTracked: td.daysTracked || 0,
                  });
                  rangeMin = range.min;
                  rangeMax = range.max;
                }
              }

              // Kaynak etiketi
              const src = td.estimateSource || "none";
              const sourceLabel = src === "weekly_avg" ? "7 gun verisine dayali"
                : src === "daily_avg" ? (td.totalReadings || 0) + " olcum verisi"
                : td.daysTracked >= 3 ? td.daysTracked + " gun takip"
                : "erken tahmin";

              // Satış trendi SVG chart
              const salesChartHtml = buildSalesChartSVG(td.last7Days);

              // Virtual stok badge
              const virtualBadge = td.stockTrackingMode === 'virtual'
                ? '<span style="font-size:8px;background:#fff3e0;color:#e65100;padding:1px 4px;border-radius:3px;margin-left:4px" title="Sanal stok tespit edildi — satis tahmini socialProof bazli">SANAL</span>'
                : '';

              trackingHtml += `
              <div class="tk-tracking-summary">
                <div class="tk-ts-header">
                  <span class="tk-ts-label">Satis Analizi${virtualBadge}</span>
                  <span class="tk-ts-readings">${td.totalReadings || 0} olcum</span>
                </div>
                <div class="tk-ts-grid">
                  <div class="tk-ts-cell">
                    <span class="tk-ts-num">${blendedDaily}</span>
                    <span class="tk-ts-unit">/ gun</span>
                  </div>
                  <div class="tk-ts-cell">
                    <span class="tk-ts-num">${blendedWeekly}</span>
                    <span class="tk-ts-unit">/ hafta</span>
                  </div>
                  <div class="tk-ts-cell">
                    <span class="tk-ts-num">${rangeMin > 0 && rangeMax > 0 ? (rangeMin.toLocaleString("tr-TR") + " - " + rangeMax.toLocaleString("tr-TR")) : Math.round(blendedMonthly)}</span>
                    <span class="tk-ts-unit">/ ay</span>
                  </div>
                </div>
                ${salesChartHtml}
                <div class="tk-ts-footer">
                  <div class="tk-rank-bar">${[...Array(10)].map((_, i) => `<div class="tk-rank-pip ${i < dataRank ? (dataRank >= 7 ? 'tk-rank-green' : dataRank >= 5 ? 'tk-rank-yellow' : dataRank >= 3 ? 'tk-rank-orange' : 'tk-rank-gray') : 'tk-rank-empty'}"></div>`).join('')}<span class="tk-rank-num">${dataRank}</span></div>
                  <span class="tk-ts-source">${sourceLabel}</span>
                </div>
              </div>`;

              // Fiyat trendi SVG chart (dailySummaries'den priceHistory varsa göster)
              if (td.priceHistory && td.priceHistory.length >= 2) {
                trackingHtml += buildPriceChartSVG(td.priceHistory);
              }
            }

            // Fiyat değişikliği
            if (td && td.lastPriceChange) {
              const pc = td.lastPriceChange;
              const direction = pc.newPrice > pc.oldPrice ? "up" : "down";
              const arrow = direction === "up" ? "&#8593;" : "&#8595;";
              const daysAgo = Math.floor((Date.now() - pc.ts) / 86400000);
              trackingHtml += `
              <div class="tk-row">
                <span class="tk-row-label">Fiyat Degisikligi</span>
                <span class="tk-row-value">
                  <span class="tk-price-change tk-price-${direction}">
                    ${arrow} ${TrendyolCalculator.formatPrice(pc.oldPrice)} &rarr; ${TrendyolCalculator.formatPrice(pc.newPrice)}
                    (${daysAgo === 0 ? "bugun" : daysAgo + " gun once"})
                  </span>
                </span>
              </div>`;
            }

            return trackingHtml;
          })()}
          <!-- Satış tahmini otomatik hesaplanır -->
        </div>

        <!-- Özet — Karlılık Hesabı (KDV Hariç) -->
        <div class="tk-summary">
          <div class="tk-summary-title">Karlılık Hesabı (KDV Hariç)</div>
          <div class="tk-row">
            <span class="tk-row-label">Satış Fiyatı (KDV Hariç)</span>
            <span class="tk-row-value">${TrendyolCalculator.formatPrice(result.salePriceExVat)}</span>
          </div>
          <div class="tk-row">
            <span class="tk-row-label">(-) Komisyon (${TrendyolCalculator.formatPercent(c.rate)})</span>
            <span class="tk-row-value tk-negative">-${TrendyolCalculator.formatPrice(c.amountExVat)}</span>
          </div>
          <div class="tk-row">
            <span class="tk-row-label">(-) Hizmet Bedeli</span>
            <span class="tk-row-value tk-negative">-${TrendyolCalculator.formatPrice(result.hizmetBedeliExVat)}</span>
          </div>
          <div class="tk-row">
            <span class="tk-row-label">(-) Kargo</span>
            <span class="tk-row-value tk-negative">-${TrendyolCalculator.formatPrice(result.shippingExVat)}</span>
          </div>
          <div class="tk-row" style="border-top:2px solid #f27a1a;padding-top:8px">
            <span class="tk-row-label"><b>Hesaba Geçecek</b></span>
            <span class="tk-row-value" style="font-size:15px;color:#f27a1a"><b>${TrendyolCalculator.formatPrice(result.hesabaGececek)}</b></span>
          </div>
          <div class="tk-row">
            <span class="tk-row-label">(-) Maliyet ${result.costPrice ? "(Manuel)" : `(Hedef %${Math.round(result.targetProfitPct)} kâr)`}</span>
            <span class="tk-row-value">-${TrendyolCalculator.formatPrice(result.usedCostPrice)}</span>
          </div>
          <div class="tk-row" style="border-top:2px solid #00b365;padding-top:8px">
            <span class="tk-row-label"><b>Kâr</b></span>
            <span class="tk-row-value ${result.profit >= 0 ? "tk-positive" : "tk-negative"}" style="font-size:15px"><b>${TrendyolCalculator.formatPrice(result.profit)}</b></span>
          </div>
          <div class="tk-row">
            <span class="tk-row-label">Kâr Marjı</span>
            <span class="tk-row-value ${result.profitMargin >= 0 ? "tk-positive" : "tk-negative"}">${TrendyolCalculator.formatPercent(result.profitMargin)}</span>
          </div>
          <div class="tk-category-match" style="font-size:9px;margin-top:4px;color:#999">
            KDV Dahil: Satış ${TrendyolCalculator.formatPrice(result.salePrice)} | Komisyon ${TrendyolCalculator.formatPrice(c.amount)} | KDV %${Math.round(result.kdv.rate)}${productData.taxRate !== undefined ? ' ✓' : ''}
          </div>
        </div>

        <!-- Maliyet & Kâr Ayarları -->
        <div class="tk-section">
          <div class="tk-section-title">Maliyet & Kâr</div>
          <div class="tk-row">
            <span class="tk-row-label">Maks. Maliyet</span>
            <span class="tk-row-value">${TrendyolCalculator.formatPrice(result.autoEstimatedCost)}</span>
          </div>
          <div class="tk-category-match" style="font-size:10px">
            Bu fiyatın altında alırsanız %${Math.round(result.targetProfitPct)} kâr edersiniz
          </div>
          <div class="tk-input-group" style="margin-top:6px">
            <label>Hedef Kâr Oranı (%)</label>
            <input type="number" class="tk-input tk-target-profit-input"
              value="${Math.round(result.targetProfitPct)}" step="1" min="0" max="100"
              placeholder="Ör: 20">
          </div>
          <div class="tk-input-group" style="margin-top:6px">
            <label>Gerçek Maliyet Fiyatı (TL)</label>
            <input type="number" class="tk-input tk-cost-input"
              value="" step="0.01" min="0"
              placeholder="Maliyeti girin (opsiyonel)">
          </div>
          <div class="tk-row" style="border-top:1px dashed #ddd;padding-top:8px;margin-top:4px">
            <span class="tk-row-label">Tahmini Aylık Satış</span>
            <span class="tk-row-value">${result.estimatedMin > 0 && result.estimatedMax > 0 ? (result.estimatedMin.toLocaleString("tr-TR") + " - " + result.estimatedMax.toLocaleString("tr-TR")) : result.estimatedMonthlySales} adet</span>
          </div>
          <div class="tk-row">
            <span class="tk-row-label">Tahmini Aylık Kâr</span>
            <span class="tk-row-value tk-highlight" style="font-size:15px"><b>${result.estimatedMin > 0 ? (TrendyolCalculator.formatPrice(result.profit * result.estimatedMin) + " - " + TrendyolCalculator.formatPrice(result.profit * result.estimatedMax)) : TrendyolCalculator.formatPrice(result.monthlyProfit)}</b></span>
          </div>
        </div>
      </div>
    `;
  }

  function attachPanelEvents(panel, toggleBtn, productData) {
    // Kapatma
    panel.querySelector(".tk-btn-close").addEventListener("click", () => {
      panel.classList.add("tk-collapsed");
      toggleBtn.classList.remove("tk-hidden");
    });

    // Küçültme
    const body = panel.querySelector(".tk-body");
    let minimized = false;
    panel.querySelector(".tk-btn-minimize").addEventListener("click", (e) => {
      minimized = !minimized;
      body.style.display = minimized ? "none" : "block";
      e.target.textContent = minimized ? "+" : "−";
    });

    // Komisyon değişikliği
    const commInput = panel.querySelector(".tk-commission-input");
    commInput.addEventListener("change", () => {
      recalculate(panel, toggleBtn, productData);
    });

    // Kargo firması değişikliği
    const carrierSelect = panel.querySelector(".tk-carrier-select");
    carrierSelect.addEventListener("change", () => {
      settings.carrier = carrierSelect.value;
      recalculate(panel, toggleBtn, productData);
    });

    // Desi girişi — kargo desi bazlı hesaplansın
    const desiInput = panel.querySelector(".tk-desi-input");
    if (desiInput) {
      desiInput.addEventListener("input", () => {
        const desiVal = parseFloat(desiInput.value);
        productData._userDesi = (desiVal > 0) ? desiVal : null;
        recalculate(panel, toggleBtn, productData);
      });
    }

    // Maliyet girişi
    const costInput = panel.querySelector(".tk-cost-input");
    costInput.addEventListener("input", () => {
      recalculate(panel, toggleBtn, productData);
    });

    // Hedef kâr % girişi
    const targetProfitInput = panel.querySelector(".tk-target-profit-input");
    if (targetProfitInput) {
      targetProfitInput.addEventListener("input", () => {
        recalculate(panel, toggleBtn, productData);
      });
    }

    // ─── Takip Et butonu ───
    const trackBtn = panel.querySelector(".tk-track-btn");
    if (trackBtn) {
      trackBtn.addEventListener("click", () => {
        const pid = trackBtn.dataset.productId;
        const isTracked = trackBtn.classList.contains("tk-tracking");

        if (isTracked) {
          // Takibi bırak
          chrome.runtime.sendMessage({ action: "UNTRACK_PRODUCT", productId: pid }, (resp) => {
            if (resp && resp.status === "ok") {
              trackBtn.classList.remove("tk-tracking");
              trackBtn.innerHTML = "+ Takip Et";
              trackedProductIds.delete(pid);
              panelTrackedIds.delete(pid);
            }
          });
        } else {
          // Takip başlat
          chrome.runtime.sendMessage({
            action: "TRACK_PRODUCT",
            product: {
              id: pid,
              slug: trackBtn.dataset.slug || window.location.pathname,
              name: trackBtn.dataset.name || productData.title || "",
              brand: productData.brand || "",
              price: parseFloat(trackBtn.dataset.price) || productData.price || 0,
              imageUrl: productData.imageUrl || "",
              stock: productData.stockQuantity || 0,
              merchantId: productData.merchantId || undefined,
              categoryPath: productData.categoryPath || "",
              reviewCount: productData.reviewCount || 0,
              favoriteCount: productData.favoriteCount || 0,
              otherMerchantIds: productData.otherMerchantIds || [],
              merchantName: productData.merchantName || '',
              otherMerchantNames: productData.otherMerchantNames || {},
            }
          }, (resp) => {
            if (resp && resp.status === "ok") {
              trackBtn.classList.add("tk-tracking");
              trackBtn.innerHTML = "&#10003; Takipte";
              trackedProductIds.add(pid);
              panelTrackedIds.add(pid);
            } else if (resp && resp.status === "already_tracking") {
              trackBtn.classList.add("tk-tracking");
              trackBtn.innerHTML = "&#10003; Takipte";
              trackedProductIds.add(pid);
              panelTrackedIds.add(pid);
            } else if (resp && resp.status === "limit_reached") {
              trackBtn.style.background = "#ef4444";
              trackBtn.style.color = "#fff";
              trackBtn.innerHTML = "Limit Doldu";
              setTimeout(() => {
                trackBtn.style.background = "";
                trackBtn.style.color = "";
                trackBtn.innerHTML = "+ Takip Et";
              }, 3000);
            }
          });
        }
      });
    }

    // Sürükle
    makeDraggable(panel, panel.querySelector(".tk-header"));
  }

  function recalculate(panel, toggleBtn, productData) {
    const commInput = panel.querySelector(".tk-commission-input");
    const costInput = panel.querySelector(".tk-cost-input");
    const desiInput = panel.querySelector(".tk-desi-input");
    const targetProfitInput = panel.querySelector(".tk-target-profit-input");

    const manualCommission = commInput ? parseFloat(commInput.value) : undefined;
    const costPrice = costInput && costInput.value ? parseFloat(costInput.value) : undefined;
    const desiVal = desiInput && desiInput.value ? parseFloat(desiInput.value) : undefined;
    const targetProfitPct = targetProfitInput && targetProfitInput.value ? parseFloat(targetProfitInput.value) : undefined;

    // Desi ve hedef kâr değerlerini productData'ya kaydet (panel yenilenince korunsun)
    if (desiVal > 0) productData._userDesi = desiVal;
    if (targetProfitPct !== undefined && !isNaN(targetProfitPct)) productData._userTargetProfit = targetProfitPct;

    // ─── Scroll ve pozisyon kaydet (panel rebuild sirasinda kaybolmasin) ───
    const panelScrollTop = panel.scrollTop;
    const pageScrollY = window.scrollY;
    const panelRect = panel.getBoundingClientRect();
    const panelTop = panel.style.top;
    const panelLeft = panel.style.left;
    const panelRight = panel.style.right;

    const result = TrendyolCalculator.calculate({
      salePrice: productData.price,
      breadcrumbs: productData.breadcrumbs,
      carrier: settings.carrier,
      desi: productData._userDesi || undefined,
      reviewCount: productData.reviewCount,
      rating: productData.rating,
      salesMultiplier: settings.salesMultiplier,
      costPrice: costPrice,
      manualCommission: manualCommission,
      badges: productData.badges,
      recentSalesData: productData.recentSalesData,
      socialProofData: productData.socialProofData,
      kdvRate: productData.taxRate,
      categoryTree: productData.categoryTree,
      targetProfitPct: productData._userTargetProfit
    });

    // Panel T0 bazlı ensemble (T0+T1+T3) hesaplıyor — tüm kaynaklar burada.
    // Panel tahmini varsa → authoritative kabul et (tek kaynak prensibi).
    // Extension hesabı sadece panel tahmini yoksa kullanılır (takip dışı ürün).
    const pe = (productData.trackingData && productData.trackingData.panelEstimate) || productData.panelEstimate;
    if (pe && pe.monthly > 0) {
      const extMonthly = result.estimatedMonthlySales || 0;
      // Panel tahmini var → panel'i kullan (T0 bazlı ensemble en doğru sonuç)
      result.estimatedMonthlySales = pe.monthly;
      result.estimatedMin = pe.min || Math.round(pe.monthly * 0.8);
      result.estimatedMax = pe.max || Math.round(pe.monthly * 1.2);
      result.salesTier = pe.tier || "T0";
      result.salesTierLabel = pe.tier || "T0";
      tkLog(`Panel T0 bazlı tahmin kullanıldı: ${pe.monthly}/ay ${pe.tier} (extension: ${extMonthly}/ay)`);
    }

    createPanel(result, productData);

    // Girilen değerleri geri koy
    const newPanel = document.querySelector(".tk-panel");
    if (costPrice !== undefined) {
      newPanel.querySelector(".tk-cost-input").value = costPrice;
    }
    // Desi değerini geri koy
    if (productData._userDesi > 0) {
      const newDesiInput = newPanel.querySelector(".tk-desi-input");
      if (newDesiInput) newDesiInput.value = productData._userDesi;
    }

    // ─── Scroll ve pozisyon geri yukle ───
    if (panelTop) newPanel.style.top = panelTop;
    if (panelLeft) newPanel.style.left = panelLeft;
    if (panelRight !== undefined) newPanel.style.right = panelRight;
    newPanel.scrollTop = panelScrollTop;
    // Sayfa scroll pozisyonunu koru (panel rebuild scrollu tetiklemesin)
    requestAnimationFrame(() => { window.scrollTo(0, pageScrollY); });
  }

  // Sürükle-bırak — singleton handlers (listener birikimini önle)
  let _dragMoveHandler = null;
  let _dragUpHandler = null;

  function makeDraggable(panel, handle) {
    let offsetX, offsetY, isDragging = false;

    // Önceki document listener'ları varsa kaldır (panel yeniden oluşturulduğunda)
    if (_dragMoveHandler) document.removeEventListener("mousemove", _dragMoveHandler);
    if (_dragUpHandler) document.removeEventListener("mouseup", _dragUpHandler);

    handle.addEventListener("mousedown", (e) => {
      isDragging = true;
      offsetX = e.clientX - panel.getBoundingClientRect().left;
      offsetY = e.clientY - panel.getBoundingClientRect().top;
      panel.style.transition = "none";
    });

    _dragMoveHandler = (e) => {
      if (!isDragging) return;
      const x = e.clientX - offsetX;
      const y = e.clientY - offsetY;
      panel.style.left = x + "px";
      panel.style.top = y + "px";
      panel.style.right = "auto";
    };

    _dragUpHandler = () => {
      isDragging = false;
      panel.style.transition = "transform 0.3s ease, opacity 0.3s ease";
    };

    document.addEventListener("mousemove", _dragMoveHandler);
    document.addEventListener("mouseup", _dragUpHandler);
  }

  // ===== LİSTE SAYFASI =====

  let tkFloatBtn = null;
  let tkOverlay = null;
  let tkCurrentProducts = [];
  let tkLastPageData = null;

  // Manuel override değerleri (tablo geneli)
  let tkManualOverrides = {
    kargo: null,       // Manuel kargo ücreti (TL)
    desi: null,        // Manuel desi değeri (kg) — kargo desi bazlı hesaplanır
    kdvRate: null,      // Manuel KDV oranı (%)
    satisAdedi: null,   // Manuel aylık satış adedi
    maliyet: null       // Manuel maliyet (TL)
  };

  // ═══════════════════════════════════════════════════════════════════
  //  AYLIK SATIŞ TAHMİNİ (Merkezi Hesaplama)
  // ═══════════════════════════════════════════════════════════════════
  function estimateMonthlySales(reviewCount, rating, price, badges = [], recentSalesData = null, socialProofData = null) {
    return TrendyolCalculator.estimateMonthlySales(reviewCount, rating, price, badges, recentSalesData, socialProofData);
  }

  // Kargo hesaplaması — TrendyolCalculator.getShippingCost(salePrice, carrier, desi, categoryPath)
  function getShippingWithDesi(salePrice, carrier, desi) {
    return TrendyolCalculator.getShippingCost(salePrice, carrier, desi);
  }

  function processListingPage(pageData) {
    tkLastPageData = pageData;
    if (!tkFloatBtn || !document.body.contains(tkFloatBtn)) {
      createFloatingButton();
    }
    addBadgesToCards();
  }

  function createFloatingButton() {
    if (document.getElementById("tk-float-btn")) return;

    tkFloatBtn = document.createElement("button");
    tkFloatBtn.id = "tk-float-btn";
    tkFloatBtn.className = "tk-float-btn";
    tkFloatBtn.innerHTML = '<svg viewBox="0 0 100 100" width="28" height="28"><circle cx="50" cy="50" r="48" fill="#FF6B00"/><circle cx="50" cy="50" r="34" fill="#0f172a"/><path d="M50 20 A30 30 0 0 1 80 50" stroke="#3b82f6" stroke-width="8" stroke-linecap="round" fill="none"/><path d="M80 50 A30 30 0 0 1 50 80" stroke="#2563eb" stroke-width="8" stroke-linecap="round" fill="none"/><path d="M50 80 A30 30 0 0 1 20 50" stroke="#3b82f6" stroke-width="8" stroke-linecap="round" fill="none"/><path d="M20 50 A30 30 0 0 1 50 20" stroke="#2563eb" stroke-width="8" stroke-linecap="round" fill="none"/><circle cx="50" cy="50" r="8" fill="#3b82f6"/></svg>';
    tkFloatBtn.title = "TPro360 Satış Analizi";
    tkFloatBtn.addEventListener("click", () => startTableWithAutoScroll());
    document.body.appendChild(tkFloatBtn);
  }

  // ─── Basit kart sayacı (hızlı, fiyat kontrolü YOK — sadece sayım için) ───
  function countProductCards() {
    // Yeni yapı
    let cards = document.querySelectorAll("div.search-result-content > a.product-card");
    if (cards.length > 0) return cards.length;
    // Eski yapı
    cards = document.querySelectorAll("div.p-card-wrppr");
    if (cards.length > 0) return cards.length;
    // Genel fallback
    cards = document.querySelectorAll("a.product-card");
    return cards.length;
  }

  // ─── Benzersiz ürün kartlarını bul (scrape için kullanılır) ───
  function findProductCards() {
    // Kademeli selector stratejisi — en spesifikten en genele
    const selectors = [
      "div.search-result-content > a.product-card",  // Yeni Trendyol yapısı (2024+)
      "div.p-card-wrppr",                             // Eski Trendyol wrapper'ı
      "a.product-card",                               // Genel product-card fallback
    ];

    let rawCards = [];
    for (const sel of selectors) {
      const found = document.querySelectorAll(sel);
      if (found.length > 0) {
        rawCards = Array.from(found);
        break;
      }
    }

    // Son çare: genel p-card seçici
    if (rawCards.length === 0) {
      rawCards = Array.from(document.querySelectorAll("[class*='p-card']"));
    }

    // href bazlı deduplication (fiyat filtresi burada YAPILMAZ — scrape aşamasında yapılır)
    const seen = new Set();
    const uniqueCards = [];
    for (const card of rawCards) {
      // Duplicate kontrolü: href bazlı
      let link = card.href || "";
      if (!link) {
        const linkEl = card.querySelector("a[href]");
        link = linkEl ? linkEl.href : "";
      }

      // İç içe kartları atla
      let isNested = false;
      for (const existing of uniqueCards) {
        if (existing.contains(card) || card.contains(existing)) {
          isNested = true;
          break;
        }
      }
      if (isNested) continue;

      // Link bazlı deduplication
      const dedupeKey = link || (card.textContent || "").substring(0, 80).trim();
      if (seen.has(dedupeKey)) continue;
      seen.add(dedupeKey);

      uniqueCards.push(card);
    }

    return uniqueCards;
  }

  // ─── Yardımcı: Belirli ms kadar bekle ───
  function sleep(ms) {
    return new Promise(r => setTimeout(r, ms));
  }

  // ─── Yardımcı: DOM'daki kart sayısının artmasını bekle ───
  function waitForMoreCards(previousCount, timeoutMs) {
    timeoutMs = timeoutMs || 7000;
    return new Promise(resolve => {
      // Hemen kontrol et
      if (countProductCards() > previousCount) {
        // Batch yüklemesi bitsin diye 1sn stabilite bekle
        setTimeout(() => resolve(countProductCards()), 1000);
        return;
      }

      let resolved = false;
      let debounceTimer = null;

      const observer = new MutationObserver(() => {
        if (resolved) return;
        const nowCount = countProductCards();
        if (nowCount > previousCount) {
          // Batch tamamlansın — 1.2sn sessizlik bekle
          clearTimeout(debounceTimer);
          debounceTimer = setTimeout(() => {
            if (resolved) return;
            resolved = true;
            observer.disconnect();
            resolve(countProductCards());
          }, 1200);
        }
      });

      observer.observe(document.body, { childList: true, subtree: true });

      setTimeout(() => {
        if (!resolved) {
          resolved = true;
          clearTimeout(debounceTimer);
          observer.disconnect();
          resolve(countProductCards());
        }
      }, timeoutMs);
    });
  }

  // ─── Ana fonksiyon: Otomatik scroll ile ürünleri yükle (max 50) ───
  async function startTableWithAutoScroll() {
    if (tkOverlay && document.body.contains(tkOverlay)) {
      tkOverlay.remove();
      tkOverlay = null;
      return;
    }

    // Loading göster — pointer-events:none sayesinde sayfa scroll'u engellenmez
    const loadingDiv = document.createElement("div");
    loadingDiv.id = "tk-loading";
    loadingDiv.style.cssText = "position:fixed;top:0;left:0;right:0;bottom:0;z-index:999999;background:rgba(0,0,0,0.85);display:flex;align-items:center;justify-content:center;flex-direction:column;pointer-events:none;font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif;";
    loadingDiv.innerHTML = `
      <div class="tk-loading-spinner" style="width:48px;height:48px;border-width:5px;margin:0 auto 20px;"></div>
      <h2 style="color:#f27a1a;font-size:22px;margin:0 0 8px 0;">Ürünler taranıyor...</h2>
      <p style="color:#aaa;font-size:14px;" id="tk-scroll-count">Sayfa aşağı kaydırılarak daha fazla ürün yükleniyor</p>
    `;
    document.body.appendChild(loadingDiv);

    const MAX_PRODUCTS = 200;
    const MAX_ROUNDS = 40;
    let failedRounds = 0;
    const MAX_FAILS = 4;

    // ── 1) Sayfanın başına git ──
    window.scrollTo({ top: 0, behavior: "instant" });
    await sleep(500);

    for (let round = 0; round < MAX_ROUNDS; round++) {
      const beforeCount = countProductCards();
      const statusEl = document.getElementById("tk-scroll-count");

      if (beforeCount >= MAX_PRODUCTS) {
        if (statusEl) statusEl.textContent = `${beforeCount} ürün bulundu ✓`;
        break;
      }

      if (statusEl) statusEl.textContent = `${beforeCount} ürün bulundu, daha fazlası yükleniyor… (sayfa ${round + 1})`;

      // ── 2) Sayfanın en altına kadar küçük adımlarla scroll et ──
      //    Sentinel'i kesin viewport'a sokmak için alt kısma doğru adım adım in
      const startY = window.scrollY;
      const maxY = document.body.scrollHeight;
      const step = Math.round(window.innerHeight * 0.7);
      let scrolled = false;

      for (let y = startY; y < maxY; y += step) {
        window.scrollTo({ top: y, behavior: "instant" });

        // Her adımda kart sayısı arttı mı diye bak
        await sleep(200);
        if (countProductCards() > beforeCount) {
          scrolled = true;
          break;
        }
      }

      // Eğer adımlar sırasında tetiklenmediyse en alta git
      if (!scrolled) {
        window.scrollTo({ top: document.body.scrollHeight, behavior: "instant" });
        await sleep(300);
      }

      // ── 3) Yeni kartların yüklenmesini bekle (max 7sn) ──
      const afterCount = await waitForMoreCards(beforeCount, 7000);

      // ── 4) Sonuç kontrolü ──
      if (afterCount > beforeCount) {
        failedRounds = 0;
        // Kartların tam render olması için ek bekleme
        await sleep(800);
      } else {
        failedRounds++;

        // Ek deneme: sayfanın en altına scroll + biraz bekle
        if (failedRounds <= 2) {
          window.scrollTo({ top: document.body.scrollHeight, behavior: "instant" });
          await sleep(2000);

          // Sentinel'e scrollIntoView dene (varsa)
          const sentinel = document.querySelector('[data-testid="infinite-scroll-div-2"]');
          if (sentinel) {
            sentinel.scrollIntoView({ behavior: "instant", block: "center" });
            await sleep(2000);
          }

          const retryCount = countProductCards();
          if (retryCount > beforeCount) {
            failedRounds = 0;
            await sleep(800);
            continue;
          }
        }

        if (failedRounds >= MAX_FAILS) {
          if (statusEl) statusEl.textContent = `${countProductCards()} ürün bulundu (sayfa sonu)`;
          break;
        }
      }
    }

    // ── 5) Scroll'u en üste geri al ──
    window.scrollTo({ top: 0, behavior: "instant" });
    await sleep(500);

    // Loading kaldır
    loadingDiv.remove();

    // Scrape öncesi DOM'daki gerçek kart sayısını logla
    const finalDOMCount = countProductCards();
    const finalFindCount = findProductCards().length;
    tkLog(`Scroll tamamlandı — DOM'da ${finalDOMCount} kart, findProductCards: ${finalFindCount} kart`);

    const products = scrapeAllProducts(tkLastPageData);
    if (products.length === 0) {
      alert("Bu sayfada ürün kartı bulunamadı!\nTrendyol ürün listesi veya arama sayfasında deneyin.");
      return;
    }

    tkCurrentProducts = products;
    showFullTable(products);

    // Arka planda T1 verisini çek: socialProof'u olmayan ürünler için
    enrichProductsWithT1(products);
  }

  // ═══════════════════════════════════════════════════════════════
  //  ARKA PLANDA T1 VERİ ZENGİNLEŞTİRME
  //  socialProof verisi olmayan ürünler için background API'den çeker
  //  ve tabloyu yerinde günceller.
  // ═══════════════════════════════════════════════════════════════
  async function enrichProductsWithT1(products) {
    // URL'den ürün ID'si çıkarılabilen tüm ürünleri al
    const productIdMap = {}; // pid → product index
    products.forEach((p, idx) => {
      const match = (p.url || p.link) && (p.url || p.link).match(/-p-(\d+)/);
      if (match) productIdMap[match[1]] = idx;
    });

    const productIds = Object.keys(productIdMap);
    if (productIds.length === 0) {
      tkLog("Enrichment: URL'den ID çıkarılabilecek ürün yok");
      return;
    }

    const breadcrumbs = getBreadcrumbsFromURL();
    const categoryPath = (breadcrumbs || []).join(" > ");
    let totalUpdated = 0;
    let totalStockUpdated = 0;

    tkLog(`Enrichment başlıyor: ${productIds.length} ürün`);

    // ─── Loading bar göster ───
    const loadingBar = document.createElement("div");
    loadingBar.id = "tk-enrich-loading";
    loadingBar.innerHTML = `
      <div class="tk-loading-icon">⏳</div>
      <div class="tk-loading-text">
        <span id="tk-loading-label">Satış verileri ve stok bilgileri yükleniyor...</span>
        <div class="tk-loading-progress">
          <div class="tk-loading-fill" id="tk-loading-fill" style="width:0%"></div>
        </div>
        <span class="tk-loading-detail" id="tk-loading-detail">0 / ${productIds.length} ürün</span>
      </div>
    `;
    const tableScroll = tkOverlay && tkOverlay.querySelector(".tk-table-scroll");
    if (tableScroll) tableScroll.parentNode.insertBefore(loadingBar, tableScroll);

    function updateLoadingBar(loaded, total, phase) {
      const pct = Math.round((loaded / total) * 100);
      const fill = document.getElementById("tk-loading-fill");
      const detail = document.getElementById("tk-loading-detail");
      const label = document.getElementById("tk-loading-label");
      if (fill) fill.style.width = pct + "%";
      if (detail) detail.textContent = `${loaded} / ${total} ürün`;
      if (label && phase) label.textContent = phase;
    }

    // ─── Adım 0: Panel tahminleri çek (takip edilen ürünler için T0+T1+T3 blend) ───
    // Panel endpoint max 50 ID kabul eder — batch'le gönder
    updateLoadingBar(0, productIds.length, "Panel verileri kontrol ediliyor...");
    let panelEstimateCount = 0;
    const PE_BATCH = 50;
    for (let peStart = 0; peStart < productIds.length; peStart += PE_BATCH) {
      const peBatch = productIds.slice(peStart, peStart + PE_BATCH);
      try {
        const peResponse = await new Promise((resolve) => {
          chrome.runtime.sendMessage(
            { action: "FETCH_PANEL_ESTIMATE", productIds: peBatch },
            (resp) => resolve(resp)
          );
        });
        if (peResponse && peResponse.status === "ok" && peResponse.estimates) {
          Object.entries(peResponse.estimates).forEach(([pid, est]) => {
            const idx = productIdMap[pid];
            if (idx === undefined) return;
            const p = products[idx];
            if (!p || !est) return;

            const monthly = est.monthly || 0;
            if (monthly > 0 && monthly > p.estSales) {
              p.estSales = monthly;
              p.origEstSales = monthly;
              p.salesTier = est.tier || "Panel";
              p.salesTierLabel = est.tierLabel || "Panel Tahmini";
              p.monthlyProfit = p.profit * monthly;
              if (est.min > 0) p.estSalesMin = est.min;
              if (est.max > 0) p.estSalesMax = est.max;
              panelEstimateCount++;
              totalUpdated++;
            }
          });
        }
      } catch (e) {
        tkLog("Panel estimate batch hatası:", e.message);
      }
    }
    if (panelEstimateCount > 0) {
      tkLog(`Panel estimate: ${panelEstimateCount} takip edilen ürün güncellendi`);
      tkCurrentProducts = products;
      renderTableRows(products);
    }

    // ─── Adım 1: SocialProof çek + panel ile tutarlı satış tahmini ───
    // 1a) apigw'den socialProof çek (ORDER_COUNT, BASKET_COUNT, FAVORITE_COUNT)
    // 1b) socialProof'u panel'e gönder → computePanelEstimate (research ile aynı hesaplama)
    // 1c) Panel erişilemezse yerel TrendyolCalculator fallback
    updateLoadingBar(0, productIds.length, "Satış verileri yükleniyor...");
    const productsForPanelEstimate = []; // Panel'e gönderilecek ürünler
    try {
      const spResponse = await new Promise((resolve) => {
        chrome.runtime.sendMessage(
          { action: "BULK_FETCH_SOCIALPROOF", productIds },
          (resp) => resolve(resp)
        );
      });

      if (spResponse && spResponse.results) {
        const results = spResponse.results;
        let spCount = 0;
        Object.entries(results).forEach(([pid, spData]) => {
          spCount++;
          const idx = productIdMap[pid];
          if (idx === undefined) return;
          const p = products[idx];
          if (!p) return;

          // Adım 0'da panel estimate zaten geldiyse bu ürünü atla
          if (p.salesTier && p.salesTier !== "T4" && p.salesTier !== "-") return;

          // spData SCREAMING_SNAKE: ORDER_COUNT, BASKET_COUNT, FAVORITE_COUNT
          const hasSignal = spData.ORDER_COUNT || spData.BASKET_COUNT || spData.FAVORITE_COUNT;
          if (!hasSignal) return;

          // SocialProof'u ürüne sakla (panel estimate + fallback için)
          p._socialProof = spData;

          // Panel estimate için ürün bilgilerini topla
          productsForPanelEstimate.push({
            id: pid,
            socialProof: {
              ORDER_COUNT: spData.ORDER_COUNT || null,
              BASKET_COUNT: spData.BASKET_COUNT || null,
              FAVORITE_COUNT: spData.FAVORITE_COUNT || null,
            },
            price: p.price || 0,
            reviewCount: p.reviewCount || 0,
            rating: p.rating || 0,
            categoryPath: categoryPath || "",
            favoriteCount: p.favoriteCount || 0,
          });
        });

        updateLoadingBar(spCount, productIds.length, "Satış tahmini hesaplanıyor...");
        tkLog(`SocialProof: ${spCount} ürün çekildi, ${productsForPanelEstimate.length} panel'e gönderilecek`);
      }
    } catch (e) {
      tkLog("SocialProof fetch hatası:", e.message);
    }

    // ─── Adım 1b: Panel'den tutarlı satış tahmini al ───
    // Research/scanner ile aynı computePanelEstimate hesaplama zinciri
    let panelEstimateFailed = false;
    if (productsForPanelEstimate.length > 0) {
      const EST_BATCH = 50;
      try {
        for (let estStart = 0; estStart < productsForPanelEstimate.length; estStart += EST_BATCH) {
          const estBatch = productsForPanelEstimate.slice(estStart, estStart + EST_BATCH);
          const estResponse = await new Promise((resolve) => {
            chrome.runtime.sendMessage(
              { action: "BULK_ESTIMATE_SALES", products: estBatch },
              (resp) => resolve(resp)
            );
          });

          if (estResponse && estResponse.status === "ok" && estResponse.estimates) {
            Object.entries(estResponse.estimates).forEach(([pid, est]) => {
              const idx = productIdMap[pid];
              if (idx === undefined) return;
              const p = products[idx];
              if (!p || !est) return;

              const monthly = est.monthly || 0;
              if (monthly > 0) {
                p.estSales = monthly;
                p.origEstSales = monthly;
                p.salesTier = est.tier || "Panel";
                p.salesTierLabel = est.tierLabel || "Panel Tahmini";
                p.monthlyProfit = p.profit * monthly;
                if (est.min > 0) p.estSalesMin = est.min;
                if (est.max > 0) p.estSalesMax = est.max;
                totalUpdated++;
              }
            });
          } else {
            panelEstimateFailed = true;
          }
        }
      } catch (e) {
        tkLog("Panel estimate hatası:", e.message);
        panelEstimateFailed = true;
      }

      // ─── 1c: Panel erişilemezse yerel TrendyolCalculator fallback ───
      if (panelEstimateFailed) {
        tkLog("Panel erişilemedi — yerel hesaplama ile devam ediliyor");
        productsForPanelEstimate.forEach(({ id: pid }) => {
          const idx = productIdMap[pid];
          if (idx === undefined) return;
          const p = products[idx];
          if (!p || !p._socialProof) return;
          // Zaten panel'den güncellendiyse atla
          if (p.salesTier && p.salesTier !== "T4" && p.salesTier !== "-") return;

          const spData = p._socialProof;
          const socialProofData = {
            orderCount: spData.ORDER_COUNT || null,
            basketCount: spData.BASKET_COUNT || null,
            favoriteCount: spData.FAVORITE_COUNT || null,
            basketConversionRate: panelCalibration.basket_to_sales_rate || 0.10,
            categoryPath: categoryPath
          };

          const salesResult = TrendyolCalculator.estimateMonthlySalesDetailed(
            p.reviewCount || 0, p.rating || 0, p.price, p.badges || [], null, socialProofData
          );
          if (salesResult.count > 0 && (salesResult.tier !== "T4" || salesResult.count > p.estSales)) {
            p.estSales = salesResult.count;
            p.origEstSales = salesResult.count;
            p.salesTier = salesResult.tier;
            p.salesTierLabel = salesResult.tierLabel;
            p.monthlyProfit = p.profit * p.estSales;
            totalUpdated++;
          }
        });
      }

      updateLoadingBar(productsForPanelEstimate.length, productIds.length,
        `Satış tahmini tamamlandı (${totalUpdated} güncellendi). Stok bilgileri yükleniyor...`);
      tkLog(`Panel estimate enrichment: ${totalUpdated}/${productIds.length} ürün güncellendi`);

      if (totalUpdated > 0) {
        tkCurrentProducts = products;
        renderTableRows(products);
      }
    }

    // ─── Adım 2: Stok + ek socialProof çek (detay sayfası SSR — kademeli) ───
    const allRequests = productIds.map(pid => {
      const idx = productIdMap[pid];
      const p = products[idx];
      const productUrl = p.url || p.link || "";
      try {
        const urlObj = new URL(productUrl, "https://www.trendyol.com");
        return { id: pid, slug: urlObj.pathname };
      } catch (e) {
        return { id: pid, slug: productUrl };
      }
    });

    const BATCH_SIZE = 25;
    let processed = 0;
    const step2PanelQueue = []; // Adım 2'de yeni socialProof alan ürünler

    for (let batchStart = 0; batchStart < allRequests.length; batchStart += BATCH_SIZE) {
      const batchRequests = allRequests.slice(batchStart, batchStart + BATCH_SIZE);

      updateLoadingBar(processed, allRequests.length, `Stok bilgileri yükleniyor...`);

      try {
        const bgResponse = await new Promise((resolve) => {
          chrome.runtime.sendMessage(
            { action: "FETCH_PRODUCT_DETAILS", products: batchRequests, fetchStock: true },
            (resp) => resolve(resp)
          );
        });

        if (!bgResponse || !bgResponse.results) {
          processed += batchRequests.length;
          continue;
        }

        const results = bgResponse.results;
        let batchStockUpdated = 0;

        products.forEach((p) => {
          const match = (p.url || p.link) && (p.url || p.link).match(/-p-(\d+)/);
          if (!match) return;
          const pid = match[1];
          const apiData = results[pid];
          if (!apiData) return;

          // Stok bilgisi ekle
          if (apiData.stockQuantity > 0 && !p.stockQty) {
            p.stockQty = apiData.stockQuantity;
            batchStockUpdated++;
          }

          // Adım 1'de enrichment yapılmadıysa burada socialProof topla (panel'e gönderilecek)
          if (p.salesTier === "T4" || !p.salesTier || p.salesTier === "-") {
            const sp = apiData.socialProof;
            if (sp) {
              const hasSignal = sp.orderCount || sp.basketCount || sp.favoriteCount;
              if (hasSignal) {
                // Panel estimate için kuyruğa ekle
                step2PanelQueue.push({
                  id: pid,
                  socialProof: {
                    ORDER_COUNT: sp.orderCount || null,
                    BASKET_COUNT: sp.basketCount || null,
                    FAVORITE_COUNT: sp.favoriteCount || null,
                  },
                  price: p.price || 0,
                  reviewCount: p.reviewCount || 0,
                  rating: p.rating || 0,
                  categoryPath: categoryPath || "",
                  favoriteCount: p.favoriteCount || 0,
                });
              }
            }
          }
        });

        totalStockUpdated += batchStockUpdated;
        processed += batchRequests.length;
        updateLoadingBar(processed, allRequests.length, `Stok bilgileri yükleniyor...`);

        if (batchStockUpdated > 0) {
          tkCurrentProducts = products;
          renderTableRows(products);
        }
      } catch (e) {
        processed += batchRequests.length;
        tkLog(`Stok enrichment batch hatası:`, e.message);
      }
    }

    // ─── Adım 2b: Detay SSR'dan gelen yeni socialProof'ları panel'e gönder ───
    if (step2PanelQueue.length > 0) {
      updateLoadingBar(processed, allRequests.length, "Ek satış tahminleri hesaplanıyor...");
      const EST_BATCH2 = 50;
      try {
        for (let estStart = 0; estStart < step2PanelQueue.length; estStart += EST_BATCH2) {
          const estBatch = step2PanelQueue.slice(estStart, estStart + EST_BATCH2);
          const estResponse = await new Promise((resolve) => {
            chrome.runtime.sendMessage(
              { action: "BULK_ESTIMATE_SALES", products: estBatch },
              (resp) => resolve(resp)
            );
          });
          if (estResponse && estResponse.status === "ok" && estResponse.estimates) {
            Object.entries(estResponse.estimates).forEach(([pid, est]) => {
              const idx = productIdMap[pid];
              if (idx === undefined) return;
              const p = products[idx];
              if (!p || !est) return;
              const monthly = est.monthly || 0;
              if (monthly > 0) {
                p.estSales = monthly;
                p.origEstSales = monthly;
                p.salesTier = est.tier || "Panel";
                p.salesTierLabel = est.tierLabel || "Panel Tahmini";
                p.monthlyProfit = p.profit * monthly;
                if (est.min > 0) p.estSalesMin = est.min;
                if (est.max > 0) p.estSalesMax = est.max;
                totalUpdated++;
              }
            });
          }
        }
      } catch (e) {
        tkLog("Adım 2 panel estimate hatası:", e.message);
        // Fallback: yerel hesaplama
        step2PanelQueue.forEach(({ id: pid, socialProof: sp }) => {
          const idx = productIdMap[pid];
          if (idx === undefined) return;
          const p = products[idx];
          if (!p) return;
          if (p.salesTier && p.salesTier !== "T4" && p.salesTier !== "-") return;
          const socialProofData = {
            orderCount: sp.ORDER_COUNT || null,
            basketCount: sp.BASKET_COUNT || null,
            favoriteCount: sp.FAVORITE_COUNT || null,
            basketConversionRate: panelCalibration.basket_to_sales_rate || 0.10,
            categoryPath: categoryPath
          };
          const salesResult = TrendyolCalculator.estimateMonthlySalesDetailed(
            p.reviewCount || 0, p.rating || 0, p.price, p.badges || [], null, socialProofData
          );
          if (salesResult.count > 0 && salesResult.tier !== "T4") {
            p.estSales = salesResult.count;
            p.origEstSales = salesResult.count;
            p.salesTier = salesResult.tier;
            p.salesTierLabel = salesResult.tierLabel;
            p.monthlyProfit = p.profit * p.estSales;
            totalUpdated++;
          }
        });
      }

      if (totalUpdated > 0) {
        tkCurrentProducts = products;
        renderTableRows(products);
      }
      tkLog(`Adım 2 panel estimate: ${step2PanelQueue.length} ürün, toplam ${totalUpdated} güncellendi`);
    }

    // ─── Tamamlandı — loading bar'ı güncelle ───
    if (loadingBar) {
      const extras = [];
      if (panelEstimateCount > 0) extras.push(`${panelEstimateCount} takip edilen`);
      if (totalUpdated - panelEstimateCount > 0) extras.push(`${totalUpdated - panelEstimateCount} satış tahmini`);
      if (totalStockUpdated > 0) extras.push(`${totalStockUpdated} stok`);
      if (extras.length > 0) {
        loadingBar.innerHTML = `
          <div class="tk-loading-icon">✅</div>
          <div class="tk-loading-text">
            <span>${extras.join(" + ")} güncellendi</span>
          </div>
        `;
        loadingBar.classList.add("tk-loading-done");
        setTimeout(() => { if (loadingBar.parentNode) loadingBar.remove(); }, 4000);
      } else {
        loadingBar.innerHTML = `
          <div class="tk-loading-icon">ℹ️</div>
          <div class="tk-loading-text">
            <span>Ek satış/stok verisi bulunamadı — sayfa verileri kullanılıyor</span>
          </div>
        `;
        loadingBar.classList.add("tk-loading-info");
        setTimeout(() => { if (loadingBar.parentNode) loadingBar.remove(); }, 5000);
      }
    }

    tkLog(`Enrichment tamamlandı: ${totalUpdated} satış + ${totalStockUpdated} stok / ${productIds.length} ürün`);
  }

  // ═══════════════════════════════════════════════════════════════
  //  Trendyol Hidden Data — SocialProof Çekici
  // ═══════════════════════════════════════════════════════════════
  //  Arama/listeleme sayfalarında window['__single-search-result__PROPS']
  //  içinde her ürün için socialProof verisi bulunur:
  //    orderCount: "800+" (son dönem sipariş)
  //    basketCount: "3K"  (anlık sepet)
  //    favoriteCount: "102K" (toplam favori)
  //    pageViewCount: "899"  (görüntülenme)
  //
  //  Bu veri, product card'ın DOM id'si ile eşleştirilir.
  // ═══════════════════════════════════════════════════════════════
  function buildSocialProofMap(pageData) {
    const map = {};
    try {
      const products = pageData && pageData.listingProducts;
      if (products && Array.isArray(products)) {
        products.forEach(p => {
          if (p.id && p.socialProof && Array.isArray(p.socialProof)) {
            const spObj = {};
            p.socialProof.forEach(sp => {
              spObj[sp.key] = sp.value;
            });
            // ratingScore verisini de ekle (tam sayılar)
            if (p.ratingScore) {
              spObj._ratingTotal = p.ratingScore.totalCount || 0;
              spObj._ratingAvg = p.ratingScore.averageRating || 0;
            }
            map[String(p.id)] = spObj;
          }
        });
        tkLog(`SocialProof map oluşturuldu: ${Object.keys(map).length} ürün`);
      }
    } catch (e) {
      tkLog("SocialProof map oluşturulamadı:", e.message);
    }
    return map;
  }

  function scrapeAllProducts(pageData) {
    const cards = findProductCards();

    tkLog("scrapeAllProducts başladı — findProductCards:", cards.length, "kart");

    // SocialProof map'i oluştur (Trendyol hidden data — inject edilmiş)
    const socialProofMap = buildSocialProofMap(pageData);

    const products = [];
    const breadcrumbs = getBreadcrumbsFromURL();
    const HIZMET_BEDELI = 10.20;
    const kdvRate = tkManualOverrides.kdvRate !== null ? tkManualOverrides.kdvRate / 100 : (settings.kdvRate || 20) / 100;
    let skippedNoPrice = 0;
    // (skippedParseFail kaldırıldı — fiyat bulunamayan kartlar skippedNoPrice ile sayılır)

    cards.forEach((card, idx) => {
      if (products.length >= 200) return; // Net 200 ürün limiti

      // ── Fiyat: geniş selector ile bul ──
      // Öncelik: en spesifik ve doğru fiyatı veren element
      let priceEl = null;
      let price = 0;

      // Strateji 1: Bilinen class'lar (en güvenilir → en genel sırasıyla)
      const priceSelectors = [
        ".price-value",          // Yaprak fiyat elementi (yeni yapı 2025+, en spesifik)
        ".discounted-price",     // İndirimli fiyat (yeni yapı 2025+)
        ".sale-price",           // İndirimli fiyat (bazı kartlarda)
        ".price-section",        // Ana fiyat (yeni yapı 2025+)
        ".single-price",         // Tekil fiyat (yeni yapı 2025+)
        ".prc-box-dscntd",       // İndirimli fiyat (eski yapı)
        ".prc-box-sllng",        // Satış fiyatı (eski yapı)
        "[class*='prc-box']",    // Genel fiyat kutusu (eski)
        "[class*='product-price']"
      ];

      for (const sel of priceSelectors) {
        const el = card.querySelector(sel);
        if (el) {
          const parsed = TrendyolCalculator.parsePrice(el.textContent);
          if (parsed > 0) {
            priceEl = el;
            price = parsed;
            break;
          }
        }
      }

      // Strateji 2: "TL" içeren herhangi bir yaprak element
      if (!price) {
        const allEls = card.querySelectorAll("span, div");
        for (const el of allEls) {
          const text = el.textContent.trim();
          if (/\d/.test(text) && /TL/i.test(text) && text.length < 50) {
            const parsed = TrendyolCalculator.parsePrice(text);
            if (parsed > 0) {
              priceEl = el;
              price = parsed;
              break;
            }
          }
        }
      }

      if (!price) {
        skippedNoPrice++;
        if (idx < 30) tkLog(`Kart ${idx} ATLANDI — fiyat bulunamadı. HTML:`, card.outerHTML.substring(0, 200));
        return;
      }

      // ── Marka + İsim ──
      const brandEl = card.querySelector(".product-brand") ||
        card.querySelector(".prdct-desc-cntnr-ttl") ||
        card.querySelector("[class*='brand-name']") ||
        card.querySelector("[data-testid='brand-name-wrapper'] span");
      const nameEl = card.querySelector(".product-name") ||
        card.querySelector(".prdct-desc-cntnr-name") ||
        card.querySelector(".title") ||
        card.querySelector("h2.title") ||
        card.querySelector("[class*='product-desc']");
      const brand = brandEl ? brandEl.textContent.trim() : "";
      const name = nameEl ? nameEl.textContent.trim() : "";
      const fullName = (brand ? brand + " " : "") + (name || "Ürün " + (idx + 1));

      // ── Yorum sayısı ──
      const rcEl = card.querySelector(".total-count") ||
        card.querySelector("[class*='ratingCount']") ||
        card.querySelector("[class*='review-count']") ||
        card.querySelector("[class*='rating-count']");
      let reviewCount = 0;
      if (rcEl) {
        const m = rcEl.textContent.match(/[\d.]+/);
        reviewCount = m ? parseInt(m[0].replace(/\./g, "")) : 0;
      }

      // ── Puan ──
      const ratEl = card.querySelector(".average-rating") ||
        card.querySelector("[class*='ratingScore']") ||
        card.querySelector("[class*='rating-score']");
      const rating = ratEl ? parseFloat(ratEl.textContent) : 0;

      // ── Link ──
      let link = card.href || "";
      if (!link) {
        const linkEl = card.querySelector("a[href]");
        link = linkEl ? linkEl.href : "";
      }

      // Hesapla
      const commInfo = TrendyolCalculator.findCommission(breadcrumbs);
      const commRate = commInfo.komisyon;
      const commAmount = price * (commRate / 100);

      // Kargo: manuel override > desi override > otomatik
      // Tablo rakamları zaten KDV hariç — doğrudan base kullan, KDV ekleme/düşme yok
      let shipCost; // KDV hariç kargo tutarı
      let shipTier;
      if (tkManualOverrides.kargo !== null) {
        shipCost = tkManualOverrides.kargo;
        shipTier = "Manuel";
      } else {
        const desiVal = tkManualOverrides.desi || undefined;
        const shipping = getShippingWithDesi(price, settings.carrier, desiVal);
        shipCost = shipping.base || 0;
        shipTier = shipping.tier || "";
      }

      // Rozetleri topla (Listing sayfasında)
      const badges = [];
      card.querySelectorAll(".stamps .promotion-stamp, .campaign-name, .stamp-container, .active-campaign-wrapper").forEach(el => {
        badges.push(el.textContent.trim());
      });

      // SocialProof: Ürün ID'si ile Trendyol hidden data eşleştir
      // Öncelik: 1) URL'den çıkarılan content ID, 2) data-id, 3) card.id
      const cardId = card.id || card.getAttribute("data-id") || "";
      const urlContentId = link && link.match(/-p-(\d+)/) ? link.match(/-p-(\d+)/)[1] : "";
      const dataContentId = card.getAttribute("data-contentid") || card.getAttribute("data-content-id") || "";
      const socialProofData = socialProofMap[urlContentId] || socialProofMap[dataContentId] || socialProofMap[cardId] || null;
      if (socialProofData) {
        tkLog(`Kart ${idx} socialProof bulundu (ID:${urlContentId||cardId}): orderCount=${socialProofData.orderCount || "-"}, basketCount=${socialProofData.basketCount || "-"}`);
      }

      // Satış adedi: manuel override veya akıllı tahmin (socialProof v4)
      let estSales, salesTier, salesTierLabel;
      if (tkManualOverrides.satisAdedi !== null) {
        estSales = tkManualOverrides.satisAdedi;
        salesTier = "Manuel";
        salesTierLabel = "Manuel";
      } else {
        // Kalibrasyon oranını + kategori yolunu inject et
        const spWithCal = socialProofData
          ? { ...socialProofData, basketConversionRate: panelCalibration.basket_to_sales_rate || 0.10, categoryPath: (breadcrumbs || []).join(' > ') }
          : null;
        const salesResult = TrendyolCalculator.estimateMonthlySalesDetailed(reviewCount, rating, price, badges, null, spWithCal);
        estSales = salesResult.count;
        salesTier = salesResult.tier;
        salesTierLabel = salesResult.tierLabel;
      }

      // Karlılık hesabı: tüm kalemlerden KDV çıkar
      // Komisyon KDV dahil fiyat üzerinden kesilir (Trendyol kuralı) — commAmount doğru
      const salePriceExVat = price / (1 + kdvRate);
      const commExVat = commAmount / (1 + kdvRate);
      const hizmetExVat = HIZMET_BEDELI / (1 + kdvRate);
      // Kargo zaten KDV hariç — doğrudan kullan
      const kalanKDVHaric = salePriceExVat - commExVat - hizmetExVat - shipCost;
      const kalanKDVDahil = price - commAmount - HIZMET_BEDELI - shipCost;
      const kdvAmount = price - salePriceExVat;

      // Maliyet: manuel override veya tahmini (%20 kâr bırakan)
      const usedCost = tkManualOverrides.maliyet !== null ? tkManualOverrides.maliyet : kalanKDVHaric * 0.80;
      const profit = kalanKDVHaric - usedCost;
      const monthlyProfit = profit * estSales;

      // Stok bilgisi (arama sayfasında genellikle yok, ama map'te varsa al)
      const stockQty = socialProofData && socialProofData._stockQuantity ? socialProofData._stockQuantity : null;

      products.push({
        idx: idx + 1,
        brand: brand || "-",
        name: fullName.substring(0, 65),
        price,
        rating,
        reviewCount,
        link,
        url: link,
        commRate,
        commAmount,
        commMatch: commInfo.match,
        shipCost,
        shipTier,
        kalanKDVDahil,
        kdvAmount,
        kalanKDVHaric,
        autoEstCost: usedCost,
        profit,
        estSales,
        salesTier,
        salesTierLabel,
        stockQty,
        monthlyProfit,
        // Orijinal değerler (recalc için)
        origShipCost: shipCost,
        origShipTier: shipTier,
        origEstSales: estSales,
        origAutoEstCost: usedCost,
        // Satır bazlı override
        rowOverrides: { kargo: null, kdvRate: null, maliyet: null, satisAdedi: null }
      });
    });

    tkLog(`scrapeAllProducts bitti — ${products.length} ürün çekildi, ${skippedNoPrice} fiyat elem yok`);
    return products;
  }

  // Excel XLS Export (HTML table → .xls, Türkçe şablon)
  function exportToXLS(products) {
    const fp = TrendyolCalculator.formatPrice;
    const dateStr = new Date().toLocaleDateString("tr-TR");

    const xml = `
<html xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:x="urn:schemas-microsoft-com:office:excel" xmlns="http://www.w3.org/TR/REC-html40">
<head>
<meta charset="UTF-8">
<!--[if gte mso 9]><xml><x:ExcelWorkbook><x:ExcelWorksheets><x:ExcelWorksheet>
<x:Name>TPro360 Analiz</x:Name>
<x:WorksheetOptions><x:DisplayGridlines/></x:WorksheetOptions>
</x:ExcelWorksheet></x:ExcelWorksheets></x:ExcelWorkbook></xml><![endif]-->
<style>
  td, th { mso-number-format:"\\@"; font-family: Calibri, sans-serif; }
  .num { mso-number-format:"\\#\\,\\#\\#0\\.00"; }
  .pct { mso-number-format:"\\#\\,\\#\\#0\\.0\\%"; }
  .int { mso-number-format:"\\#\\,\\#\\#0"; }
</style>
</head>
<body>
<table>
  <tr>
    <td colspan="16" style="font-size:18px;font-weight:bold;color:#f27a1a;padding:8px;">TPro360 — Ürün Karlılık Analizi</td>
  </tr>
  <tr>
    <td colspan="16" style="font-size:11px;color:#666;padding:4px 8px 12px;">${dateStr} tarihli analiz — ${products.length} ürün — Kargo: ${settings.carrier}</td>
  </tr>
  <tr style="background:#f27a1a;color:#fff;font-weight:bold;font-size:11px;">
    <th style="padding:8px;text-align:center;">#</th>
    <th style="padding:8px;text-align:left;">Marka</th>
    <th style="padding:8px;text-align:left;">Ürün Adı</th>
    <th style="padding:8px;text-align:right;">Fiyat (₺)</th>
    <th style="padding:8px;text-align:center;">Puan</th>
    <th style="padding:8px;text-align:center;">Yorum</th>
    <th style="padding:8px;text-align:right;">Komisyon %</th>
    <th style="padding:8px;text-align:right;">Komisyon (₺)</th>
    <th style="padding:8px;text-align:right;">Kargo (₺)</th>
    <th style="padding:8px;text-align:right;">Hizmet (₺)</th>
    <th style="padding:8px;text-align:right;">Hesaba Geçecek (₺)</th>
    <th style="padding:8px;text-align:right;">Maks. Maliyet (₺)</th>
    <th style="padding:8px;text-align:right;">Kâr/Satış (₺)</th>
    <th style="padding:8px;text-align:center;">Aylık Satış</th>
    <th style="padding:8px;text-align:center;">Stok</th>
    <th style="padding:8px;text-align:right;">Aylık Kâr (₺)</th>
  </tr>
${products.map((p, i) => {
  const profitColor = p.profit >= 0 ? "#00b365" : "#e53e3e";
  const monthlyColor = p.monthlyProfit >= 0 ? "#00b365" : "#e53e3e";
  const bgColor = i % 2 === 0 ? "#fff" : "#f9fafb";
  return `  <tr style="background:${bgColor};font-size:12px;">
    <td style="text-align:center;color:#999;">${p.idx}</td>
    <td style="font-weight:600;">${(p.brand || "-").replace(/</g,"&lt;")}</td>
    <td>${p.name.replace(/</g,"&lt;")}</td>
    <td class="num" style="text-align:right;font-weight:600;color:#f27a1a;">${p.price.toFixed(2)}</td>
    <td style="text-align:center;">${p.rating > 0 ? p.rating.toFixed(1) : "-"}</td>
    <td class="int" style="text-align:center;">${p.reviewCount || "-"}</td>
    <td class="num" style="text-align:right;">${p.commRate.toFixed(1)}</td>
    <td class="num" style="text-align:right;color:#e53e3e;">${p.commAmount.toFixed(2)}</td>
    <td class="num" style="text-align:right;color:#e53e3e;">${p.shipCost.toFixed(2)}</td>
    <td class="num" style="text-align:right;color:#e53e3e;">10.20</td>
    <td class="num" style="text-align:right;font-weight:600;">${p.kalanKDVHaric.toFixed(2)}</td>
    <td class="num" style="text-align:right;">${p.autoEstCost.toFixed(2)}</td>
    <td class="num" style="text-align:right;font-weight:700;color:${profitColor};">${p.profit.toFixed(2)}</td>
    <td class="int" style="text-align:center;">${p.estSales}</td>
    <td class="int" style="text-align:center;">${p.stockQty || "-"}</td>
    <td class="num" style="text-align:right;font-weight:700;color:${monthlyColor};">${p.monthlyProfit.toFixed(2)}</td>
  </tr>`;
}).join("\n")}
</table>
</body>
</html>`;

    const blob = new Blob([xml], { type: "application/vnd.ms-excel;charset=utf-8" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = "TPro360_Analiz_" + new Date().toISOString().slice(0, 10) + ".xls";
    a.click();
    URL.revokeObjectURL(url);
  }

  function showFullTable(products) {
    if (tkOverlay) tkOverlay.remove();

    tkOverlay = document.createElement("div");
    tkOverlay.id = "tk-overlay";
    tkOverlay.className = "tk-table-overlay";

    const carrierOptions = [
      { value: "TEX-PTT", label: "TEX-PTT (En Ucuz)" },
      { value: "ARAS", label: "Aras Kargo" },
      { value: "SÜRAT", label: "Sürat Kargo" },
      { value: "KOLAY GELSİN", label: "Kolay Gelsin" },
      { value: "DHLeCommerce", label: "DHL eCommerce" },
      { value: "YK", label: "Yurtiçi Kargo" }
    ];
    const carrierHtml = carrierOptions.map(o =>
      `<option value="${o.value}"${settings.carrier === o.value ? " selected" : ""}>${o.label}</option>`
    ).join("");

    const firstMatch = products.length > 0 ? products[0].commMatch : "Varsayılan";

    tkOverlay.innerHTML = `
      <div class="tk-table-container">
        <div class="tk-table-header">
          <div style="display:flex;align-items:center;gap:12px;">
            <svg width="36" height="36" viewBox="0 0 100 100" fill="none"><circle cx="50" cy="50" r="48" fill="#FF6B00"/><circle cx="50" cy="50" r="34" fill="#0f172a"/><path d="M50 20 A30 30 0 0 1 80 50" stroke="#3b82f6" stroke-width="8" stroke-linecap="round" fill="none"/><path d="M80 50 A30 30 0 0 1 50 80" stroke="#2563eb" stroke-width="8" stroke-linecap="round" fill="none"/><path d="M50 80 A30 30 0 0 1 20 50" stroke="#3b82f6" stroke-width="8" stroke-linecap="round" fill="none"/><path d="M20 50 A30 30 0 0 1 50 20" stroke="#2563eb" stroke-width="8" stroke-linecap="round" fill="none"/><circle cx="50" cy="50" r="8" fill="#3b82f6"/></svg>
            <div>
              <h2 class="tk-table-title"><span style="color:#ea580c;font-weight:800;">TPro360</span> Satış Analizi</h2>
              <span class="tk-table-subtitle" id="tk-tbl-subtitle">${products.length} ürün bulundu · Kargo: ${settings.carrier} · Kategori: ${firstMatch}</span>
            </div>
          </div>
          <div class="tk-table-actions">
            <select id="tk-tbl-carrier" class="tk-tbl-select">${carrierHtml}</select>
            <button id="tk-tbl-export" class="tk-tbl-export-btn">📥 Excel İndir</button>
            <button id="tk-tbl-close" class="tk-tbl-close-btn">Kapat (ESC)</button>
          </div>
        </div>

        <div class="tk-table-scroll">
          <table class="tk-full-table">
            <thead>
              <tr>
                <th class="tk-th-center tk-th-sortable" data-sort="idx" data-sort-type="num">#</th>
                <th class="tk-th-left tk-th-sortable" data-sort="brand" data-sort-type="str">Marka ↕</th>
                <th class="tk-th-left tk-th-name tk-th-sortable" data-sort="name" data-sort-type="str">Ürün Adı ↕</th>
                <th class="tk-th-right tk-th-sortable" data-sort="price" data-sort-type="num">Fiyat ↕</th>
                <th class="tk-th-center tk-th-sortable" data-sort="rating" data-sort-type="num">Puan ↕</th>
                <th class="tk-th-center tk-th-sortable" data-sort="reviewCount" data-sort-type="num">Yorum ↕</th>
                <th class="tk-th-right tk-th-sortable" data-sort="commRate" data-sort-type="num">Komisyon ↕</th>
                <th class="tk-th-right">Kargo</th>
                <th class="tk-th-right">Hizmet</th>
                <th class="tk-th-right tk-th-highlight tk-th-sortable" data-sort="kalanKDVHaric" data-sort-type="num">Hesaba Geçecek ↕</th>
                <th class="tk-th-right tk-th-sortable" data-sort="autoEstCost" data-sort-type="num">Maks.Maliyet ↕</th>
                <th class="tk-th-right tk-th-highlight tk-th-sortable" data-sort="profit" data-sort-type="num">Kâr/Satış ↕</th>
                <th class="tk-th-center tk-th-sortable" data-sort="estSales" data-sort-type="num" title="Aylık Satış Tahmini">Ay.Satış ↕</th>
                <th class="tk-th-center tk-th-sortable" data-sort="stockQty" data-sort-type="num" title="Satıcı Stok Adedi">Stok ↕</th>
                <th class="tk-th-center" title="Stok Delta Takibi">Takip</th>
                <th class="tk-th-right tk-th-highlight tk-th-sortable" data-sort="monthlyProfit" data-sort-type="num">Aylık Kâr ↕</th>
              </tr>
            </thead>
            <tbody id="tk-tbl-body"></tbody>
          </table>
        </div>
      </div>
    `;

    document.body.appendChild(tkOverlay);
    renderTableRows(products);

    // ─── Overlay cleanup: tüm document listener'larını temizle ───
    const _overlayCleanupFns = [];
    function cleanupOverlay() {
      _overlayCleanupFns.forEach(fn => fn());
      _overlayCleanupFns.length = 0;
      if (tkOverlay) { tkOverlay.remove(); tkOverlay = null; }
    }

    // Kapat butonu
    document.getElementById("tk-tbl-close").addEventListener("click", cleanupOverlay);

    // Excel export
    document.getElementById("tk-tbl-export").addEventListener("click", () => {
      exportToXLS(tkCurrentProducts);
    });

    // Kargo firması değiştir
    document.getElementById("tk-tbl-carrier").addEventListener("change", function () {
      settings.carrier = this.value;
      // Tüm satırları yeniden hesapla (satır kargo override'ı yoksa yeni firmaya göre)
      for (let i = 0; i < tkCurrentProducts.length; i++) {
        recalcRow(i);
      }
      renderTableRows(tkCurrentProducts);
      updateSubtitle(tkCurrentProducts);
    });

    // Sıralama (tüm sütunlar, string + number desteği)
    const sortDir = {};
    tkOverlay.querySelectorAll(".tk-th-sortable").forEach(th => {
      th.style.cursor = "pointer";
      th.addEventListener("click", () => {
        const key = th.getAttribute("data-sort");
        const type = th.getAttribute("data-sort-type") || "num";
        sortDir[key] = sortDir[key] === "asc" ? "desc" : "asc";
        const dir = sortDir[key];

        // Aktif sıralama göstergesi
        tkOverlay.querySelectorAll(".tk-th-sortable").forEach(h => {
          h.classList.remove("tk-th-sort-active");
        });
        th.classList.add("tk-th-sort-active");

        products.sort((a, b) => {
          let va = a[key], vb = b[key];
          if (type === "str") {
            va = (va || "").toString().toLowerCase();
            vb = (vb || "").toString().toLowerCase();
            const cmp = va.localeCompare(vb, "tr");
            return dir === "asc" ? cmp : -cmp;
          }
          va = parseFloat(va) || 0;
          vb = parseFloat(vb) || 0;
          return dir === "asc" ? va - vb : vb - va;
        });
        renderTableRows(products);
      });
    });

    // ESC ile kapat
    const escHandler = (e) => {
      if (e.key === "Escape") {
        cleanupOverlay();
      }
    };
    document.addEventListener("keydown", escHandler);
    _overlayCleanupFns.push(() => document.removeEventListener("keydown", escHandler));
  }

  function updateSubtitle(products) {
    const el = document.getElementById("tk-tbl-subtitle");
    if (!el) return;
    const firstMatch = products.length > 0 ? products[0].commMatch : "Varsayılan";
    let text = `${products.length} ürün · Kargo: ${settings.carrier} · Kategori: ${firstMatch}`;
    el.textContent = text;
  }

  // Tek satır yeniden hesaplama
  function recalcRow(idx) {
    const p = tkCurrentProducts[idx];
    if (!p) return;
    const HIZMET_BEDELI = 10.20;

    // Kargo: satır override > üst panel kargo > üst panel desi > otomatik
    // Tablo rakamları zaten KDV hariç — doğrudan base kullan
    if (p.rowOverrides.kargo !== null) {
      p.shipCost = p.rowOverrides.kargo;
      p.shipTier = "Manuel (satır)";
    } else if (tkManualOverrides.kargo !== null) {
      p.shipCost = tkManualOverrides.kargo;
      p.shipTier = "Manuel";
    } else {
      const desiVal = tkManualOverrides.desi || undefined;
      const shipping = getShippingWithDesi(p.price, settings.carrier, desiVal);
      p.shipCost = shipping.base || 0;
      p.shipTier = shipping.tier || "";
    }

    // KDV oranı: satır override > üst panel > ayarlar
    const kdvPct = p.rowOverrides.kdvRate !== null ? p.rowOverrides.kdvRate
      : tkManualOverrides.kdvRate !== null ? tkManualOverrides.kdvRate
        : (settings.kdvRate || 20);
    const kdvRate = kdvPct / 100;

    // Satış adedi: satır override > üst panel > akıllı tahmin
    if (p.rowOverrides.satisAdedi !== null) {
      p.estSales = p.rowOverrides.satisAdedi;
      p.salesTier = "Manuel";
      p.salesTierLabel = "Manuel";
    } else if (tkManualOverrides.satisAdedi !== null) {
      p.estSales = tkManualOverrides.satisAdedi;
      p.salesTier = "Manuel";
      p.salesTierLabel = "Manuel";
    } else {
      // Orijinal tier'ı koru (recalc sadece sayıyı günceller, kaynak değişmez)
      p.estSales = p.origEstSales;
    }

    // Karlılık hesabı: tüm kalemlerden KDV çıkar
    // Kargo zaten KDV hariç — doğrudan kullan
    const salePriceExVat = p.price / (1 + kdvRate);
    const commExVat = p.commAmount / (1 + kdvRate);
    const hizmetExVat = HIZMET_BEDELI / (1 + kdvRate);
    p.kalanKDVHaric = salePriceExVat - commExVat - hizmetExVat - p.shipCost;
    p.kalanKDVDahil = p.price - p.commAmount - HIZMET_BEDELI - p.shipCost;
    p.kdvAmount = p.price - salePriceExVat;

    // Maliyet: satır override > üst panel > tahmini (%20 kâr hedefi)
    p.autoEstCost = p.rowOverrides.maliyet !== null ? p.rowOverrides.maliyet
      : tkManualOverrides.maliyet !== null ? tkManualOverrides.maliyet
        : p.kalanKDVHaric * 0.80;

    p.profit = p.kalanKDVHaric - p.autoEstCost;
    p.monthlyProfit = p.profit * p.estSales;

    // DOM güncelle
    updateRowCells(idx);
  }

  // Tek satırdaki sonuç hücrelerini güncelle
  function updateRowCells(idx) {
    const p = tkCurrentProducts[idx];
    const fp = TrendyolCalculator.formatPrice;
    const row = document.querySelector(`#tk-tbl-body tr[data-idx="${idx}"]`);
    if (!row) return;

    const profitColor = p.profit >= 0 ? "#059669" : "#dc2626";
    const kalanColor = p.kalanKDVHaric >= 0 ? "#059669" : "#dc2626";
    const monthlyColor = p.monthlyProfit >= 0 ? "#059669" : "#dc2626";

    const cells = row.querySelectorAll("td");
    // cells[0]=#, cells[1]=Marka, cells[2]=Ad, cells[3]=Fiyat, cells[4]=Puan,
    // cells[5]=Yorum, cells[6]=Komisyon, cells[7]=Kargo, cells[8]=Hizmet,
    // cells[9]=Hesaba Geçecek, cells[10]=Maks.Maliyet, cells[11]=Kâr,
    // cells[12]=Satış, cells[13]=Stok, cells[14]=Takip, cells[15]=Aylık Kâr
    const kargoResult = cells[7].querySelector(".tk-row-kargo-result");
    if (kargoResult) kargoResult.innerHTML = `-${fp(p.shipCost)}<br><span class="tk-td-small tk-td-muted">${p.shipTier}</span>`;

    cells[9].style.color = kalanColor;
    cells[9].innerHTML = fp(p.kalanKDVHaric);

    // Maliyet input değerini güncelleme (input + sonuç)
    const maliyetResult = cells[10].querySelector(".tk-row-maliyet-result");
    if (maliyetResult) maliyetResult.textContent = fp(p.autoEstCost);

    cells[11].style.color = profitColor;
    cells[11].innerHTML = fp(p.profit);

    // Satış adedi sonuç
    const satisResult = cells[12].querySelector(".tk-row-satis-result");
    if (satisResult) satisResult.textContent = p.estSales;

    // cells[13] = Stok (değişmez, güncellemeye gerek yok)

    cells[15].style.color = monthlyColor;
    cells[15].innerHTML = fp(p.monthlyProfit);
  }

  function renderTableRows(products) {
    const tbody = document.getElementById("tk-tbl-body");
    if (!tbody) return;
    const HIZMET_BEDELI = 10.20;
    const fp = TrendyolCalculator.formatPrice;
    const fpct = TrendyolCalculator.formatPercent;

    let html = "";
    for (let i = 0; i < products.length; i++) {
      const p = products[i];
      const profitColor = p.profit >= 0 ? "#059669" : "#dc2626";
      const kalanColor = p.kalanKDVHaric >= 0 ? "#059669" : "#dc2626";
      const monthlyColor = p.monthlyProfit >= 0 ? "#059669" : "#dc2626";

      // Kargo input - satır override varsa göster
      const kargoVal = p.rowOverrides.kargo !== null ? p.rowOverrides.kargo : "";
      // Maliyet input
      const maliyetVal = p.rowOverrides.maliyet !== null ? p.rowOverrides.maliyet : "";
      // Satış adedi input
      const satisVal = p.rowOverrides.satisAdedi !== null ? p.rowOverrides.satisAdedi : "";

      html += `<tr data-idx="${i}">
        <td class="tk-td-center tk-td-muted">${p.idx}</td>
        <td class="tk-td-left tk-td-brand" title="${p.brand}">${p.brand}</td>
        <td class="tk-td-name">${p.link ? `<a href="${p.link}" target="_blank" class="tk-td-link">` : ""}
          <div class="tk-td-name-text" title="${p.name}">${p.name}</div>
        ${p.link ? "</a>" : ""}</td>
        <td class="tk-td-right tk-td-price">${fp(p.price)}</td>
        <td class="tk-td-center tk-td-muted">${p.rating > 0 ? p.rating.toFixed(1) : "-"}</td>
        <td class="tk-td-center tk-td-muted">${p.reviewCount || "-"}</td>
        <td class="tk-td-right tk-td-red">${fpct(p.commRate)}<br><span class="tk-td-small">-${fp(p.commAmount)}</span></td>
        <td class="tk-td-right tk-td-red">
          <input type="number" class="tk-row-input${kargoVal !== "" ? " tk-row-edited" : ""}" data-field="kargo" data-row="${i}" placeholder="${p.shipCost.toFixed(1)}" value="${kargoVal}" step="0.1" min="0" title="Kargo (TL)">
          <div class="tk-row-kargo-result">-${fp(p.shipCost)}<br><span class="tk-td-small tk-td-muted">${p.shipTier}</span></div>
        </td>
        <td class="tk-td-right tk-td-red">-${fp(HIZMET_BEDELI)}</td>
        <td class="tk-td-right tk-td-bold" style="color:${kalanColor}">${fp(p.kalanKDVHaric)}</td>
        <td class="tk-td-right tk-td-muted">
          <input type="number" class="tk-row-input${maliyetVal !== "" ? " tk-row-edited" : ""}" data-field="maliyet" data-row="${i}" placeholder="${p.autoEstCost.toFixed(1)}" value="${maliyetVal}" step="0.1" min="0" title="Maliyet (TL)">
          <div class="tk-row-maliyet-result">${fp(p.autoEstCost)}</div>
        </td>
        <td class="tk-td-right tk-td-bold" style="color:${profitColor}">${fp(p.profit)}</td>
        <td class="tk-td-center">
          <input type="number" class="tk-row-input tk-row-input-sm${satisVal !== "" ? " tk-row-edited" : ""}" data-field="satisAdedi" data-row="${i}" placeholder="${p.estSales}" value="${satisVal}" step="1" min="0" title="Aylık Satış Adedi">
          <div class="tk-row-satis-result tk-td-muted">${p.estSalesMin && p.estSalesMax ? (p.estSalesMin + "-" + p.estSalesMax) : p.estSales}</div>
        </td>
        <td class="tk-td-center tk-td-muted">${p.stockQty ? p.stockQty.toLocaleString("tr-TR") : "-"}</td>
        <td class="tk-td-center">
          ${(() => {
            const pid = (p.link || p.url || "").match(/-p-(\d+)/);
            if (!pid) return "-";
            const id = pid[1];
            const tracked = isProductTracked(id);
            const slug = (p.link || p.url || "").replace(/https?:\/\/www\.trendyol\.com/, "").split("?")[0];
            const safeName = (p.name || "").replace(/"/g, "&quot;").replace(/'/g, "&#39;");
            return '<button class="tk-track-mini-btn' + (tracked ? ' tk-tracking' : '') + '" data-product-id="' + id + '" data-slug="' + slug + '" data-name="' + safeName + '" data-price="' + (p.price || 0) + '">' + (tracked ? "&#10003;" : "+") + '</button>';
          })()}
        </td>
        <td class="tk-td-right tk-td-bold tk-td-big" style="color:${monthlyColor}">${fp(p.monthlyProfit)}</td>
      </tr>`;
    }
    tbody.innerHTML = html;

    // Satır inputlarına event listener ekle
    tbody.querySelectorAll(".tk-row-input").forEach(input => {
      input.addEventListener("change", function () {
        const row = parseInt(this.getAttribute("data-row"));
        const field = this.getAttribute("data-field");
        const val = this.value;

        tkCurrentProducts[row].rowOverrides[field] = val !== "" ? parseFloat(val) : null;

        // Edited class toggle
        if (val !== "") {
          this.classList.add("tk-row-edited");
        } else {
          this.classList.remove("tk-row-edited");
        }

        recalcRow(row);
      });
    });

    // ─── Takip mini butonları (event delegation — renderTableRows'dan bağımsız) ───
    // Önceki delegated listener varsa kaldır
    if (tbody._tkTrackHandler) {
      tbody.removeEventListener("click", tbody._tkTrackHandler);
    }
    tbody._tkTrackHandler = function (e) {
      const btn = e.target.closest(".tk-track-mini-btn");
      if (!btn) return;
      e.stopPropagation();

      const pid = btn.dataset.productId;
      if (!pid) return;
      const isTracked = btn.classList.contains("tk-tracking");

      // Anlık görsel geri bildirim
      btn.disabled = true;
      const origHTML = btn.innerHTML;
      btn.innerHTML = "⋯";

      if (isTracked) {
        chrome.runtime.sendMessage({ action: "UNTRACK_PRODUCT", productId: pid }, (resp) => {
          btn.disabled = false;
          if (chrome.runtime.lastError) {
            tkLog("Track hata:", chrome.runtime.lastError.message);
            btn.innerHTML = origHTML;
            return;
          }
          if (resp && resp.status === "ok") {
            btn.classList.remove("tk-tracking");
            btn.innerHTML = "+";
            trackedProductIds.delete(pid);
            panelTrackedIds.delete(pid);
          } else {
            btn.innerHTML = origHTML;
          }
        });
      } else {
        chrome.runtime.sendMessage({
          action: "TRACK_PRODUCT",
          product: {
            id: pid,
            slug: btn.dataset.slug || "",
            name: btn.dataset.name || "",
            brand: "",
            price: parseFloat(btn.dataset.price) || 0,
            imageUrl: ""
          }
        }, (resp) => {
          btn.disabled = false;
          if (chrome.runtime.lastError) {
            tkLog("Track hata:", chrome.runtime.lastError.message);
            btn.innerHTML = "+";
            btn.style.background = "#fef2f2";
            btn.style.color = "#dc2626";
            btn.title = "Bağlantı hatası — sayfayı yenileyin";
            setTimeout(() => { btn.style.background = ""; btn.style.color = ""; btn.innerHTML = "+"; btn.title = ""; }, 3000);
            return;
          }
          if (resp && (resp.status === "ok" || resp.status === "already_tracking")) {
            btn.classList.add("tk-tracking");
            btn.innerHTML = "&#10003;";
            trackedProductIds.add(pid);
            panelTrackedIds.add(pid);
          } else if (resp && resp.status === "limit_reached") {
            btn.style.background = "#ef4444";
            btn.style.color = "#fff";
            btn.innerHTML = "!";
            btn.title = "Ürün takip limitine ulaştınız";
            setTimeout(() => { btn.style.background = ""; btn.style.color = ""; btn.innerHTML = "+"; btn.title = ""; }, 3000);
          } else {
            btn.innerHTML = "+";
            tkLog("Track yanıt:", JSON.stringify(resp));
          }
        });
      }
    };
    tbody.addEventListener("click", tbody._tkTrackHandler);
  }

  function addBadgesToCards() {
    const cards = findProductCards();

    cards.forEach(card => {
      if (card.querySelector(".tk-badge")) return;

      // Fiyat bul — scrapeAllProducts ile aynı mantık
      let price = 0;
      const priceSelectors = [".price-value", ".discounted-price", ".sale-price", ".price-section", ".single-price", ".prc-box-dscntd", ".prc-box-sllng", "[class*='prc-box']", "[class*='product-price']"];
      for (const sel of priceSelectors) {
        const el = card.querySelector(sel);
        if (el) {
          const parsed = TrendyolCalculator.parsePrice(el.textContent);
          if (parsed > 0) { price = parsed; break; }
        }
      }
      if (!price) return;

      const breadcrumbs = getBreadcrumbsFromURL();

      const result = TrendyolCalculator.calculate({
        salePrice: price,
        breadcrumbs: breadcrumbs,
        carrier: settings.carrier,
        desi: tkManualOverrides.desi || undefined,
        reviewCount: 0,
        salesMultiplier: settings.salesMultiplier
      });

      const badge = document.createElement("div");
      badge.className = "tk-badge";
      badge.innerHTML = `
        <div class="tk-badge-item">
          <span class="tk-badge-label">Komisyon</span>
          <span class="tk-badge-value">${TrendyolCalculator.formatPercent(result.commission.rate)}</span>
        </div>
        <div class="tk-badge-item">
          <span class="tk-badge-label">Kargo</span>
          <span class="tk-badge-value">${TrendyolCalculator.formatPrice(result.shipping.base)}</span>
        </div>
        <div class="tk-badge-item">
          <span class="tk-badge-label">Hesaba Geçen</span>
          <span class="tk-badge-value ${result.hesabaGececek >= 0 ? "tk-positive" : "tk-negative"}">${TrendyolCalculator.formatPrice(result.hesabaGececek)}</span>
        </div>
      `;

      const descContainer = card.querySelector(".info-wrapper, .prdct-desc-cntnr, [class*='product-desc'], [class*='card-body']");
      if (descContainer) {
        descContainer.appendChild(badge);
      } else {
        card.appendChild(badge);
      }
    });
  }

  function getBreadcrumbsFromURL() {
    // ─── 1) Ürün detay sayfasındaki gerçek breadcrumb (en doğru kaynak) ───
    const breadcrumbSelectors = [
      ".breadcrumb-wrapper a",
      ".breadcrumb a",
      "[class*='breadcrumb'] a",
      ".product-detail-breadcrumb a",
      "[data-testid='breadcrumb'] a",
      "ul.breadcrumb-list a",
      "nav[aria-label='breadcrumb'] a",
      "li[class*='breadcrumb'] a"
    ];

    for (const sel of breadcrumbSelectors) {
      const els = document.querySelectorAll(sel);
      if (els.length >= 2) {
        const crumbs = Array.from(els)
          .map(a => a.textContent.trim())
          .filter(t => t && t.toLowerCase() !== "trendyol" && t.length > 1 && !t.includes(">"));
        if (crumbs.length > 0) {
          tkLog("Breadcrumb DOM'dan bulundu:", crumbs.join(" > "));
          return crumbs;
        }
      }
    }

    // ─── 2) Arama/Kategori sayfasında: sidebar kategori bilgisi + arama terimini kullan ───
    const crumbs = [];

    // Sidebar'daki "Kategori" başlığı altındaki seçili filtre
    // Trendyol'da kategori filtresi: .fltr-cntnr-ttl "Kategori" başlığının altındaki checkbox
    const catFilterSections = document.querySelectorAll(".fltr-cntnr-ttl, [class*='filter-title']");
    for (const titleSection of catFilterSections) {
      const titleText = (titleSection.textContent || "").trim().toLowerCase();
      if (titleText === "kategori" || titleText.includes("kategori")) {
        // Bu başlığın sonraki kardeş elementindeki checked checkbox'ı bul
        const filterContainer = titleSection.closest(".fltr-cntnr, [class*='filter-container']") || titleSection.parentElement;
        if (filterContainer) {
          const checkedCat = filterContainer.querySelector('input[type="checkbox"]:checked');
          if (checkedCat) {
            const label = checkedCat.closest("label") || checkedCat.parentElement;
            const catText = label ? label.textContent.trim().replace(/\(\d+\)/, "").trim() : "";
            if (catText.length > 1) crumbs.push(catText);
          }
        }
        break;
      }
    }

    // Arama terimi (q parametresi)
    const searchQuery = new URLSearchParams(window.location.search).get("q");
    if (searchQuery) {
      const terms = searchQuery
        .replace(/[+]/g, " ")
        .split(/\s+/)
        .map(w => w.charAt(0).toUpperCase() + w.slice(1))
        .join(" ");
      crumbs.push(terms);
    }

    // Sayfa başlığındaki kategori bilgisi (ör: h1 "Tuy Toplayici")
    const titleEl = document.querySelector("h1, .srch-rslt-ttl, [class*='result-title']");
    if (titleEl) {
      const titleText = titleEl.textContent.trim();
      if (titleText.length > 1 && !crumbs.includes(titleText)) {
        crumbs.push(titleText);
      }
    }

    if (crumbs.length > 0) {
      tkLog("Breadcrumb arama sayfasından:", crumbs.join(" > "));
      return crumbs;
    }

    // ─── 3) Fallback: URL path'inden çıkar ───
    const url = window.location.pathname;
    const parts = url.split("/").filter(p => p && p !== "sr");
    const urlCrumbs = parts.map(p =>
      p.replace(/-x-.*$/, "")
        .replace(/-c\d+$/, "")
        .replace(/-/g, " ")
        .replace(/\b\w/g, l => l.toUpperCase())
    );

    tkLog("Breadcrumb URL'den çıkarıldı:", urlCrumbs.join(" > "));
    return urlCrumbs;
  }

  // ===== MUTATION OBSERVER (Dinamik içerik için) =====

  let _pageObserver = null; // Singleton — disconnect için referans tut

  function observePageChanges() {
    // Önceki observer varsa temizle (tekrar çağrılırsa leak olmaz)
    if (_pageObserver) {
      _pageObserver.disconnect();
      _pageObserver = null;
    }

    let debounceTimer = null;
    _pageObserver = new MutationObserver(() => {
      // Debounce: Hızlı DOM değişikliklerinde yalnızca son çağrıyı işle
      clearTimeout(debounceTimer);
      debounceTimer = setTimeout(() => {
        const pageType = getPageType();
        if (pageType === "listing") {
          processListingPage(tkLastPageData);
        }
      }, 800);
    });

    _pageObserver.observe(document.body, {
      childList: true,
      subtree: true
    });
  }

  // ===== URL DEĞİŞİKLİĞİ TAKİBİ (SPA navigasyon) =====

  function observeURLChanges() {
    let lastURL = window.location.href;

    const checkURL = () => {
      if (window.location.href !== lastURL) {
        lastURL = window.location.href;
        // Eski paneli kaldır (yeni sayfa için yeniden oluşturulacak)
        const oldPanel = document.querySelector(".tk-panel");
        if (oldPanel) oldPanel.remove();
        // inject.js kendi URL takibinde veriyi zaten temizler ve yeniden yazar
        // Content script biraz daha geç başlasın ki inject.js yeni veriyi yazsın
        setTimeout(init, 2500);
      }
    };

    // Pushstate ve popstate takibi
    const origPushState = history.pushState;
    history.pushState = function () {
      origPushState.apply(this, arguments);
      checkURL();
    };

    const origReplaceState = history.replaceState;
    history.replaceState = function () {
      origReplaceState.apply(this, arguments);
      checkURL();
    };

    window.addEventListener("popstate", checkURL);
  }

  // ===== ANA BAŞLATMA =====

  // Panel auth kontrolu — giris yapilmamissa false doner
  async function checkPanelAuth() {
    return new Promise(resolve => {
      if (!chrome.runtime || !chrome.runtime.sendMessage) { resolve(null); return; }
      chrome.runtime.sendMessage({ action: "GET_PANEL_AUTH" }, (resp) => {
        if (chrome.runtime.lastError) { resolve(null); return; }
        resolve(resp && resp.auth && resp.auth.token ? resp.auth : null);
      });
    });
  }

  async function init() {
    await loadSettings();

    // Giris yapilmamissa panel/popup acma
    const auth = await checkPanelAuth();
    if (!auth) {
      tkLog("Panel girisi yapilmamis — icerik gosterilmeyecek");
      return;
    }

    // ─── Trendyol sayfası açıkken proaktif sync tetikle ───
    // Kullanıcı Trendyol'un herhangi bir sayfasındayken background'a sinyal gönder
    // Background bu sinyali alınca tracked ürünleri arka planda güncelleyebilir
    if (chrome.runtime && chrome.runtime.sendMessage) {
      chrome.runtime.sendMessage({
        action: "TRENDYOL_PAGE_ACTIVE",
        pageType: getPageType(),
        url: window.location.href
      }, () => {
        if (chrome.runtime.lastError) { /* silent */ }
      });
    }

    await loadTrackedProductIds();

    // Kalibrasyon verilerini yükle (panel sync'den kaydedilmiş)
    try {
      const calData = await new Promise(resolve => {
        chrome.storage.local.get("panelCalibration", (result) => resolve(result.panelCalibration || null));
      });
      if (calData) panelCalibration = calData;
    } catch (e) { /* kalibrasyon yüklenemedi — varsayılan kullanılacak */ }

    // ─── Kargo config'ini yükle (panel'den güncel fiyatlar) ───
    // Önce storage'dan hızlı yükle, sonra background'dan taze config iste
    try {
      const stored = await new Promise(resolve => {
        chrome.storage.local.get("shippingConfig", (result) => resolve(result.shippingConfig || null));
      });
      if (stored) {
        _applyShippingConfig(stored);
      }
      // Arka planda taze config iste — sayfa yüklemesini bloklamaz
      if (chrome.runtime && chrome.runtime.sendMessage) {
        chrome.runtime.sendMessage({ action: "FETCH_SHIPPING_CONFIG" }, (resp) => {
          if (chrome.runtime.lastError) return;
          if (resp && resp.status === "ok" && resp.config && !resp.cached) {
            _applyShippingConfig(resp.config);
            tkLog("Shipping config panel'den güncellendi");
          }
        });
      }
    } catch (e) { /* shipping config yüklenemedi — hardcoded varsayılanlar kullanılacak */ }

    const pageType = getPageType();

    // Sayfa JS değişkenlerini inject ile oku (isolated world bypass)
    const pageData = await readPageData();
    tkLog("Page data injected:", pageData ? Object.keys(pageData) : "null");

    if (pageType === "product") {
      // Sayfanın tamamen yüklenmesini bekle (çok satıcılı sayfa dahil)
      await waitForElement(".discounted-price, .sale-price, .price-section, .single-price, .price-container .discounted, .prc-dsc, .prc-box-dscntd, .price-current-price");

      // Social proof section'ının yüklenmesini de bekle (carousel animasyonlu — geç yüklenebilir)
      // Max 3 saniye bekle — social proof olmasa da devam et
      await waitForElement("[data-testid='social-proof'], .social-proof-social-proof-wrapper, [class*='social-proof-wrapper']", 3000).catch(() => {});

      let productData = extractProductData(pageData);

      // ─── Social Proof Retry ───
      // Social proof section geç yüklenebilir (carousel animasyonu, lazy load).
      // İlk extract'te bulunamadıysa 2 saniye bekleyip tekrar dene.
      if (!productData.recentSalesData && (!productData.socialProofData || !productData.socialProofData.basketCount)) {
        tkLog("Social proof ilk denemede bulunamadı — 2s sonra tekrar denenecek");
        await new Promise(r => setTimeout(r, 2000));
        const retryData = extractProductData(pageData);
        // Sadece eksik verileri doldur
        if (retryData.recentSalesData && !productData.recentSalesData) {
          productData.recentSalesData = retryData.recentSalesData;
          tkLog("Social proof RETRY başarılı: recentSalesData=" + JSON.stringify(retryData.recentSalesData));
        }
        if (retryData.socialProofData) {
          if (!productData.socialProofData) productData.socialProofData = {};
          if (retryData.socialProofData.basketCount && !productData.socialProofData.basketCount) {
            productData.socialProofData.basketCount = retryData.socialProofData.basketCount;
            tkLog("Social proof RETRY başarılı: basketCount=" + retryData.socialProofData.basketCount);
          }
        }
      }

      // URL'den ürün ID'si çıkar
      const pidMatch = window.location.href.match(/-p-(\d+)/);
      const productId = pidMatch ? pidMatch[1] : null;

      // Arka planda T1 veri çek (Trendyol arama API'sinden orderCount + socialProof)
      if (productId) {
        try {
          const bgResponse = await new Promise((resolve) => {
            chrome.runtime.sendMessage(
              { action: "FETCH_SINGLE_PRODUCT", productId: productId, slug: window.location.pathname },
              (resp) => resolve(resp)
            );
          });

          if (bgResponse && bgResponse.result && bgResponse.result.socialProof) {
            const sp = bgResponse.result.socialProof;
            tkLog("T1 veri (background API):", JSON.stringify(sp));

            // socialProofData yoksa oluştur
            if (!productData.socialProofData) {
              productData.socialProofData = {};
            }

            // orderCount varsa → T1 verisi!
            if (sp.orderCount) {
              productData.socialProofData.orderCount = sp.orderCount;
              tkLog(`  T1 orderCount: ${sp.orderCount}`);
            }

            // basketCount varsa ve mevcut yoksa → ekle
            if (sp.basketCount && !productData.socialProofData.basketCount) {
              productData.socialProofData.basketCount = sp.basketCount;
            }

            // favoriteCount varsa ve mevcut yoksa → ekle
            if (sp.favoriteCount && !productData.socialProofData.favoriteCount) {
              productData.socialProofData.favoriteCount = sp.favoriteCount;
            }
          }
        } catch (e) {
          tkLog("T1 background veri çekme hatası:", e.message);
        }

        // ─── T0: Stok Delta Takip verisi çek ───
        try {
          const trackingResp = await new Promise((resolve) => {
            chrome.runtime.sendMessage(
              { action: "GET_PRODUCT_TRACKING_DATA", productId: productId },
              (resp) => resolve(resp)
            );
          });

          if (trackingResp && trackingResp.data) {
            const td = trackingResp.data;
            if (!productData.socialProofData) productData.socialProofData = {};

            // T0 stok delta: EVENT-BAZLI ortalama (panel ile aynı mantık)
            // Eski: hourly projeksiyon (şişirilmiş) → Yeni: gerçek satış event ortalaması
            // last7Days'den satış olan günleri say ve ortalamasını al
            let eventBasedMonthly = 0;
            let eventDayCount = 0;
            if (td.last7Days && td.last7Days.length > 0) {
              let totalSold = 0;
              let daysWithData = 0;
              for (const day of td.last7Days) {
                if (day.sales > 0) {
                  totalSold += day.sales;
                  daysWithData++;
                }
              }
              if (daysWithData > 0) {
                const avgDaily = totalSold / daysWithData;
                eventBasedMonthly = Math.round(avgDaily * 30);
                eventDayCount = daysWithData;
              }
            }
            // Fallback: totalRealSold / daysTracked (last7Days yoksa veya boşsa)
            if (eventBasedMonthly === 0 && td.totalRealSold > 0 && td.daysTracked > 0) {
              eventBasedMonthly = Math.round((td.totalRealSold / Math.max(td.daysTracked, 1)) * 30);
              eventDayCount = td.daysTracked;
            }

            productData.socialProofData.stockDeltaMonthlySales = eventBasedMonthly;
            productData.socialProofData.stockDeltaConfidence = td.confidence;
            productData.socialProofData.stockDeltaDaysTracked = eventDayCount || td.daysTracked;
            productData.isTracked = true;
            productData.trackingData = td;

            // Panel'den gelen socialProof verisi (RECENT_SALES_COUNT, ORDER_COUNT vb.)
            // DOM'da bu veriler olmayabilir — panel'den alınan veriyi fallback olarak kullan
            if (td.panelSocialProof) {
              const psp = td.panelSocialProof;
              // RECENT_SALES_COUNT: DOM'da "satıldı" metni yoksa panel verisini kullan
              if (!productData.recentSalesData && psp.RECENT_SALES_COUNT > 0) {
                productData.recentSalesData = {
                  count: psp.RECENT_SALES_COUNT,
                  days: psp.RECENT_SALES_DAYS || 3
                };
                tkLog("T1 panel fallback: RECENT_SALES_COUNT=" + psp.RECENT_SALES_COUNT + "/" + (psp.RECENT_SALES_DAYS || 3) + "gün");
              }
              // ORDER_COUNT: socialProofData'da yoksa panel verisini kullan
              if (!productData.socialProofData.orderCount && psp.ORDER_COUNT) {
                productData.socialProofData.orderCount = psp.ORDER_COUNT;
                tkLog("T1 panel fallback: ORDER_COUNT=" + psp.ORDER_COUNT);
              }
              // BASKET_COUNT: DOM'da yoksa panel verisini kullan
              if (!productData.socialProofData.basketCount && psp.BASKET_COUNT) {
                productData.socialProofData.basketCount = psp.BASKET_COUNT;
                tkLog("T3 panel fallback: BASKET_COUNT=" + psp.BASKET_COUNT);
              }
            }

            tkLog("T0 stok takip verisi:", JSON.stringify({
              monthly: td.estimatedMonthlySales,
              confidence: td.confidence,
              daysTracked: td.daysTracked,
              panelSP: td.panelSocialProof ? 'var' : 'yok'
            }));
          }
        } catch (e) {
          tkLog("T0 tracking veri çekme hatası:", e.message);
        }
      }

      // ─── Panel Lookup: Panel DB'den taze tahmin çek (tracked ve untracked) ───
      // Storage'daki panelEstimate stale olabilir — her zaman panelden taze veri al
      if (productId && chrome.runtime && chrome.runtime.sendMessage) {
        try {
          const panelResp = await new Promise((resolve) => {
            chrome.runtime.sendMessage(
              { action: "FETCH_PANEL_ESTIMATE", productIds: [productId] },
              (resp) => resolve(resp)
            );
          });

          if (panelResp && panelResp.status === "ok" && panelResp.estimates) {
            const pe = panelResp.estimates[productId];
            if (pe && pe.daily > 0) {
              if (!productData.socialProofData) productData.socialProofData = {};
              productData.socialProofData.stockDeltaMonthlySales = pe.monthly;
              productData.socialProofData.stockDeltaPanelTier = pe.tier;
              // trackingData varsa panelEstimate ve rank güncelle, yoksa ayrı field
              if (productData.trackingData) {
                productData.trackingData.panelEstimate = pe;
                if (pe.dataQualityRank) productData.trackingData.dataQualityRank = pe.dataQualityRank;
              } else {
                productData.panelEstimate = pe;
              }
              tkLog("Panel estimate (taze):", JSON.stringify(pe));
            }
          }
        } catch (e) {
          tkLog("Panel estimate lookup hatası:", e.message);
        }
      }

      // productId'yi kaydet (panel ve takip butonları için)
      productData.productId = productId;

      // ─── Web Panel Sync: syncItem'ı hazırla (hesaplama sonrası gönderilecek) ───
      let _detailSyncItem = null;
      if (productId && chrome.runtime && chrome.runtime.sendMessage) {
        try {
          const syncItem = { productId: productId };

          // Slug — havuza ekleme için gerekli (DB'de yoksa slug ile oluşturulur)
          syncItem.slug = window.location.pathname;
          if (productData.title) syncItem.name = productData.title;
          if (productData.brand) syncItem.brand = productData.brand;
          if (productData.imageUrl || (pageData && pageData.detailProduct && pageData.detailProduct.imageUrl)) {
            syncItem.imageUrl = productData.imageUrl || pageData.detailProduct.imageUrl;
          }
          if (productData.categoryPath) syncItem.categoryPath = productData.categoryPath;

          // socialProofData (basketCount, orderCount, favoriteCount)
          if (productData.socialProofData) {
            const sp = {};
            if (productData.socialProofData.orderCount) sp.ORDER_COUNT = productData.socialProofData.orderCount;
            if (productData.socialProofData.basketCount) sp.BASKET_COUNT = productData.socialProofData.basketCount;
            if (productData.socialProofData.favoriteCount) sp.FAVORITE_COUNT = productData.socialProofData.favoriteCount;
            if (Object.keys(sp).length > 0) syncItem.socialProof = sp;
          }

          // recentSalesData (viral satış — "3 günde 10B+ satıldı")
          if (productData.recentSalesData) {
            syncItem.socialProof = syncItem.socialProof || {};
            syncItem.socialProof.RECENT_SALES_COUNT = productData.recentSalesData.count;
            syncItem.socialProof.RECENT_SALES_DAYS = productData.recentSalesData.days;
          }

          // stockDelta verisi (extension'in kendi T0 hesabi)
          if (productData.socialProofData && productData.socialProofData.stockDeltaMonthlySales > 0) {
            syncItem.stockDeltaMonthlySales = productData.socialProofData.stockDeltaMonthlySales;
            syncItem.stockDeltaDaysTracked = productData.socialProofData.stockDeltaDaysTracked || 0;
          }

          // Stok, fiyat, yorum, rating, buybox satıcı ID
          if (productData.stockQuantity) syncItem.stock = productData.stockQuantity;
          if (productData.price) syncItem.price = productData.price;
          if (productData.plusPrice) syncItem.plusPrice = productData.plusPrice;
          if (productData.originalPrice) syncItem.originalPrice = productData.originalPrice;
          if (productData.reviewCount) syncItem.reviewCount = productData.reviewCount;
          if (productData.rating) syncItem.averageRating = productData.rating;
          if (productData.merchantId) syncItem.merchantId = productData.merchantId;
          if (productData.merchantName) syncItem.merchantName = productData.merchantName;

          // Satıcı sayısı + merchantId'leri — önce inject.js (SSR) verisi, yoksa DOM'dan
          if (productData.otherMerchantIds && productData.otherMerchantIds.length > 0) {
            syncItem.otherMerchantIds = productData.otherMerchantIds;
            syncItem.totalMerchantCount = productData.otherMerchantIds.length + 1; // +1 buybox
            if (productData.otherMerchantNames) {
              syncItem.otherMerchantNames = productData.otherMerchantNames;
            }
          } else {
            // Fallback: DOM'dan merchant ID'leri
            try {
              const sellerLinks = document.querySelectorAll("[class*='other-merchant'] a[href*='merchantId='], [class*='other-sellers'] a[href*='merchantId=']");
              const merchantIds = [];
              sellerLinks.forEach(function(a) {
                const match = a.href.match(/merchantId=(\d+)/);
                if (match) merchantIds.push(parseInt(match[1]));
              });
              const uniqueMerchantIds = [...new Set(merchantIds)];
              if (uniqueMerchantIds.length > 0) {
                syncItem.otherMerchantIds = uniqueMerchantIds;
                syncItem.totalMerchantCount = uniqueMerchantIds.length + 1; // +1 buybox
              }
            } catch (e) { /* silent */ }
          }

          // Diğer satıcıların stok bilgileri (inject.js merchantListing hydration)
          // Panel'e tüm satıcı stoklarını göndererek daha doğru totalStock hesabı yapılır
          if (pageData && pageData.detailProduct && pageData.detailProduct.otherMerchants) {
            const otherStocks = {};
            pageData.detailProduct.otherMerchants.forEach(om => {
              if (om.id && om.stock > 0) otherStocks[String(om.id)] = om.stock;
            });
            if (Object.keys(otherStocks).length > 0) {
              syncItem.otherMerchantStocks = otherStocks;
            }
          }

          // Favori sayısı (inject.js'den)
          if (pageData && pageData.detailProduct && pageData.detailProduct.favoriteCount) {
            syncItem.favoriteCount = pageData.detailProduct.favoriteCount;
          }

          _detailSyncItem = syncItem;
        } catch (e) {
          tkLog("Panel sync hazırlama hatası:", e.message);
        }
      }

      if (productData.price > 0) {
        // Kalibrasyon oranını + kategori yolunu inject et
        if (productData.socialProofData) {
          productData.socialProofData.basketConversionRate = panelCalibration.basket_to_sales_rate || 0.10;
          productData.socialProofData.categoryPath = productData.categoryPath || "";
        }

        const result = TrendyolCalculator.calculate({
          salePrice: productData.price,
          breadcrumbs: productData.breadcrumbs,
          carrier: settings.carrier,
          desi: productData._userDesi || undefined,
          reviewCount: productData.reviewCount,
          rating: productData.rating,
          salesMultiplier: settings.salesMultiplier,
          badges: productData.badges,
          recentSalesData: productData.recentSalesData,
          socialProofData: productData.socialProofData,
          kdvRate: productData.taxRate,
          categoryTree: productData.categoryTree,
          targetProfitPct: productData._userTargetProfit
        });

        // Panel T0 bazlı ensemble (T0+T1+T3) en doğru hesap.
        // Panel tahmini varsa → authoritative (tek kaynak prensibi).
        // Panel tahmini yoksa → extension kendi hesabını kullanır (takip dışı ürün).
        const pe = (productData.trackingData && productData.trackingData.panelEstimate) || productData.panelEstimate;
        if (pe && pe.monthly > 0) {
          const extMonthly = result.estimatedMonthlySales || 0;
          result.estimatedMonthlySales = pe.monthly;
          result.estimatedMin = pe.min || Math.round(pe.monthly * 0.8);
          result.estimatedMax = pe.max || Math.round(pe.monthly * 1.2);
          result.salesTier = pe.tier || "T0";
          result.salesTierLabel = pe.tier || "T0";
          tkLog(`Panel T0 bazlı tahmin kullanıldı: ${pe.monthly}/ay ${pe.tier} (extension ham: ${extMonthly}/ay)`);
        }

        // ─── Birleşik Panel Sync: socialProof + stok + extensionEstimate TEK mesajda ───
        // Eski: 2 ayrı SYNC_SOCIAL_PROOF mesajı → 2 HTTP POST. Yeni: tek mesaj.
        if (_detailSyncItem && chrome.runtime && chrome.runtime.sendMessage) {
          try {
            // extensionEstimate'ı syncItem'a ekle
            if (result.estimatedMonthlySales > 0) {
              _detailSyncItem.extensionEstimate = {
                monthly: result.estimatedMonthlySales,
                daily: Math.round(result.estimatedMonthlySales / 30),
                tier: result.salesTier || "T0",
                min: result.estimatedMin || 0,
                max: result.estimatedMax || 0,
              };
            }
            if (_detailSyncItem.socialProof || _detailSyncItem.stock || _detailSyncItem.price) {
              chrome.runtime.sendMessage({
                action: "SYNC_SOCIAL_PROOF",
                products: [_detailSyncItem]
              }, () => { if (chrome.runtime.lastError) { /* silent */ } });
              tkLog("Panel sync gönderildi (birleşik):", JSON.stringify(_detailSyncItem));
            }
          } catch (e) { /* silent */ }
        } else if (_detailSyncItem && (_detailSyncItem.socialProof || _detailSyncItem.stock || _detailSyncItem.price)) {
          // Hesaplama yapılmadıysa (fiyat yok vs.) yine de socialProof/stok gönder
          try {
            chrome.runtime.sendMessage({
              action: "SYNC_SOCIAL_PROOF",
              products: [_detailSyncItem]
            }, () => { if (chrome.runtime.lastError) { /* silent */ } });
            tkLog("Panel sync gönderildi (sadece veri):", JSON.stringify(_detailSyncItem));
          } catch (e) { /* silent */ }
        }

        createPanel(result, productData);
        _detailSyncItem = null; // Gönderildi, tekrar gönderme
      }

      // Fallback: Fiyat yoksa hesaplama yapılamadı ama socialProof/stok hala gönderilmeli
      if (_detailSyncItem && (_detailSyncItem.socialProof || _detailSyncItem.stock || _detailSyncItem.price)) {
        try {
          chrome.runtime.sendMessage({
            action: "SYNC_SOCIAL_PROOF",
            products: [_detailSyncItem]
          }, () => { if (chrome.runtime.lastError) { /* silent */ } });
          tkLog("Panel sync gönderildi (fiyat yok fallback):", JSON.stringify(_detailSyncItem));
        } catch (e) { /* silent */ }
      }

      // ─── Fiyat geçmişi: Her durumda (fiyat olsun olmasın) enjekte et ───
      if (productId && !document.getElementById("tk-price-history-widget")) {
        tkLog("Fiyat geçmişi enjeksiyon başlatılıyor: productId=" + productId);
        injectPriceHistoryChart(productId);
      }
    } else if (pageType === "listing") {
      await waitForElement("a.product-card, .p-card-wrppr, [class*='p-card']");
      processListingPage(pageData);
      observePageChanges();

      // ─── Web Panel Sync: Listing sayfasındaki socialProof + temel ürün bilgileri ───
      // Havuza ekleme: slug (url) gönderilerek DB'de olmayan ürünler otomatik havuza eklenir
      if (pageData && pageData.listingProducts && pageData.listingProducts.length > 0 &&
          chrome.runtime && chrome.runtime.sendMessage) {
        try {
          const syncItems = [];
          for (const lp of pageData.listingProducts) {
            if (!lp.id) continue;
            const item = { productId: String(lp.id) };

            // socialProof parse — camelCase → SCREAMING_SNAKE_CASE normalize
            // Trendyol 2026: listing SSR camelCase key'ler dönüyor (basketCount, orderCount)
            if (lp.socialProof && Array.isArray(lp.socialProof) && lp.socialProof.length > 0) {
              const CAMEL_TO_SNAKE = {
                orderCount: 'ORDER_COUNT', basketCount: 'BASKET_COUNT',
                favoriteCount: 'FAVORITE_COUNT', pageViewCount: 'PAGE_VIEW_COUNT',
                recentSalesCount: 'RECENT_SALES_COUNT',
              };
              const sp = {};
              lp.socialProof.forEach(s => {
                if (s.key && s.value) {
                  const normalizedKey = CAMEL_TO_SNAKE[s.key] || s.key;
                  sp[normalizedKey] = s.value;
                }
              });
              if (Object.keys(sp).length > 0) item.socialProof = sp;
            }

            // Havuza ekleme için temel bilgiler (slug, name, brand, price, image, category)
            if (lp.url) item.slug = lp.url;
            if (lp.name) item.name = lp.name;
            if (lp.brand) item.brand = lp.brand;
            if (lp.price) item.price = lp.price;
            if (lp.image) item.imageUrl = lp.image;
            if (lp.category) {
              // category: { name: "Vücut Spreyi", id: 672 } veya string
              item.categoryPath = typeof lp.category === 'object'
                ? (lp.category.name || '')
                : String(lp.category);
            }

            // En az socialProof veya slug olmalı (boş item gönderme)
            if (item.socialProof || item.slug) {
              syncItems.push(item);
            }
          }
          if (syncItems.length > 0) {
            chrome.runtime.sendMessage({
              action: "SYNC_SOCIAL_PROOF",
              products: syncItems
            }, () => {
              if (chrome.runtime.lastError) { /* silent */ }
            });
            tkLog(`Listing sync: ${syncItems.length} ürün socialProof gönderildi`);
          }
        } catch (e) {
          tkLog("Listing sync hatası:", e.message);
        }
      }
    }
  }

  function waitForElement(selector, timeout) {
    timeout = timeout || 10000;
    return new Promise((resolve) => {
      const el = document.querySelector(selector);
      if (el) return resolve(el);

      const observer = new MutationObserver(() => {
        const el = document.querySelector(selector);
        if (el) {
          observer.disconnect();
          resolve(el);
        }
      });

      observer.observe(document.body, { childList: true, subtree: true });

      setTimeout(() => {
        observer.disconnect();
        resolve(null);
      }, timeout);
    });
  }

  // ===== KATEGORİ TARAYICI =====

  // Sessiz scrape — background'dan çağrılır (UI göstermez)
  async function silentScrapeForScanner(minProducts) {
    minProducts = minProducts || 50;
    const MAX_PRODUCTS = 80;
    const MAX_SCROLL_ATTEMPTS = 60;
    const SCROLL_STEP = Math.round(window.innerHeight * 0.8);
    let lastCount = 0;
    let sameCountRounds = 0;
    let retried = false;

    // Sayfanın yüklenmesini bekle
    await waitForElement("div.p-card-wrppr, a.product-card, [class*='p-card']", 8000);
    await new Promise(r => setTimeout(r, 1500));

    for (let attempt = 0; attempt < MAX_SCROLL_ATTEMPTS; attempt++) {
      const realCards = findProductCards();
      const currentCount = realCards.length;

      if (currentCount >= MAX_PRODUCTS) break;
      if (currentCount >= minProducts && sameCountRounds >= 5) break;
      if (sameCountRounds >= 8) break;

      if (currentCount === lastCount) {
        sameCountRounds++;
        // Retry: sayfa en altına git (arka plan tab'da lazy-load tetiklemek için)
        if (sameCountRounds === 2 && !retried) {
          retried = true;
          window.scrollTo({ top: document.body.scrollHeight, behavior: "smooth" });
          await new Promise(r => setTimeout(r, 2500));
          continue;
        }
      } else {
        sameCountRounds = 0;
        retried = false;
      }
      lastCount = currentCount;

      // Kontrollü tek adım scroll
      window.scrollBy({ top: SCROLL_STEP, behavior: "smooth" });

      // DOM'a yeni ürün kartı eklenmesini bekle (arka plan tab daha yavaş, 4s timeout)
      await waitForMoreCards(currentCount, 4000);

      // Ek stabilite bekleme
      await new Promise(r => setTimeout(r, 400));
    }

    window.scrollTo({ top: 0 });

    return scrapeAllProducts(tkLastPageData);
  }

  // Scanner Dashboard — tüm kategorilerin sonuçlarını gösterir
  function showScannerDashboard(allProducts, scanDate) {
    // Mevcut overlay varsa kaldır
    const existingOverlay = document.getElementById("tk-scanner-overlay");
    if (existingOverlay) existingOverlay.remove();

    const overlay = document.createElement("div");
    overlay.id = "tk-scanner-overlay";
    overlay.style.cssText = "position:fixed;top:0;left:0;right:0;bottom:0;z-index:999999;background:#0d1117;overflow-y:auto;font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif;";

    // Kategorileri al
    const categories = [...new Set(allProducts.map(p => p.scanCategory))].sort();

    // İstatistikler
    const totalProducts = allProducts.length;
    const profitableProducts = allProducts.filter(p => p.profit > 0);
    const avgProfit = profitableProducts.length > 0
      ? profitableProducts.reduce((s, p) => s + p.profit, 0) / profitableProducts.length : 0;
    const bestCategory = categories.reduce((best, cat) => {
      const catProducts = allProducts.filter(p => p.scanCategory === cat && p.profit > 0);
      const catAvg = catProducts.length > 0 ? catProducts.reduce((s, p) => s + p.profit, 0) / catProducts.length : 0;
      return catAvg > best.avg ? { name: cat, avg: catAvg } : best;
    }, { name: "-", avg: 0 });

    // Karlılık sıralı
    const sortedProducts = [...allProducts].sort((a, b) => b.monthlyProfit - a.monthlyProfit);

    const dateStr = scanDate ? new Date(scanDate).toLocaleDateString("tr-TR", {
      day: "2-digit", month: "2-digit", year: "numeric", hour: "2-digit", minute: "2-digit"
    }) : "";

    const fp = TrendyolCalculator.formatPrice;

    // Kategori seçim HTML
    const categoryOptionsHtml = categories.map(c =>
      `<option value="${c}">${c}</option>`
    ).join("");

    overlay.innerHTML = `
      <div style="max-width:1400px;margin:0 auto;padding:20px;">
        <!-- Header -->
        <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:20px;">
          <div>
            <h1 style="color:#ffd700;font-size:24px;margin:0;">🔍 Kategori Tarayıcı Sonuçları</h1>
            <p style="color:#8b949e;font-size:13px;margin-top:4px;">${dateStr} tarihli tarama — ${categories.length} kategori tarandı</p>
          </div>
          <div style="display:flex;gap:10px;align-items:center;">
            <button id="tk-scan-export" style="padding:8px 16px;background:#238636;color:#fff;border:none;border-radius:6px;font-size:13px;font-weight:600;cursor:pointer;">📥 Excel İndir</button>
            <button id="tk-scan-close" style="padding:8px 16px;background:#da3633;color:#fff;border:none;border-radius:6px;font-size:13px;font-weight:600;cursor:pointer;">✕ Kapat</button>
          </div>
        </div>

        <!-- Summary Cards -->
        <div style="display:grid;grid-template-columns:repeat(4,1fr);gap:14px;margin-bottom:20px;">
          <div style="background:#161b22;border:1px solid #30363d;border-radius:10px;padding:16px;">
            <div style="color:#8b949e;font-size:11px;margin-bottom:4px;">Toplam Ürün</div>
            <div style="color:#fff;font-size:22px;font-weight:700;">${totalProducts}</div>
          </div>
          <div style="background:#161b22;border:1px solid #30363d;border-radius:10px;padding:16px;">
            <div style="color:#8b949e;font-size:11px;margin-bottom:4px;">Kârlı Ürün Sayısı</div>
            <div style="color:#3fb950;font-size:22px;font-weight:700;">${profitableProducts.length}</div>
          </div>
          <div style="background:#161b22;border:1px solid #30363d;border-radius:10px;padding:16px;">
            <div style="color:#8b949e;font-size:11px;margin-bottom:4px;">Ort. Kâr (Kârlı Ürünler)</div>
            <div style="color:#ffd700;font-size:22px;font-weight:700;">${fp(avgProfit)}</div>
          </div>
          <div style="background:#161b22;border:1px solid #30363d;border-radius:10px;padding:16px;">
            <div style="color:#8b949e;font-size:11px;margin-bottom:4px;">En Kârlı Kategori</div>
            <div style="color:#f78166;font-size:16px;font-weight:700;line-height:1.3;">${bestCategory.name}</div>
          </div>
        </div>

        <!-- Filters -->
        <div style="background:#161b22;border:1px solid #30363d;border-radius:10px;padding:14px;margin-bottom:16px;display:flex;gap:14px;align-items:center;flex-wrap:wrap;">
          <div>
            <label style="color:#8b949e;font-size:11px;display:block;margin-bottom:3px;">Kategori</label>
            <select id="tk-scan-filter-cat" style="padding:6px 10px;background:#0d1117;color:#c9d1d9;border:1px solid #30363d;border-radius:6px;font-size:12px;">
              <option value="">Tümü</option>
              ${categoryOptionsHtml}
            </select>
          </div>
          <div>
            <label style="color:#8b949e;font-size:11px;display:block;margin-bottom:3px;">Min. Fiyat (TL)</label>
            <input type="number" id="tk-scan-filter-minprice" placeholder="0" style="width:90px;padding:6px 10px;background:#0d1117;color:#c9d1d9;border:1px solid #30363d;border-radius:6px;font-size:12px;">
          </div>
          <div>
            <label style="color:#8b949e;font-size:11px;display:block;margin-bottom:3px;">Max. Fiyat (TL)</label>
            <input type="number" id="tk-scan-filter-maxprice" placeholder="∞" style="width:90px;padding:6px 10px;background:#0d1117;color:#c9d1d9;border:1px solid #30363d;border-radius:6px;font-size:12px;">
          </div>
          <div>
            <label style="color:#8b949e;font-size:11px;display:block;margin-bottom:3px;">Min. Kâr (TL)</label>
            <input type="number" id="tk-scan-filter-minprofit" placeholder="0" style="width:90px;padding:6px 10px;background:#0d1117;color:#c9d1d9;border:1px solid #30363d;border-radius:6px;font-size:12px;">
          </div>
          <div>
            <label style="color:#8b949e;font-size:11px;display:block;margin-bottom:3px;">Min. Yorum</label>
            <input type="number" id="tk-scan-filter-minreview" placeholder="0" style="width:80px;padding:6px 10px;background:#0d1117;color:#c9d1d9;border:1px solid #30363d;border-radius:6px;font-size:12px;">
          </div>
          <div style="margin-left:auto;">
            <label style="color:transparent;font-size:11px;display:block;margin-bottom:3px;">.</label>
            <button id="tk-scan-filter-apply" style="padding:6px 16px;background:#f27a1a;color:#fff;border:none;border-radius:6px;font-size:12px;font-weight:600;cursor:pointer;">Filtrele</button>
          </div>
        </div>

        <!-- Table -->
        <div style="background:#161b22;border:1px solid #30363d;border-radius:10px;overflow:hidden;">
          <div style="overflow-x:auto;">
            <table style="width:100%;border-collapse:collapse;font-size:12px;">
              <thead>
                <tr style="background:#21262d;">
                  <th style="padding:10px 8px;color:#8b949e;font-weight:600;text-align:center;white-space:nowrap;border-bottom:1px solid #30363d;">#</th>
                  <th style="padding:10px 8px;color:#8b949e;font-weight:600;text-align:left;white-space:nowrap;border-bottom:1px solid #30363d;">Kategori</th>
                  <th style="padding:10px 8px;color:#8b949e;font-weight:600;text-align:left;white-space:nowrap;border-bottom:1px solid #30363d;min-width:200px;">Ürün Adı</th>
                  <th class="tk-scan-sortable" data-sort="price" style="padding:10px 8px;color:#ffd700;font-weight:600;text-align:right;white-space:nowrap;border-bottom:1px solid #30363d;cursor:pointer;">Fiyat ↕</th>
                  <th style="padding:10px 8px;color:#8b949e;font-weight:600;text-align:center;white-space:nowrap;border-bottom:1px solid #30363d;">Puan</th>
                  <th style="padding:10px 8px;color:#8b949e;font-weight:600;text-align:center;white-space:nowrap;border-bottom:1px solid #30363d;">Yorum</th>
                  <th style="padding:10px 8px;color:#8b949e;font-weight:600;text-align:right;white-space:nowrap;border-bottom:1px solid #30363d;">Komisyon</th>
                  <th style="padding:10px 8px;color:#8b949e;font-weight:600;text-align:right;white-space:nowrap;border-bottom:1px solid #30363d;">Kargo</th>
                  <th style="padding:10px 8px;color:#8b949e;font-weight:600;text-align:right;white-space:nowrap;border-bottom:1px solid #30363d;">Hesaba Geçecek</th>
                  <th class="tk-scan-sortable" data-sort="profit" style="padding:10px 8px;color:#3fb950;font-weight:600;text-align:right;white-space:nowrap;border-bottom:1px solid #30363d;cursor:pointer;">Kâr/Satış ↕</th>
                  <th style="padding:10px 8px;color:#8b949e;font-weight:600;text-align:center;white-space:nowrap;border-bottom:1px solid #30363d;">Ay.Satış</th>
                  <th class="tk-scan-sortable" data-sort="monthlyProfit" style="padding:10px 8px;color:#ffd700;font-weight:600;text-align:right;white-space:nowrap;border-bottom:1px solid #30363d;cursor:pointer;">Aylık Kâr ↕</th>
                </tr>
              </thead>
              <tbody id="tk-scan-tbody"></tbody>
            </table>
          </div>
        </div>

        <div style="text-align:center;color:#484f58;font-size:11px;padding:20px;">
          TPro360 — Kategori Tarayıcı v1.2.0
        </div>
      </div>
    `;

    document.body.appendChild(overlay);

    // Global state
    let currentSort = { key: "monthlyProfit", dir: "desc" };
    let filteredProducts = sortedProducts;

    function renderScanTable(products) {
      const tbody = document.getElementById("tk-scan-tbody");
      if (!tbody) return;

      tbody.innerHTML = products.map((p, i) => {
        const profitColor = p.profit >= 0 ? "#3fb950" : "#f85149";
        const monthlyColor = p.monthlyProfit >= 0 ? "#3fb950" : "#f85149";
        const kalanColor = p.kalanKDVHaric >= 0 ? "#3fb950" : "#f85149";
        const rowBg = i % 2 === 0 ? "transparent" : "rgba(255,255,255,0.02)";

        return `<tr style="background:${rowBg};border-bottom:1px solid #21262d;">
          <td style="padding:8px;color:#484f58;text-align:center;">${i + 1}</td>
          <td style="padding:8px;color:#f78166;font-size:11px;white-space:nowrap;">${p.scanCategory || ""}</td>
          <td style="padding:8px;color:#c9d1d9;max-width:250px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap;">
            ${p.link ? `<a href="${p.link}" target="_blank" style="color:#58a6ff;text-decoration:none;" title="${p.name}">${p.name}</a>` : p.name}
          </td>
          <td style="padding:8px;color:#c9d1d9;text-align:right;font-weight:600;">${fp(p.price)}</td>
          <td style="padding:8px;color:${p.rating >= 4 ? '#3fb950' : '#c9d1d9'};text-align:center;">${p.rating > 0 ? p.rating.toFixed(1) : "-"}</td>
          <td style="padding:8px;color:#c9d1d9;text-align:center;">${p.reviewCount || 0}</td>
          <td style="padding:8px;color:#f85149;text-align:right;font-size:11px;">%${p.commRate.toFixed(1)}<br><span style="color:#484f58;">${fp(p.commAmount)}</span></td>
          <td style="padding:8px;color:#f85149;text-align:right;font-size:11px;">${fp(p.shipCost)}</td>
          <td style="padding:8px;color:${kalanColor};text-align:right;font-weight:600;">${fp(p.kalanKDVHaric)}</td>
          <td style="padding:8px;color:${profitColor};text-align:right;font-weight:700;">${fp(p.profit)}</td>
          <td style="padding:8px;color:#c9d1d9;text-align:center;">${p.estSales}</td>
          <td style="padding:8px;color:${monthlyColor};text-align:right;font-weight:700;font-size:13px;">${fp(p.monthlyProfit)}</td>
        </tr>`;
      }).join("");
    }

    renderScanTable(sortedProducts);

    // ─── Overlay cleanup: document listener'larını temizle ───
    const _scanCleanupFns = [];
    function cleanupScanOverlay() {
      _scanCleanupFns.forEach(fn => fn());
      _scanCleanupFns.length = 0;
      if (overlay) overlay.remove();
    }

    // Kapat butonu
    document.getElementById("tk-scan-close").addEventListener("click", cleanupScanOverlay);

    // ESC ile kapat
    const escHandler = (e) => {
      if (e.key === "Escape") {
        cleanupScanOverlay();
      }
    };
    document.addEventListener("keydown", escHandler);
    _scanCleanupFns.push(() => document.removeEventListener("keydown", escHandler));

    // Sıralama
    const sortDirs = {};
    overlay.querySelectorAll(".tk-scan-sortable").forEach(th => {
      th.addEventListener("click", () => {
        const key = th.getAttribute("data-sort");
        sortDirs[key] = sortDirs[key] === "asc" ? "desc" : "asc";
        const dir = sortDirs[key];
        filteredProducts.sort((a, b) => dir === "asc" ? a[key] - b[key] : b[key] - a[key]);
        currentSort = { key, dir };
        renderScanTable(filteredProducts);
      });
    });

    // Filtrele
    document.getElementById("tk-scan-filter-apply").addEventListener("click", () => {
      const cat = document.getElementById("tk-scan-filter-cat").value;
      const minPrice = parseFloat(document.getElementById("tk-scan-filter-minprice").value) || 0;
      const maxPrice = parseFloat(document.getElementById("tk-scan-filter-maxprice").value) || Infinity;
      const minProfit = parseFloat(document.getElementById("tk-scan-filter-minprofit").value) || -Infinity;
      const minReview = parseInt(document.getElementById("tk-scan-filter-minreview").value) || 0;

      filteredProducts = sortedProducts.filter(p => {
        if (cat && p.scanCategory !== cat) return false;
        if (p.price < minPrice || p.price > maxPrice) return false;
        if (p.profit < minProfit) return false;
        if ((p.reviewCount || 0) < minReview) return false;
        return true;
      });

      filteredProducts.sort((a, b) =>
        currentSort.dir === "asc" ? a[currentSort.key] - b[currentSort.key] : b[currentSort.key] - a[currentSort.key]
      );

      renderScanTable(filteredProducts);
    });

    // Excel XLS Export
    document.getElementById("tk-scan-export").addEventListener("click", () => {
      const dateStr = new Date().toLocaleDateString("tr-TR");
      const xml = `
<html xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:x="urn:schemas-microsoft-com:office:excel" xmlns="http://www.w3.org/TR/REC-html40">
<head>
<meta charset="UTF-8">
<!--[if gte mso 9]><xml><x:ExcelWorkbook><x:ExcelWorksheets><x:ExcelWorksheet>
<x:Name>Kategori Tarama</x:Name>
<x:WorksheetOptions><x:DisplayGridlines/></x:WorksheetOptions>
</x:ExcelWorksheet></x:ExcelWorksheets></x:ExcelWorkbook></xml><![endif]-->
<style>
  td, th { mso-number-format:"\\@"; font-family: Calibri, sans-serif; }
  .num { mso-number-format:"\\#\\,\\#\\#0\\.00"; }
  .int { mso-number-format:"\\#\\,\\#\\#0"; }
</style>
</head>
<body>
<table>
  <tr><td colspan="15" style="font-size:18px;font-weight:bold;color:#f27a1a;padding:8px;">TPro360 — Kategori Tarama Sonuçları</td></tr>
  <tr><td colspan="15" style="font-size:11px;color:#666;padding:4px 8px 12px;">${dateStr} tarihli tarama — ${filteredProducts.length} ürün</td></tr>
  <tr style="background:#f27a1a;color:#fff;font-weight:bold;font-size:11px;">
    <th style="padding:8px;text-align:center;">#</th>
    <th style="padding:8px;text-align:left;">Kategori</th>
    <th style="padding:8px;text-align:left;">Ürün Adı</th>
    <th style="padding:8px;text-align:right;">Fiyat (₺)</th>
    <th style="padding:8px;text-align:center;">Puan</th>
    <th style="padding:8px;text-align:center;">Yorum</th>
    <th style="padding:8px;text-align:right;">Komisyon %</th>
    <th style="padding:8px;text-align:right;">Komisyon (₺)</th>
    <th style="padding:8px;text-align:right;">Kargo (₺)</th>
    <th style="padding:8px;text-align:right;">Hizmet (₺)</th>
    <th style="padding:8px;text-align:right;">Hesaba Geçecek (₺)</th>
    <th style="padding:8px;text-align:right;">Maks. Maliyet (₺)</th>
    <th style="padding:8px;text-align:right;">Kâr/Satış (₺)</th>
    <th style="padding:8px;text-align:center;">Aylık Satış</th>
    <th style="padding:8px;text-align:right;">Aylık Kâr (₺)</th>
  </tr>
${filteredProducts.map((p, i) => {
  const profitColor = p.profit >= 0 ? "#00b365" : "#e53e3e";
  const monthlyColor = p.monthlyProfit >= 0 ? "#00b365" : "#e53e3e";
  const bgColor = i % 2 === 0 ? "#fff" : "#f9fafb";
  return `  <tr style="background:${bgColor};font-size:12px;">
    <td style="text-align:center;color:#999;">${i + 1}</td>
    <td>${(p.scanCategory || "").replace(/</g,"&lt;")}</td>
    <td>${p.name.replace(/</g,"&lt;")}</td>
    <td class="num" style="text-align:right;font-weight:600;color:#f27a1a;">${p.price.toFixed(2)}</td>
    <td style="text-align:center;">${p.rating > 0 ? p.rating.toFixed(1) : "-"}</td>
    <td class="int" style="text-align:center;">${p.reviewCount || "-"}</td>
    <td class="num" style="text-align:right;">${p.commRate.toFixed(1)}</td>
    <td class="num" style="text-align:right;color:#e53e3e;">${p.commAmount.toFixed(2)}</td>
    <td class="num" style="text-align:right;color:#e53e3e;">${p.shipCost.toFixed(2)}</td>
    <td class="num" style="text-align:right;color:#e53e3e;">10.20</td>
    <td class="num" style="text-align:right;font-weight:600;">${p.kalanKDVHaric.toFixed(2)}</td>
    <td class="num" style="text-align:right;">${p.autoEstCost.toFixed(2)}</td>
    <td class="num" style="text-align:right;font-weight:700;color:${profitColor};">${p.profit.toFixed(2)}</td>
    <td class="int" style="text-align:center;">${p.estSales}</td>
    <td class="num" style="text-align:right;font-weight:700;color:${monthlyColor};">${p.monthlyProfit.toFixed(2)}</td>
  </tr>`;
}).join("\n")}
</table>
</body>
</html>`;

      const blob = new Blob([xml], { type: "application/vnd.ms-excel;charset=utf-8" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = "TPro360_Kategori_Tarama_" + new Date().toISOString().slice(0, 10) + ".xls";
      a.click();
      URL.revokeObjectURL(url);
    });
  }

  // Background'dan gelen mesajları dinle
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    if (msg.action === "SCRAPE_CATEGORY") {
      silentScrapeForScanner(msg.minProducts || 50).then(products => {
        chrome.runtime.sendMessage({
          action: "SCRAPE_RESULT",
          products: products,
          category: msg.category
        });
        sendResponse({ status: "done", count: products.length });
      }).catch(err => {
        console.warn("Scrape hatası:", err);
        chrome.runtime.sendMessage({
          action: "SCRAPE_RESULT",
          products: [],
          category: msg.category
        });
        sendResponse({ status: "error" });
      });
      return true; // async yanıt
    }

    if (msg.action === "SHOW_SCANNER_DASHBOARD") {
      showScannerDashboard(msg.products || [], msg.scanDate);
      sendResponse({ status: "ok" });
      return;
    }
  });

  // Başlat
  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", () => {
      setTimeout(init, 500);
    });
  } else {
    setTimeout(init, 500);
  }

  observeURLChanges();

})();
