// Trendyol Karlılık Hesaplama Motoru

// Sabit hizmet bedeli (KDV dahil, sipariş başına)
const HIZMET_BEDELI = 10.20;

const TrendyolCalculator = {
  // Türkçe karakter normalizasyonu
  _normalize(str) {
    return str
      .toLowerCase()
      .replace(/ğ/g, "g").replace(/ü/g, "u").replace(/ş/g, "s")
      .replace(/ı/g, "i").replace(/ö/g, "o").replace(/ç/g, "c")
      .replace(/â/g, "a").replace(/î/g, "i").replace(/û/g, "u")
      .replace(/&/g, " ").replace(/\s+/g, " ").trim();
  },

  // İki string arasındaki eşleşme skoru (0-1 arası)
  // search: kullanıcı tarafından aranan terim (breadcrumb)
  // target: komisyon tablosundaki kategori/altKategori/ürünGrubu ismi
  _phraseScore(search, target) {
    if (search === target) return 1.0;
    // target, search'ü tamamen içeriyor → iyi eşleşme
    // ör: target="cocuk bebek", search="cocuk" → "cocuk" target'ın parçası
    if (target.includes(search)) return 0.85;
    // search, target'ı tamamen içeriyor
    // ör: search="cocuk giyim", target="giyim" → giyim tek kelime, ambiguous → düşük skor
    // ör: search="cocuk bebek giyim", target="cocuk bebek" → çok kelimeli, spesifik → 0.80
    // ör: search="kozmetik kisisel bakim", target="kozmetik" → tek kelime ama ana terim → 0.75
    if (search.includes(target)) {
      const ratio = target.length / search.length;
      if (ratio >= 0.65) return 0.80;
      // Target birden fazla kelime ve hepsi search'te var → iyi eşleşme
      const targetWords = target.split(/\s+/).filter(w => w.length > 2);
      if (targetWords.length >= 2 && targetWords.every(w => search.includes(w))) return 0.78;
      // Target tek kelime ama search'ün ilk kelimesi ile aynı → ana kategori eşleşmesi
      const searchFirstWord = search.split(/\s+/)[0];
      if (target === searchFirstWord && target.length >= 4) return 0.75;
      return 0;
    }
    return 0;
  },

  // Breadcrumb + Kategori Ağacı (ID) eşleştirmesi
  // Öncelik:
  //   1) categoryTree ID eşleştirmesi (inject.js'den gelen Trendyol kategori ID'leri)
  //   2) Breadcrumb metin eşleştirmesi (eski yöntem — fallback)
  //
  // categoryTree formatı: [{ id: 1234, name: "Süpermarket" }, { id: 5678, name: "Temizlik" }, ...]
  findCommission(breadcrumbs, categoryTree) {
    // ─── ÖNCELİK 1: categoryHierarchy breadcrumbs zaten doğru sırada gelir ───
    // categoryTree ise Trendyol'dan TERS sırada gelir (alt→üst):
    //   ["Temizlik Bezleri", "Temizlik Gereçleri", "Ev Bakım ve Temizlik", "Süpermarket"]
    // Bu yüzden: breadcrumbs (hierarchy'den) varsa ona öncelik ver.
    // Sadece breadcrumbs boşsa categoryTree'yi reverse edip kullan.
    if ((!breadcrumbs || breadcrumbs.length === 0) && categoryTree && Array.isArray(categoryTree) && categoryTree.length > 0) {
      const treeBreadcrumbs = categoryTree
        .map(c => c.name)
        .filter(n => n && n !== "Trendyol" && n.length > 1);
      // inject.js'de zaten reverse edildi — doğru sırada (üst→alt)
      if (treeBreadcrumbs.length > 0) {
        breadcrumbs = treeBreadcrumbs;
      }
    }

    if (!breadcrumbs || breadcrumbs.length === 0) {
      return { komisyon: DEFAULT_COMMISSION, vade: DEFAULT_VADE, match: "Varsayılan", confidence: 0 };
    }

    // Tüm breadcrumb'ları normalize et
    const searchTerms = breadcrumbs
      .map(b => this._normalize(b))
      .filter(b => b && b !== "trendyol" && b.length > 1);

    if (searchTerms.length === 0) {
      return { komisyon: DEFAULT_COMMISSION, vade: DEFAULT_VADE, match: "Varsayılan", confidence: 0 };
    }

    // Tüm breadcrumb'ları tek string olarak birleştir
    const fullSearchText = searchTerms.join(" ");

    // Tüm breadcrumb kelimelerini set olarak tut
    const allSearchWords = new Set();
    searchTerms.forEach(term => term.split(/\s+/).forEach(w => { if (w.length > 2) allSearchWords.add(w); }));

    // ─── ADIM 1: Kategori seviyesi filtrele ───
    // Breadcrumb'daki ilk eleman genellikle ana kategori (Süpermarket, Elektronik, vs.)
    // İlk elemana öncelik ver, eğer eşleşmezse diğerlerini de dene.
    // Birden fazla eşleşme varsa en yüksek skorlu olanı seç.
    let categoryFilteredItems = TRENDYOL_COMMISSIONS;
    let matchedKategori = "";
    let bestCatScore = 0;

    for (let ti = 0; ti < searchTerms.length; ti++) {
      const term = searchTerms[ti];
      const matching = [];
      let termBestScore = 0;

      for (const item of TRENDYOL_COMMISSIONS) {
        const score = this._phraseScore(term, this._normalize(item.kategori));
        if (score >= 0.80) {
          matching.push(item);
          termBestScore = Math.max(termBestScore, score);
        }
      }

      if (matching.length > 0) {
        // İlk breadcrumb elemanına bonus ver (ana kategori olma olasılığı yüksek)
        const positionBonus = ti === 0 ? 0.10 : (ti === 1 ? 0.05 : 0);
        const effectiveScore = termBestScore + positionBonus;

        if (effectiveScore > bestCatScore) {
          bestCatScore = effectiveScore;
          categoryFilteredItems = matching;
          matchedKategori = term;
        }
      }
    }

    // ─── ADIM 2: altKategori seviyesi filtrele ───
    // Breadcrumb'daki 2. veya 3. eleman alt kategori (Ev Bakım ve Temizlik, Temizlik, vs.)
    let altFilteredItems = categoryFilteredItems;
    let matchedAlt = "";
    let bestAltScore = 0;

    for (let ti = 0; ti < searchTerms.length; ti++) {
      const term = searchTerms[ti];
      if (term === matchedKategori) continue; // Zaten kategori olarak eşleşti

      const matching = [];
      let termBestScore = 0;

      for (const item of categoryFilteredItems) {
        const score = this._phraseScore(term, this._normalize(item.altKategori));
        if (score >= 0.80) {
          matching.push(item);
          termBestScore = Math.max(termBestScore, score);
        }
      }

      if (matching.length > 0 && termBestScore > bestAltScore) {
        bestAltScore = termBestScore;
        altFilteredItems = matching;
        matchedAlt = term;
      }
    }

    // ─── ADIM 3: ürünGrubu seviyesi eşleşme (en spesifik) ───
    let bestMatch = null;
    let bestScore = 0;

    for (const item of altFilteredItems) {
      const normUrun = this._normalize(item.urunGrubu);
      let score = 0;

      // Breadcrumb terimlerinden biriyle ürün grubu eşleşmesi
      for (const term of searchTerms) {
        if (term === matchedKategori || term === matchedAlt) continue;
        const urunScore = this._phraseScore(term, normUrun);
        if (urunScore > 0) score += urunScore * 30;
      }

      // Full text eşleşme (breadcrumb birleşik metin içinde)
      if (fullSearchText.includes(normUrun) && normUrun.length > 3) score += 15;

      // Kelime bazlı eşleşme
      const urunWords = normUrun.split(/\s+/).filter(w => w.length > 2);
      let wordHits = 0;
      for (const uw of urunWords) {
        if (allSearchWords.has(uw)) wordHits++;
      }
      if (urunWords.length > 0 && wordHits > 0) {
        score += (wordHits / urunWords.length) * 10;
      }

      if (score > bestScore) {
        bestScore = score;
        bestMatch = item;
      }
    }

    // ─── ADIM 4: Sonuç seç ───
    // Ürün grubu eşleşmesi yeterince güçlüyse onu kullan
    if (bestMatch && bestScore >= 8) {
      return {
        komisyon: bestMatch.komisyon,
        vade: bestMatch.vade,
        match: `${bestMatch.kategori} > ${bestMatch.altKategori} > ${bestMatch.urunGrubu}`,
        confidence: Math.min(bestScore / 30, 1)
      };
    }

    // Ürün grubu birebir eşleşmedi → altKategori seviyesindeki İLK kaydı kullan
    // Bu, "Tüy Toplayıcı" gibi spesifik ürün yoksa üst kategorinin oranını alır
    if (altFilteredItems.length > 0 && matchedAlt) {
      const fallback = altFilteredItems[0];
      return {
        komisyon: fallback.komisyon,
        vade: fallback.vade,
        match: `${fallback.kategori} > ${fallback.altKategori} > ${fallback.urunGrubu} (üst kategori)`,
        confidence: 0.65
      };
    }

    // Sadece ana kategori eşleştiyse → o kategorideki ilk kaydı kullan
    if (categoryFilteredItems.length > 0 && matchedKategori) {
      const fallback = categoryFilteredItems[0];
      return {
        komisyon: fallback.komisyon,
        vade: fallback.vade,
        match: `${fallback.kategori} > ${fallback.altKategori} > ${fallback.urunGrubu} (ana kategori)`,
        confidence: 0.4
      };
    }

    // ─── ADIM 5: Hiç hiyerarşik eşleşme yoksa → eski puanlama sistemi (fallback) ───
    bestMatch = null;
    bestScore = 0;

    for (const item of TRENDYOL_COMMISSIONS) {
      const normKategori = this._normalize(item.kategori);
      const normAlt = this._normalize(item.altKategori);
      const normUrun = this._normalize(item.urunGrubu);

      let score = 0;
      for (const term of searchTerms) {
        score += this._phraseScore(term, normKategori) * 25;
        score += this._phraseScore(term, normAlt) * 20;
        score += this._phraseScore(term, normUrun) * 15;
      }
      if (fullSearchText.includes(normAlt) && normAlt.length > 3) score += 12;
      if (fullSearchText.includes(normUrun) && normUrun.length > 3) score += 10;

      const itemWords = new Set();
      [normKategori, normAlt, normUrun].forEach(t =>
        t.split(/\s+/).forEach(w => { if (w.length > 2) itemWords.add(w); })
      );
      for (const iw of itemWords) {
        if (allSearchWords.has(iw)) score += 2;
      }

      if (score > bestScore) {
        bestScore = score;
        bestMatch = item;
      }
    }

    if (bestMatch && bestScore >= 4) {
      return {
        komisyon: bestMatch.komisyon,
        vade: bestMatch.vade,
        match: `${bestMatch.kategori} > ${bestMatch.altKategori} > ${bestMatch.urunGrubu}`,
        confidence: Math.min(bestScore / 30, 1)
      };
    }

    return { komisyon: DEFAULT_COMMISSION, vade: DEFAULT_VADE, match: "Varsayılan", confidence: 0 };
  },

  // ═══════════════════════════════════════════════════════════════════
  //  KARGO HESAPLAMA MOTORU
  // ─────────────────────────────────────────────────────────────────
  //  3 mod:
  //  1) desi girildi → WEIGHT_SHIPPING tablosundan desi bazlı ücret
  //  2) desi yok + fiyat < 300 TL → BAREM_SHIPPING (fiyat aralığına göre)
  //  3) desi yok + fiyat >= 300 TL → barem 0 ama min. 3 desi ücreti uygulanır
  //
  //  Trendyol kuralı: 10 desi ve altı → barem destek uygulanır
  //                    10 desi üstü  → desi bazlı ücret uygulanır
  //  Satıcı kendi desisini biliyorsa girmeli — daha doğru hesaplama yapar.
  // ═══════════════════════════════════════════════════════════════════

  // Carrier key mapping (BAREM key → WEIGHT key)
  _carrierKeyMap: {
    "TEX-PTT": "tex", "ARAS": "aras", "SÜRAT": "surat",
    "KOLAY GELSİN": "kolayGelsin", "DHLeCommerce": "dhlEcommerce", "YK": "yurtici"
  },

  // WEIGHT_SHIPPING tablosundan desi/kg bazlı kargo ücreti çek
  _getWeightBasedCost(desi, carrier) {
    const carrierKey = this._carrierKeyMap[carrier] || "tex";

    for (const tier of WEIGHT_SHIPPING) {
      if (desi > tier.minKg && desi <= tier.maxKg) {
        const basePrice = tier[carrierKey] || tier.ptt;
        const withKDV = basePrice * (1 + KDV_RATE);
        return {
          base: basePrice,
          kdv: withKDV - basePrice,
          total: withKDV,
          carrier: carrier,
          tier: `${desi} desi (${tier.minKg}-${tier.maxKg} kg)`,
          desi: desi,
          method: "desi"
        };
      }
    }

    // Tabloda karşılık bulunamazsa en son tier
    const lastTier = WEIGHT_SHIPPING[WEIGHT_SHIPPING.length - 1];
    const basePrice = lastTier[carrierKey] || lastTier.ptt;
    const withKDV = basePrice * (1 + KDV_RATE);
    return {
      base: basePrice, kdv: withKDV - basePrice, total: withKDV,
      carrier: carrier, tier: `${desi} desi (${lastTier.maxKg}+ kg)`,
      desi: desi, method: "desi"
    };
  },

  // Barem carrier key eşleştirme
  _matchBaremCarrier(carrier) {
    const carrierKeys = Object.keys(BAREM_SHIPPING[0].carriers);
    let matched = carrierKeys.find(k => k === carrier);
    if (!matched) {
      matched = carrierKeys.find(k =>
        k.toLowerCase().includes(carrier.toLowerCase()) ||
        carrier.toLowerCase().includes(k.toLowerCase())
      );
    }
    return matched || carrierKeys[0];
  },

  // Ana kargo hesaplama fonksiyonu
  // desi: kullanıcının girdiği desi değeri (null/undefined = otomatik)
  // categoryPath: opsiyonel kategori yolu (desi tahmini için)
  //
  // Trendyol kuralı:
  //   <300 TL + ≤10 desi → barem sabit fiyat
  //   <300 TL + >10 desi → desi bazlı
  //   300+ TL → her zaman desi bazlı
  getShippingCost(salePrice, carrier, desi, categoryPath) {
    carrier = carrier || DEFAULT_CARRIER;

    // ─── MOD 1: Desi girildi ───
    if (desi !== undefined && desi !== null && desi > 0) {
      if (desi <= 10) {
        const matchedKey = this._matchBaremCarrier(carrier);
        for (const tier of BAREM_SHIPPING) {
          if (salePrice >= tier.minPrice && salePrice <= tier.maxPrice) {
            const baremPrice = tier.carriers[matchedKey];
            if (baremPrice !== undefined && baremPrice > 0) {
              // <300 TL + ≤10 desi: barem sabit fiyat
              const withKDV = baremPrice * (1 + KDV_RATE);
              return {
                base: baremPrice, kdv: withKDV - baremPrice, total: withKDV,
                carrier: matchedKey, tier: tier.label, desi, method: "barem"
              };
            }
            // baremPrice === 0 → 300+ TL: desi bazlı
            break;
          }
        }
      }
      // >10 desi VEYA 300+ TL: desi bazlı
      return this._getWeightBasedCost(desi, carrier);
    }

    // ─── MOD 2: Desi yok + Fiyat < 300 TL → Barem ───
    if (salePrice < 300) {
      // Barem sadece ≤10 desi için, büyük ürünlerde kategori tahmini ile kontrol
      if (typeof estimateDesiFromCategory === 'function' && categoryPath) {
        const estDesi = estimateDesiFromCategory(categoryPath);
        if (estDesi > 10) {
          const result = this._getWeightBasedCost(estDesi, carrier);
          result.method = "desi_estimate";
          result.estimatedDesi = estDesi;
          result.tier = `~${estDesi} desi (tahmin)`;
          return result;
        }
      }
      const matchedKey = this._matchBaremCarrier(carrier);
      for (const tier of BAREM_SHIPPING) {
        if (salePrice >= tier.minPrice && salePrice <= tier.maxPrice) {
          const basePrice = tier.carriers[matchedKey] || tier.carriers[Object.keys(tier.carriers)[0]];
          const withKDV = basePrice * (1 + KDV_RATE);
          return {
            base: basePrice, kdv: withKDV - basePrice, total: withKDV,
            carrier: matchedKey, tier: tier.label, desi: null, method: "barem"
          };
        }
      }
    }

    // ─── MOD 3: Desi yok + Fiyat >= 300 TL → Kategori bazlı desi tahmini ───
    let estimatedDesi = 3; // varsayılan fallback
    if (typeof estimateDesiFromCategory === 'function' && categoryPath) {
      estimatedDesi = estimateDesiFromCategory(categoryPath);
    }
    const result = this._getWeightBasedCost(estimatedDesi, carrier);
    result.tier = `~${estimatedDesi} desi (tahmin)`;
    result.method = "desi_estimate";
    result.estimatedDesi = estimatedDesi;
    return result;
  },

  // ═══════════════════════════════════════════════════════════════════
  //  AYLIK SATIŞ TAHMİNİ MOTORU v4 — Çoklu Veri Kaynağı
  // ═══════════════════════════════════════════════════════════════════
  //  Öncelik sırasıyla 4 katmanlı veri kaynağı kullanır:
  //
  //  T1: socialProof.orderCount ("800+") — Trendyol'un kendi sipariş verisi
  //  T2: DOM socialProof text ("3 günde 500+ satıldı") — Viral pencere
  //  T3: socialProof sinyalleri (basketCount + favoriteCount) — Dolaylı tahmin
  //  T4: Power Law model (reviewCount, rating, price) — Fallback
  //
  //  Her katman bir öncekini SADECE yokluğunda devreye girer.
  // ═══════════════════════════════════════════════════════════════════

  // socialProof value'larını sayıya çevir ("3K" → 3000, "800+" → 800, "102K" → 102000)
  _parseSocialValue(val) {
    if (!val) return 0;
    val = String(val).trim();
    // "800+" → 800
    val = val.replace("+", "");
    // "3,3B" veya "3.3B" → 3300 (Türkçe "B" = Bin = 1000)
    if (val.includes("B")) {
      return Math.round(parseFloat(val.replace("B", "").replace(",", ".")) * 1000);
    }
    // "102K" → 102000
    if (val.toUpperCase().includes("K")) {
      return Math.round(parseFloat(val.toUpperCase().replace("K", "").replace(",", ".")) * 1000);
    }
    // "3M" → 3000000
    if (val.toUpperCase().includes("M")) {
      return Math.round(parseFloat(val.toUpperCase().replace("M", "").replace(",", ".")) * 1000000);
    }
    // Düz sayı
    return parseInt(val.replace(/\./g, "").replace(",", ".")) || 0;
  },

  // Eski uyumluluk: sadece sayı döndürür
  estimateMonthlySales(reviewCount, rating, salePrice, badges = [], recentSalesData = null, socialProofData = null) {
    const result = this.estimateMonthlySalesDetailed(reviewCount, rating, salePrice, badges, recentSalesData, socialProofData);
    return result.count;
  },

  // Detaylı satış tahmini: { count, tier, tierLabel, signals, weights } döndürür
  // Multi-signal ensemble: 2+ sinyal varsa ağırlıklı ortalama — panel ile aynı mantık
  estimateMonthlySalesDetailed(reviewCount, rating, salePrice, badges = [], recentSalesData = null, socialProofData = null) {

    // ─── TÜM SİNYALLERİ TOPLA ───
    let t0Monthly = 0;
    let t0DaysTracked = 0;
    let t0Confidence = 0;
    let panelTier = null;
    let t1Monthly = 0;
    let t1PeriodDays = 0;
    let t1RawOrderCount = null;
    let t3Monthly = 0;

    // ─── T0: STOK DELTA (event-bazlı ortalama — panel ile aynı mantık) ───
    if (socialProofData && socialProofData.stockDeltaMonthlySales > 0) {
      t0Monthly = Math.round(socialProofData.stockDeltaMonthlySales);
      t0Confidence = socialProofData.stockDeltaConfidence || 0.5;
      t0DaysTracked = socialProofData.stockDeltaDaysTracked || 0;
      panelTier = socialProofData.stockDeltaPanelTier || null;
    }

    // ─── T1: Sipariş verisi (RECENT_SALES_COUNT) ───
    // Kısa dönem verisi (genelde 3 gün) — kampanya/viral dönemi olabilir
    // Sabit %20 discount: kısa dönem > uzun dönem ortalaması
    if (recentSalesData && recentSalesData.count > 0 && recentSalesData.days > 0) {
      const rawCount = recentSalesData.count;
      const orders = rawCount < 1000 ? rawCount + 50
        : rawCount < 10000 ? rawCount + 500
        : rawCount + 5000;
      const dailyRate = orders / Math.max(recentSalesData.days, 1);
      const discount = 0.80;
      t1Monthly = Math.round(dailyRate * 30 * discount);
      t1PeriodDays = recentSalesData.days;
      t1RawOrderCount = rawCount + "+";
    } else if (socialProofData && socialProofData.orderCount) {
      const rawOrders = this._parseSocialValue(socialProofData.orderCount);
      if (rawOrders > 0) {
        const orders = rawOrders < 1000 ? rawOrders + 50
          : rawOrders < 10000 ? rawOrders + 500
          : rawOrders + 5000;
        // Period tahmini: text yoksa ORDER_COUNT büyüklüğünden tahmin et
        // Trendyol mantığı: yüksek ORDER_COUNT → kısa pencere
        //   >= 3000 → "3 günde" (çok popüler)
        //   >= 500  → "7 günde" (orta)
        //   >= 50   → "14 günde" (düşük)
        let divisor = socialProofData.orderCountDays || (
          rawOrders >= 3000 ? 3 : rawOrders >= 500 ? 7 : rawOrders >= 50 ? 14 : 7
        );
        let discount = 0.80;
        if (socialProofData.basketCount) {
          const basketVal = this._parseSocialValue(socialProofData.basketCount);
          if (basketVal > 0 && rawOrders > 0) {
            const basketRatio = basketVal / rawOrders;
            if (basketRatio > 5) discount = 0.90;
            else if (basketRatio < 0.3) discount = 0.65;
            // Konservatif pencere kisaltma: ORDER_COUNT bucket eskiligini kompanse et
            // Test verisi: Kisaltma ile T1, gercek satis (EMA/T0) ile tutarli
            if (divisor >= 7) {
              if (basketRatio > 15) divisor = Math.max(Math.round(divisor * 0.65), 5);
              else if (basketRatio > 8) divisor = Math.max(Math.round(divisor * 0.80), 6);
            }
          }
        }
        const dailyRate = orders / divisor;
        t1Monthly = Math.round(dailyRate * 30 * discount);
        t1PeriodDays = divisor;
        t1RawOrderCount = String(socialProofData.orderCount);
      }
    }

    // ─── T3: Sepet/Favori sinyali — fiyat bazlı model (panel ile aynı) ───
    // Panel: monthly = basketVal * baseRate * min((refPrice/price)^priceElasticity, CAP)
    const T3_BASE_RATE = 0.25;
    const T3_REF_PRICE = 200;
    const T3_PRICE_ELASTICITY = 1.2;
    const T3_PRICE_MULTIPLIER_CAP = 8; // Ucuz urunlerde sinirsiz artisi onle (panel ile ayni)

    if (socialProofData && (socialProofData.basketCount || socialProofData.favoriteCount)) {
      const basketVal = this._parseSocialValue(socialProofData.basketCount);
      const favVal = this._parseSocialValue(socialProofData.favoriteCount);
      const catCoeff = this.getCategoryCoefficients(socialProofData.categoryPath || null);
      const price = salePrice || T3_REF_PRICE;

      if (basketVal > 0) {
        // Fiyat bazlı dönüşüm: ucuz ürünlerde yüksek, pahalıda düşük
        // CAP: Kalibrasyon verileri 150+ TL — ucuz urunlerde formul patliyor, max 8x
        const priceMultiplier = Math.min(
          Math.pow(T3_REF_PRICE / Math.max(price, 10), T3_PRICE_ELASTICITY),
          T3_PRICE_MULTIPLIER_CAP
        );
        const baseRate = socialProofData.basketConversionRate > 0
          ? socialProofData.basketConversionRate : T3_BASE_RATE;
        t3Monthly = Math.round(basketVal * baseRate * priceMultiplier);
      }
      // Favori sinyali: fiyat bazlı çarpan (panel ile aynı)
      const favMultiplier = catCoeff.favoriteMultiplier || 0.002;
      if (t3Monthly === 0 && favVal > 0) {
        const priceMultiplier = Math.min(
          Math.pow(T3_REF_PRICE / Math.max(price, 10), T3_PRICE_ELASTICITY),
          T3_PRICE_MULTIPLIER_CAP
        );
        t3Monthly = Math.round(favVal * favMultiplier * priceMultiplier);
      }
    }

    // ─── MULTI-SIGNAL ENSEMBLE ───
    const t0Daily = t0Monthly > 0 ? Math.round(t0Monthly / 30) : 0;
    const t1Daily = t1Monthly > 0 ? Math.round(t1Monthly / 30) : 0;
    const t3Daily = t3Monthly > 0 ? Math.round(t3Monthly / 30) : 0;

    const signalCount = (t0Daily > 0 ? 1 : 0) + (t1Daily > 0 ? 1 : 0) + (t3Daily > 0 ? 1 : 0);

    if (signalCount >= 2 && t0Daily > 0) {
      // T0 + diğer sinyaller — ağırlıklı ortalama (panel ile aynı)
      const dCount = t0DaysTracked || 0;
      let w0, w1, w3;

      if (t1Daily > 0 && t3Daily > 0) {
        // 3 sinyal
        if (dCount >= 7) { w0 = 0.55; w1 = 0.25; w3 = 0.20; }
        else if (dCount >= 3) { w0 = 0.40; w1 = 0.30; w3 = 0.30; }
        else { w0 = 0.20; w1 = 0.35; w3 = 0.45; }

        // T0+T1+T3 divergence-aware: Virtual vs Delta farkli davranmali
        // Extension stock_tracking_mode bilemez → socialProofData'dan trackingMode varsa kullan
        // yoksa virtual varsay (cogu urun virtual stokta)
        const ratio01 = t0Daily / t1Daily;
        const isVirtual = !(socialProofData && socialProofData.trackingMode === 'delta');
        if (ratio01 < 0.20) {
          if (isVirtual) {
            w0 = 0.10; w1 = 0.55; w3 = 0.35;
          } else {
            // Delta: T0 per-merchant stok delta = guvenilir → T0 baskin
            w0 = Math.max(w0, 0.50); w1 = 0.30; w3 = 0.20;
          }
        } else if (ratio01 < 0.40) {
          if (isVirtual) {
            w0 = Math.min(w0, 0.25); w1 = Math.max(w1, 0.45); w3 = 1 - w0 - w1;
          } else {
            w0 = Math.max(w0, 0.45); w1 = Math.min(w1, 0.30); w3 = 1 - w0 - w1;
          }
        }
      } else if (t3Daily > 0) {
        // T0 + T3 — panel ile aynı ağırlıklar
        if (dCount >= 7) { w0 = 0.70; w3 = 0.30; }
        else if (dCount >= 3) { w0 = 0.60; w3 = 0.40; }
        else { w0 = 0.50; w3 = 0.50; }
        w1 = 0;

        // T0+T3 divergence-aware: zayıf T3 güçlü T0'ı çekmemeli (ve tersi)
        const ratio03 = t0Daily / t3Daily;
        if (ratio03 >= 10 || ratio03 <= 0.10) {
          if (t0Daily > t3Daily) { w0 = 0.95; w3 = 0.05; }
          else { w3 = 0.90; w0 = 0.10; }
        } else if (ratio03 >= 5 || ratio03 <= 0.20) {
          if (t0Daily > t3Daily) { w0 = 0.90; w3 = 0.10; }
          else { w3 = 0.85; w0 = 0.15; }
        } else if (ratio03 >= 3 || ratio03 <= 0.33) {
          if (t0Daily > t3Daily) { w0 = Math.max(w0, 0.80); w3 = 1 - w0; }
          else { w3 = Math.max(w3, 0.80); w0 = 1 - w3; }
        }
      } else {
        // T0 + T1
        if (dCount >= 7) { w0 = 0.70; w1 = 0.30; }
        else if (dCount >= 3) { w0 = 0.50; w1 = 0.50; }
        else { w0 = 0.15; w1 = 0.85; }
        w3 = 0;
      }

      const blended = Math.round(t0Daily * w0 + t1Daily * (w1 || 0) + t3Daily * (w3 || 0));
      const blendMonthly = blended * 30;

      let blendTier;
      if (t1Daily > 0 && t3Daily > 0) blendTier = "T0+T1+T3";
      else if (t3Daily > 0) blendTier = "T0+T3";
      else blendTier = "T0~T1";

      return {
        count: blendMonthly,
        tier: blendTier,
        tierLabel: blendTier + " Blend (" + dCount + "g)",
        signals: { t0: t0Monthly, t1: t1Monthly || null, t3: t3Monthly || null },
        weights: { t0: w0, t1: w1 || null, t3: w3 || null },
        daysTracked: dCount
      };
    }

    // T1 + T3 (T0 yok) — divergence-aware + RECENT_SALES priority (panel ile ayni)
    if (t1Daily > 0 && t3Daily > 0) {
      let w1 = 0.55, w3 = 0.45;

      // RECENT_SALES_COUNT varsa T1 dogrudan satis verisi — daha yuksek guven
      if (recentSalesData && recentSalesData.count > 0) {
        w1 = 0.75; w3 = 0.25;
      }

      // Divergence-aware: sinyaller 3x+ farkliysa yuksek olani tercih et
      if (t3Daily > 0) {
        const ratio = t1Daily / t3Daily;
        if (ratio >= 5 || ratio <= 0.20) {
          if (t1Daily > t3Daily) { w1 = Math.max(w1, 0.85); w3 = 1 - w1; }
          else { w3 = Math.max(w3, 0.85); w1 = 1 - w3; }
        } else if (ratio >= 3 || ratio <= 0.33) {
          if (t1Daily > t3Daily) { w1 = Math.max(w1, 0.80); w3 = 1 - w1; }
          else { w3 = Math.max(w3, 0.80); w1 = 1 - w3; }
        }
      }

      const blended = Math.round(t1Daily * w1 + t3Daily * w3);
      return {
        count: blended * 30,
        tier: "T1+T3",
        tierLabel: "Sipariş+Sepet Blend",
        signals: { t0: null, t1: t1Monthly, t3: t3Monthly },
        weights: { t0: null, t1: w1, t3: w3 },
        rawOrderCount: t1RawOrderCount,
        periodDays: t1PeriodDays
      };
    }

    // Tek sinyal fallback'ler
    if (t0Monthly > 0) {
      const tierName = panelTier ? ("Panel " + panelTier) : "T0";
      const tierLbl = panelTier
        ? ("Panel " + panelTier + " verisi")
        : ("Stok Takip (" + t0DaysTracked + "g)");
      return {
        count: t0Monthly,
        tier: tierName,
        tierLabel: tierLbl,
        confidence: panelTier ? 0.75 : t0Confidence,
        daysTracked: t0DaysTracked
      };
    }

    if (t1Monthly > 0) {
      return {
        count: t1Monthly,
        tier: "T1",
        tierLabel: "Sipariş Verisi (" + t1PeriodDays + "g)",
        rawOrderCount: t1RawOrderCount,
        periodDays: t1PeriodDays
      };
    }

    if (t3Monthly > 0) {
      const organicMonthly = this._calcOrganicDaily(reviewCount, rating, salePrice) * 30;
      const bestEstimate = Math.max(organicMonthly, t3Monthly);
      const badgeMultiplier = this._getBadgeMultiplier(badges);
      return {
        count: Math.round(bestEstimate * badgeMultiplier),
        tier: "T3",
        tierLabel: "Sepet/Favori"
      };
    }

    // ─── T4: POWER LAW FALLBACK (Klasik model) ───
    const organicDaily = this._calcOrganicDaily(reviewCount, rating, salePrice);
    const badgeMultiplier = this._getBadgeMultiplier(badges);
    const finalMonthly = (organicDaily * 30) * badgeMultiplier;

    return {
      count: Math.round(finalMonthly),
      tier: "T4",
      tierLabel: "Tahmin"
    };
  },

  // Organik günlük satış hesabı (Power Law)
  _calcOrganicDaily(reviewCount, rating, salePrice) {
    const reputationPower = Math.pow(reviewCount || 0, 0.60);
    const referencePrice = 350;
    const priceAnalysis = salePrice > 0 ? salePrice : referencePrice;
    const priceElasticity = Math.pow(referencePrice / (priceAnalysis + 20), 0.45);
    const ratingInput = rating > 0 ? rating : 4.0;
    const trustCurve = Math.pow(ratingInput / 5.0, 4);
    const CALIBRATION_CONSTANT = 6.5;

    let organicDaily = (reputationPower * priceElasticity * trustCurve * CALIBRATION_CONSTANT) / 30;
    if ((reviewCount || 0) < 15) organicDaily *= 0.7;
    return Math.max(0.03, organicDaily);
  },

  // Rozet çarpanı hesabı
  _getBadgeMultiplier(badges = []) {
    const badgeStr = (badges || []).join(" ").toLowerCase();
    if (badgeStr.includes("en çok satan") || badgeStr.includes("çok satan")) return 2.5;
    if (badgeStr.includes("yıldızlı") || badgeStr.includes("yildizli")) return 1.5;
    if (badgeStr.includes("hızlı") || badgeStr.includes("hizli")) return 1.2;
    return 1.0;
  },

  // Ağırlık bazlı kargo ücreti
  getWeightShippingCost(weightKg, carrier) {
    carrier = carrier || "ptt";
    const carrierKey = carrier.toLowerCase().replace(/[\s-]/g, "");

    for (const tier of WEIGHT_SHIPPING) {
      if (weightKg > tier.minKg && weightKg <= tier.maxKg) {
        const keyMap = {
          "ptt": "ptt", "texptt": "tex", "tex": "tex",
          "aras": "aras", "surat": "surat", "sürat": "surat",
          "dhlecommerce": "dhlEcommerce", "dhl": "dhlEcommerce",
          "kolaygelsin": "kolayGelsin", "kolay": "kolayGelsin",
          "yurtici": "yurtici", "yk": "yurtici",
          "ceva": "ceva"
        };

        const key = keyMap[carrierKey] || "ptt";
        const basePrice = tier[key] || tier.ptt;
        const withKDV = basePrice * (1 + KDV_RATE);

        return {
          base: basePrice,
          kdv: withKDV - basePrice,
          total: withKDV,
          carrier: carrier,
          weightRange: `${tier.minKg}-${tier.maxKg} kg`
        };
      }
    }

    const lastTier = WEIGHT_SHIPPING[WEIGHT_SHIPPING.length - 1];
    const basePrice = lastTier.ptt;
    const withKDV = basePrice * (1 + KDV_RATE);
    return { base: basePrice, kdv: withKDV - basePrice, total: withKDV, carrier: carrier, weightRange: "500+ kg" };
  },

  // Ana karlılık hesaplaması
  calculate(params) {
    const {
      salePrice,
      breadcrumbs,
      carrier,
      desi,                       // Desi değeri (kullanıcı girişi veya otomatik)
      reviewCount,
      rating,
      salesMultiplier,
      costPrice,
      manualCommission,
      badges,
      recentSalesData,
      socialProofData,
      kdvRate: paramKdvRate,     // Gerçek KDV oranı (0-1 arası, ör: 0.10, 0.20) — inject.js'den
      categoryTree               // Kategori ağacı [{id, name}, ...] — inject.js'den
    } = params;

    // KDV oranı: parametre > global sabit
    // inject.js'den gelen taxRate (ör: 10, 20) → 0.10, 0.20'ye çevrilmiş olarak gelir
    const effectiveKdvRate = (paramKdvRate !== undefined && paramKdvRate !== null && paramKdvRate >= 0)
      ? paramKdvRate
      : KDV_RATE;

    // Komisyon
    let commissionInfo;
    if (manualCommission !== undefined && manualCommission !== null) {
      commissionInfo = { komisyon: manualCommission, vade: DEFAULT_VADE, match: "Manuel", confidence: 1 };
    } else {
      commissionInfo = this.findCommission(breadcrumbs, categoryTree);
    }

    const commissionRate = commissionInfo.komisyon;
    // 1) Komisyon: KDV DAHİL satış fiyatı üzerinden hesaplanır (Trendyol kuralı)
    const commissionAmount = salePrice * (commissionRate / 100);

    // 2) Kargo (desi verilmişse desi bazlı, yoksa barem)
    // Desi yok + fiyat ≥ 300 TL → kategoriden desi tahmini yap (eski: sabit 3)
    let effectiveDesi = desi;
    let desiEstimated = false;
    if ((!effectiveDesi || effectiveDesi <= 0) && salePrice >= 300) {
      const catPath = (breadcrumbs && breadcrumbs.length > 0) ? breadcrumbs.join(' > ') : null;
      if (typeof estimateDesiFromCategory === 'function') {
        effectiveDesi = estimateDesiFromCategory(catPath);
        desiEstimated = true;
      }
    }
    const shippingInfo = this.getShippingCost(salePrice, carrier, effectiveDesi);
    if (desiEstimated && shippingInfo) {
      shippingInfo.method = 'desi_estimate';
      shippingInfo.estimatedDesi = effectiveDesi;
    }

    // 3) Tahmini satış adedi (çoklu veri kaynağı — panel ile aynı mantık)
    const salesResult = this.estimateMonthlySalesDetailed(reviewCount || 0, rating || 0, salePrice, badges || [], recentSalesData, socialProofData);
    let estimatedMonthlySales = salesResult.count;
    let salesTier = salesResult.tier;
    let salesTierLabel = salesResult.tierLabel;
    const rawOrderCount = salesResult.rawOrderCount || null;
    const orderPeriodDays = salesResult.periodDays || null;

    // ═══════════════════════════════════════════════════════════════
    //  HESAPLAMA SIRASI (Excel formatı)
    //
    //  Trendyol komisyonu KDV DAHİL fiyat üzerinden kesilir.
    //  Ancak karlılık hesabında tüm kalemlerden KDV çıkarılır.
    //
    //  A. Satış fiyatı (KDV dahil)          = salePrice
    //  B. KDV hariç satış fiyatı            = A / (1 + KDV%)
    //  C. Komisyon (KDV dahil)              = A × komisyon%
    //  D. Komisyon (KDV hariç)              = C / (1 + KDV%)
    //  E. Hizmet bedeli (KDV dahil)         = 10.20
    //  F. Hizmet bedeli (KDV hariç)         = E / (1 + KDV%)
    //  G. Kargo (KDV hariç)                 = tablo değeri (zaten KDV hariç)
    //  H. Kargo hesapta                     = G (doğrudan kullan, KDV işlemi yok)
    //  I. Hesaba geçecek (KDV hariç)        = B - D - F - H
    //  J. Maliyet                           = costPrice veya hedef kâra göre
    //  K. Kâr TL                            = I - J
    //  L. Kâr %                             = K / I
    // ═══════════════════════════════════════════════════════════════

    // B: KDV hariç satış fiyatı
    const salePriceExVat = salePrice / (1 + effectiveKdvRate);

    // C: Komisyon KDV dahil fiyat üzerinden (Trendyol kuralı)
    // commissionAmount zaten yukarıda hesaplandı: salePrice × rate%

    // D: Komisyon KDV hariç — karlılık hesabı için KDV'yi çıkar
    const commissionAmountExVat = commissionAmount / (1 + effectiveKdvRate);

    // F: KDV hariç hizmet bedeli
    const hizmetBedeliExVat = HIZMET_BEDELI / (1 + effectiveKdvRate);

    // H: Kargo — tablo zaten KDV hariç rakamlar içeriyor, doğrudan base kullan
    const shippingExVat = shippingInfo.base;

    // I: Hesaba geçecek tutar (KDV hariç) — satıcının eline geçen net tutar
    const hesabaGececek = salePriceExVat - commissionAmountExVat - hizmetBedeliExVat - shippingExVat;

    // J: Maliyet — kullanıcı girişi veya hedef kâr %'ye göre otomatik
    const targetProfitPct = (params.targetProfitPct !== undefined && params.targetProfitPct !== null)
      ? params.targetProfitPct / 100
      : 0.20; // Varsayılan %20 kâr hedefi
    // Hedef kâr %'ye göre maks. maliyet: I × (1 - kâr%) = maliyet
    // Kâr% = K/I = (I-J)/I → J = I × (1 - kâr%)
    const autoEstimatedCost = hesabaGececek * (1 - targetProfitPct);

    const usedCostPrice = (costPrice !== undefined && costPrice !== null && costPrice > 0)
      ? costPrice : autoEstimatedCost;

    // K: Kâr TL
    const profit = hesabaGececek - usedCostPrice;

    // L: Kâr % — hesaba geçecek tutara göre (Excel: K/I)
    const profitMargin = hesabaGececek > 0 ? ((profit / hesabaGececek) * 100) : null;

    // Aylık toplam tahmini kâr
    const monthlyProfit = profit * estimatedMonthlySales;

    // ── Satış aralığı hesapla (tier bazlı) ──
    const salesRange = this.computeSalesRange(estimatedMonthlySales, {
      dataQualityRank: salesResult.dataQualityRank || (socialProofData && socialProofData.dataQualityRank) || 1,
      tier: salesTier || null,
      daysTracked: salesResult.daysTracked || (socialProofData && socialProofData.stockDeltaDaysTracked) || 0,
    });

    // ── Geriye uyumluluk: eski alan adları da döndür ──
    const kalanKDVDahil = salePrice - commissionAmount - HIZMET_BEDELI - shippingInfo.base;
    const kalanKDVHaric = hesabaGececek;

    return {
      salePrice,
      salePriceExVat,
      commission: {
        rate: commissionRate,
        amount: commissionAmount,               // KDV dahil komisyon tutarı (Trendyol kuralı)
        amountExVat: commissionAmountExVat,      // KDV hariç komisyon tutarı (karlılık hesabı)
        match: commissionInfo.match,
        confidence: commissionInfo.confidence,
        vade: commissionInfo.vade
      },
      shipping: shippingInfo,
      shippingExVat,
      hizmetBedeli: HIZMET_BEDELI,
      hizmetBedeliExVat,
      hesabaGececek,
      // Geriye uyumluluk
      kalanKDVDahil,
      kdv: {
        rate: effectiveKdvRate * 100,
        amount: salePrice - salePriceExVat  // Toplam KDV tutarı
      },
      kalanKDVHaric,
      autoEstimatedCost,
      targetProfitPct: targetProfitPct * 100,
      costPrice: (costPrice !== undefined && costPrice !== null && costPrice > 0) ? costPrice : null,
      usedCostPrice,
      profit,
      profitMargin,
      estimatedMonthlySales,
      estimatedMin: salesRange.min,
      estimatedMax: salesRange.max,
      salesSpread: salesRange.spread,
      salesTier,
      salesTierLabel,
      rawOrderCount,
      orderPeriodDays,
      monthlyProfit
    };
  },

  // Fiyat string'ini sayıya çevir ("3.299 TL" -> 3299, "3.299,99 TL" -> 3299.99)
  // "Sepette786,60 TL874 TL" gibi birleşik metinlerde ilk geçerli fiyatı alır
  parsePrice(priceStr) {
    if (!priceStr) return 0;

    // Sayı girildiyse direkt dön
    if (typeof priceStr === 'number') return priceStr;

    const str = String(priceStr).trim();

    // Strateji 1: "TL" ile biten ilk fiyat bloğunu bul
    // "276 TL299 TL" → ["276 TL", "299 TL"] — ilk bloğu al
    // "Sepette142,13 TL" → ["142,13 TL"] — prefix temizlenir
    const tlBlocks = str.match(/[\d.,\s]+TL/gi);
    if (tlBlocks && tlBlocks.length > 0) {
      const block = tlBlocks[0].replace(/TL/i, "").trim();
      // Türk formatı: "3.299,99" → binlik nokta + kuruş virgül
      // Kuruşlu: "142,13" veya "3.299,99"
      const kuruslu = block.match(/^([\d.]+),(\d{2})$/);
      if (kuruslu) {
        const val = parseFloat(kuruslu[1].replace(/\./g, "") + "." + kuruslu[2]);
        if (val > 0) return val;
      }
      // Kuruşsuz: "276" veya "3.299" (binlik ayraçlı)
      const tam = block.match(/^[\d.]+$/);
      if (tam) {
        const val = parseFloat(block.replace(/\./g, ""));
        if (val > 0) return val;
      }
    }

    // Strateji 2: Kuruşlu fiyat — "786,60" kalıbı (TL olmadan)
    const kurusMatch = str.match(/([\d.]+),(\d{2})/);
    if (kurusMatch) {
      const val = parseFloat(kurusMatch[1].replace(/\./g, "") + "." + kurusMatch[2]);
      if (val > 0) return val;
    }

    // Strateji 3: Sadece rakam (TL olmadan)
    const onlyDigits = str.replace(/[^\d.,]/g, "").trim();
    if (onlyDigits) {
      // Virgüllü ama kuruşsuz olabilir: "1,5" → 1.5
      const simple = onlyDigits.replace(/\./g, "").replace(",", ".");
      const val = parseFloat(simple);
      if (val > 0 && val < 1000000) return val; // Mantık kontrolü
    }

    return 0;
  },

  // Sayıyı TL formatına çevir (3299.50 -> "3.299,50 TL")
  formatPrice(num) {
    if (num === null || num === undefined || isNaN(num)) return "-";
    return num.toLocaleString("tr-TR", { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + " TL";
  },

  // Yüzde formatla
  formatPercent(num) {
    if (num === null || num === undefined || isNaN(num)) return "-";
    return "%" + num.toLocaleString("tr-TR", { minimumFractionDigits: 1, maximumFractionDigits: 1 });
  },

  // ═══════════════════════════════════════════════════════════════
  //  SATIŞ ARALIĞI HESAPLAMA (panel ile aynı formül)
  // ═══════════════════════════════════════════════════════════════

  // Kategori bazlı katsayılar — panelden sync ile alınan değerler veya varsayılanlar
  _defaultCategoryCoefficients: {
    'supermarket':       { basketConvRate: 0.15, favoriteMultiplier: 0.003, predictability: 0.7 },
    'kozmetik':          { basketConvRate: 0.14, favoriteMultiplier: 0.004, predictability: 0.6 },
    'ev & mobilya':      { basketConvRate: 0.12, favoriteMultiplier: 0.003, predictability: 0.6 },
    'elektronik':        { basketConvRate: 0.06, favoriteMultiplier: 0.001, predictability: 0.3 },
    'giyim':             { basketConvRate: 0.08, favoriteMultiplier: 0.002, predictability: 0.4 },
    'ayakkabi & canta':  { basketConvRate: 0.08, favoriteMultiplier: 0.002, predictability: 0.4 },
    'aksesuar':          { basketConvRate: 0.09, favoriteMultiplier: 0.002, predictability: 0.5 },
    'spor & outdoor':    { basketConvRate: 0.10, favoriteMultiplier: 0.002, predictability: 0.5 },
    'yapi market & bahce': { basketConvRate: 0.11, favoriteMultiplier: 0.002, predictability: 0.5 },
    'cocuk & bebek':     { basketConvRate: 0.13, favoriteMultiplier: 0.003, predictability: 0.5 },
    'kirtasiye & ofis':  { basketConvRate: 0.10, favoriteMultiplier: 0.002, predictability: 0.5 },
    'otomotiv':          { basketConvRate: 0.07, favoriteMultiplier: 0.001, predictability: 0.4 },
    'kitap & hobi':      { basketConvRate: 0.10, favoriteMultiplier: 0.002, predictability: 0.5 },
    'saglik':            { basketConvRate: 0.12, favoriteMultiplier: 0.003, predictability: 0.6 },
    '_default':          { basketConvRate: 0.10, favoriteMultiplier: 0.002, predictability: 0.5 },
  },

  // Türkçe karakter normalize (kategori eşleştirme için)
  _normalizeTr(str) {
    if (!str) return '';
    return str
      .toLowerCase()
      .replace(/ı/g, 'i').replace(/ğ/g, 'g').replace(/ü/g, 'u')
      .replace(/ş/g, 's').replace(/ö/g, 'o').replace(/ç/g, 'c')
      .replace(/İ/g, 'i').replace(/Ğ/g, 'g').replace(/Ü/g, 'u')
      .replace(/Ş/g, 's').replace(/Ö/g, 'o').replace(/Ç/g, 'c')
      .trim();
  },

  // Kategori katsayılarını getir — panelden sync edilen veya varsayılan
  getCategoryCoefficients(categoryPath) {
    // panelCalibration.category_coefficients varsa onu tercih et
    const coeffMap = (typeof panelCalibration !== 'undefined' && panelCalibration && panelCalibration.category_coefficients)
      ? panelCalibration.category_coefficients
      : this._defaultCategoryCoefficients;

    if (!categoryPath) return coeffMap['_default'] || this._defaultCategoryCoefficients['_default'];

    const topLevel = this._normalizeTr(categoryPath.split('>')[0]);

    // Tam eşleşme
    for (const [key, coeff] of Object.entries(coeffMap)) {
      if (key === '_default') continue;
      if (topLevel === key) return coeff;
    }

    // Kısmi eşleşme
    for (const [key, coeff] of Object.entries(coeffMap)) {
      if (key === '_default') continue;
      if (topLevel.includes(key) || key.includes(topLevel)) return coeff;
    }

    return coeffMap['_default'] || this._defaultCategoryCoefficients['_default'];
  },

  // Nice-number yuvarlama (panel ile aynı)
  _roundNice(n) {
    if (n <= 0) return 0;
    if (n < 10) return Math.round(n);
    if (n < 50) return Math.round(n / 5) * 5;
    if (n < 200) return Math.round(n / 10) * 10;
    if (n < 1000) return Math.round(n / 25) * 25;
    if (n < 5000) return Math.round(n / 100) * 100;
    if (n < 20000) return Math.round(n / 250) * 250;
    return Math.round(n / 500) * 500;
  },

  // Tier bazlı temel spread değeri
  _tierBaseSpread(tier) {
    if (!tier) return 0.35;
    const t = tier.toUpperCase();
    if (t.includes('T0') && t.includes('T1') && t.includes('T3')) return 0.12;
    if (t.includes('T0') && t.includes('T1')) return 0.15;
    if (t.includes('T0') && t.includes('T3')) return 0.18;
    if (t.includes('T1') && t.includes('T3')) return 0.20;
    if (t.includes('PANEL')) return 0.15;
    if (t.startsWith('T0') || t === 'T0~T1') return 0.20;
    if (t.startsWith('T1')) return 0.25;
    if (t.startsWith('T3')) return 0.30;
    return 0.35; // T4, bilinmeyen
  },

  // Rank ve days ile ek daraltma (0.00 - 0.15 arası)
  _qualityDiscount(rank, days) {
    const rankDisc = Math.min(0.09, (Math.max(1, rank) - 1) * 0.01);
    const dayDisc = Math.min(0.06, Math.log2(Math.max(days, 0) + 1) / Math.log2(30) * 0.06);
    return rankDisc + dayDisc;
  },

  // Satış aralığı hesapla — panel ile aynı formül (tier bazlı)
  // monthly: aylık satış tahmini (midpoint)
  // params: { dataQualityRank, tier, daysTracked, signalCount (fallback) }
  computeSalesRange(monthly, params = {}) {
    if (monthly <= 0) return { min: 0, max: 0, spread: 0, confidence: 0 };

    const {
      dataQualityRank = 1,
      tier = null,
      daysTracked = 0,
      signalCount = null,
    } = params;

    let spread;
    if (tier) {
      const base = this._tierBaseSpread(tier);
      const disc = this._qualityDiscount(dataQualityRank, daysTracked);
      spread = Math.max(0.05, base - disc);
    } else {
      const sc = signalCount || 1;
      const base = sc >= 3 ? 0.18 : sc >= 2 ? 0.25 : 0.35;
      const disc = this._qualityDiscount(dataQualityRank, daysTracked);
      spread = Math.max(0.05, base - disc);
    }

    const rawMin = monthly / (1 + spread);
    const rawMax = monthly * (1 + spread);

    return {
      min: Math.max(1, this._roundNice(rawMin)),
      max: this._roundNice(rawMax),
      spread: Math.round(spread * 100),
      confidence: Math.round((1 - spread / 0.40) * 100),
    };
  }
};
