// ==UserScript==
// @name         TK Karlilik Hesaplayici
// @namespace    trendyol-karlilik
// @version      1.0.0
// @description  Trendyol urun listesi sayfasinda karlilik analizi tablosu gosterir
// @match        https://www.trendyol.com/*
// @grant        none
// @run-at       document-idle
// ==/UserScript==

(function() {
  "use strict";

  // ===== Sayfa Turunu Belirle =====
  function isListingPage() {
    var url = window.location.href;
    // Urun detay sayfasi: -p- ardindan rakamlar
    if (/\-p\-\d+/.test(url)) return false;
    // Liste / arama sayfalari: /sr? veya -c ardindan rakamlar veya -x-
    if (/\/sr\?/.test(url) || /\-c\d+/.test(url) || /\-x\-/.test(url)) return true;
    return false;
  }

  // ===== Floating Button Olustur =====
  function createFloatingButton() {
    if (document.getElementById("tk-float-btn")) return;

    var btn = document.createElement("div");
    btn.id = "tk-float-btn";
    btn.textContent = "TK";
    btn.style.cssText = [
      "position:fixed",
      "bottom:20px",
      "right:20px",
      "width:48px",
      "height:48px",
      "border-radius:50%",
      "background:linear-gradient(135deg,#f27a1a,#ff6000)",
      "color:#fff",
      "font-weight:bold",
      "font-size:18px",
      "display:flex",
      "align-items:center",
      "justify-content:center",
      "cursor:pointer",
      "z-index:99998",
      "box-shadow:0 4px 12px rgba(242,122,26,0.5),0 2px 6px rgba(0,0,0,0.3)",
      "user-select:none",
      "transition:transform 0.2s ease",
      "font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif"
    ].join(";") + ";";

    btn.addEventListener("mouseenter", function() {
      btn.style.transform = "scale(1.1)";
    });
    btn.addEventListener("mouseleave", function() {
      btn.style.transform = "scale(1)";
    });

    btn.addEventListener("click", function() {
      toggleTable();
    });

    document.body.appendChild(btn);
  }

  // ===== Toggle Table =====
  function toggleTable() {
    var existing = document.getElementById("tk-overlay");
    if (existing) {
      existing.remove();
      return;
    }
    var products = scrapeProducts();
    if (products.length === 0) {
      alert("Bu sayfada urun karti bulunamadi!\nTrendyol urun listesi veya arama sayfasinda deneyin.");
      return;
    }
    showTable(products);
  }

  // ===== VERI: Komisyon Oranlari =====
  var COMMS = [
    { kategori: "Aksesuar", altKategori: "Altin", urunGrubu: "Islenmemis", vade: 21, komisyon: 9.00 },
    { kategori: "Aksesuar", altKategori: "Altin", urunGrubu: "Islenmis", vade: 21, komisyon: 9.00 },
    { kategori: "Aksesuar", altKategori: "M\u00FCcevher", urunGrubu: "M\u00FCcevher", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "Atk\u0131 & Bere & Eldiven", urunGrubu: "Atk\u0131 & Bere & Eldiven", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "G\u00F6zl\u00FCk", urunGrubu: "G\u00FCne\u015F G\u00F6zl\u00FC\u011F\u00FC", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "G\u00F6zl\u00FCk", urunGrubu: "Optik G\u00F6zl\u00FCk", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "Saat", urunGrubu: "Kol Saati", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "Saat", urunGrubu: "Ak\u0131ll\u0131 Saat", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "\u015Eapka & Kep", urunGrubu: "\u015Eapka & Kep", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "Kemer", urunGrubu: "Kemer", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "C\u00FCzdan", urunGrubu: "C\u00FCzdan", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "Tak\u0131", urunGrubu: "Kolye", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "Tak\u0131", urunGrubu: "Bileklik", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "Tak\u0131", urunGrubu: "K\u00FCpe", vade: 21, komisyon: 21.50 },
    { kategori: "Aksesuar", altKategori: "Tak\u0131", urunGrubu: "Y\u00FCz\u00FCk", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "Ayakkab\u0131", urunGrubu: "Kad\u0131n Ayakkab\u0131", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "Ayakkab\u0131", urunGrubu: "Erkek Ayakkab\u0131", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "Ayakkab\u0131", urunGrubu: "\u00C7ocuk Ayakkab\u0131", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "Ayakkab\u0131", urunGrubu: "Spor Ayakkab\u0131", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "Ayakkab\u0131", urunGrubu: "Sneaker", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "Ayakkab\u0131", urunGrubu: "Bot", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "Ayakkab\u0131", urunGrubu: "Sandalet", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "Ayakkab\u0131", urunGrubu: "Terlik", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "Ayakkab\u0131", urunGrubu: "Topuklu Ayakkab\u0131", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "\u00C7anta", urunGrubu: "Kad\u0131n \u00C7anta", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "\u00C7anta", urunGrubu: "Erkek \u00C7anta", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "\u00C7anta", urunGrubu: "S\u0131rt \u00C7antas\u0131", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "\u00C7anta", urunGrubu: "Bavul & Valiz", vade: 21, komisyon: 21.50 },
    { kategori: "Ayakkab\u0131 & \u00C7anta", altKategori: "\u00C7anta", urunGrubu: "El \u00C7antas\u0131", vade: 21, komisyon: 21.50 },
    { kategori: "Yap\u0131 Market & Bah\u00E7e", altKategori: "Bah\u00E7e", urunGrubu: "Bah\u00E7e Ayd\u0131nlatmas\u0131", vade: 21, komisyon: 20.50 },
    { kategori: "Yap\u0131 Market & Bah\u00E7e", altKategori: "Bah\u00E7e", urunGrubu: "Bah\u00E7e Makineleri", vade: 21, komisyon: 16.00 },
    { kategori: "Yap\u0131 Market & Bah\u00E7e", altKategori: "Bah\u00E7e", urunGrubu: "Bah\u00E7e Mobilyas\u0131", vade: 21, komisyon: 20.50 },
    { kategori: "Yap\u0131 Market & Bah\u00E7e", altKategori: "Bah\u00E7e", urunGrubu: "Bah\u00E7e S\u00FCsleme", vade: 21, komisyon: 20.50 },
    { kategori: "Yap\u0131 Market & Bah\u00E7e", altKategori: "Elektrikli El Aletleri", urunGrubu: "Elektrikli El Aletleri", vade: 21, komisyon: 16.50 },
    { kategori: "Yap\u0131 Market & Bah\u00E7e", altKategori: "H\u0131rdavat", urunGrubu: "H\u0131rdavat", vade: 21, komisyon: 20.50 },
    { kategori: "Yap\u0131 Market & Bah\u00E7e", altKategori: "Boya & Yap\u0131 Kimyasallar\u0131", urunGrubu: "Boya", vade: 21, komisyon: 20.50 },
    { kategori: "Yap\u0131 Market & Bah\u00E7e", altKategori: "Tesisat", urunGrubu: "Tesisat Malzemeleri", vade: 21, komisyon: 20.50 },
    { kategori: "Elektronik", altKategori: "Ak\u0131ll\u0131 Ev", urunGrubu: "Ak\u0131ll\u0131 Ayd\u0131nlatma \u00DCr\u00FCnleri", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "Ak\u0131ll\u0131 Ev", urunGrubu: "Ak\u0131ll\u0131 Ev Sistemleri", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "Telefon", urunGrubu: "Smartphone Android", vade: 14, komisyon: 7.00 },
    { kategori: "Elektronik", altKategori: "Telefon", urunGrubu: "Smartphone iOS", vade: 14, komisyon: 7.00 },
    { kategori: "Elektronik", altKategori: "Telefon", urunGrubu: "Cep Telefonu", vade: 14, komisyon: 7.00 },
    { kategori: "Elektronik", altKategori: "Telefon Aksesuarlar\u0131", urunGrubu: "Telefon K\u0131l\u0131f\u0131", vade: 14, komisyon: 21.50 },
    { kategori: "Elektronik", altKategori: "Telefon Aksesuarlar\u0131", urunGrubu: "\u015Earj Aleti", vade: 14, komisyon: 21.50 },
    { kategori: "Elektronik", altKategori: "Telefon Aksesuarlar\u0131", urunGrubu: "Ekran Koruyucu", vade: 14, komisyon: 21.50 },
    { kategori: "Elektronik", altKategori: "Telefon Aksesuarlar\u0131", urunGrubu: "Kulakl\u0131k", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "Telefon Aksesuarlar\u0131", urunGrubu: "Powerbank", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "Beyaz E\u015Fya", urunGrubu: "Buzdolab\u0131", vade: 14, komisyon: 15.00 },
    { kategori: "Elektronik", altKategori: "Beyaz E\u015Fya", urunGrubu: "\u00C7ama\u015F\u0131r Makinesi", vade: 14, komisyon: 15.00 },
    { kategori: "Elektronik", altKategori: "Beyaz E\u015Fya", urunGrubu: "Bula\u015F\u0131k Makinesi", vade: 14, komisyon: 15.00 },
    { kategori: "Elektronik", altKategori: "Beyaz E\u015Fya", urunGrubu: "F\u0131r\u0131n", vade: 14, komisyon: 15.00 },
    { kategori: "Elektronik", altKategori: "TV", urunGrubu: "Televizyon", vade: 14, komisyon: 15.00 },
    { kategori: "Elektronik", altKategori: "Bilgisayar", urunGrubu: "Laptop", vade: 14, komisyon: 11.00 },
    { kategori: "Elektronik", altKategori: "Bilgisayar", urunGrubu: "Masa\u00FCst\u00FC Bilgisayar", vade: 14, komisyon: 11.00 },
    { kategori: "Elektronik", altKategori: "Bilgisayar", urunGrubu: "Tablet", vade: 14, komisyon: 11.00 },
    { kategori: "Elektronik", altKategori: "Bilgisayar", urunGrubu: "Bilgisayar Aksesuarlar\u0131", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "Bilgisayar", urunGrubu: "Monit\u00F6r", vade: 14, komisyon: 11.00 },
    { kategori: "Elektronik", altKategori: "Bilgisayar", urunGrubu: "Yaz\u0131c\u0131", vade: 14, komisyon: 11.00 },
    { kategori: "Elektronik", altKategori: "Klima", urunGrubu: "Klima", vade: 14, komisyon: 11.00 },
    { kategori: "Elektronik", altKategori: "Klima", urunGrubu: "Is\u0131t\u0131c\u0131", vade: 14, komisyon: 15.00 },
    { kategori: "Elektronik", altKategori: "K\u00FC\u00E7\u00FCk Ev Aletleri", urunGrubu: "\u00DCt\u00FC", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "K\u00FC\u00E7\u00FCk Ev Aletleri", urunGrubu: "Elektrikli S\u00FCp\u00FCrge", vade: 14, komisyon: 15.00 },
    { kategori: "Elektronik", altKategori: "K\u00FC\u00E7\u00FCk Ev Aletleri", urunGrubu: "\u00C7ay Makinesi", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "K\u00FC\u00E7\u00FCk Ev Aletleri", urunGrubu: "Kahve Makinesi", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "K\u00FC\u00E7\u00FCk Ev Aletleri", urunGrubu: "Robot S\u00FCp\u00FCrge", vade: 14, komisyon: 15.00 },
    { kategori: "Elektronik", altKategori: "K\u00FC\u00E7\u00FCk Ev Aletleri", urunGrubu: "Blender", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "K\u00FC\u00E7\u00FCk Ev Aletleri", urunGrubu: "Mutfak Robotu", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "K\u00FC\u00E7\u00FCk Ev Aletleri", urunGrubu: "Tost Makinesi", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "K\u00FC\u00E7\u00FCk Ev Aletleri", urunGrubu: "Sa\u00E7 Kurutma", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "K\u00FC\u00E7\u00FCk Ev Aletleri", urunGrubu: "Sa\u00E7 D\u00FCzle\u015Ftirici", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "Kamera", urunGrubu: "Foto\u011Fraf Makinesi", vade: 14, komisyon: 11.00 },
    { kategori: "Elektronik", altKategori: "Kamera", urunGrubu: "Aksiyon Kamera", vade: 14, komisyon: 11.00 },
    { kategori: "Elektronik", altKategori: "Kamera", urunGrubu: "G\u00FCvenlik Kameras\u0131", vade: 14, komisyon: 19.50 },
    { kategori: "Elektronik", altKategori: "Oyun", urunGrubu: "Oyun Konsolu", vade: 14, komisyon: 11.00 },
    { kategori: "Elektronik", altKategori: "Oyun", urunGrubu: "Oyun Aksesuarlar\u0131", vade: 14, komisyon: 19.50 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "Bebek", urunGrubu: "Bebek Arabas\u0131", vade: 21, komisyon: 19.00 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "Bebek", urunGrubu: "Bebek Gere\u00E7leri", vade: 21, komisyon: 19.00 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "Bebek", urunGrubu: "Bebek Bezi", vade: 21, komisyon: 14.00 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "Bebek", urunGrubu: "Bebek Bak\u0131m", vade: 21, komisyon: 19.00 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "Bebek", urunGrubu: "Bebek Giyim", vade: 21, komisyon: 21.50 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "\u00C7ocuk", urunGrubu: "\u00C7ocuk Giyim", vade: 21, komisyon: 21.50 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "\u00C7ocuk", urunGrubu: "\u00C7ocuk Ayakkab\u0131", vade: 21, komisyon: 21.50 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "Oyuncak", urunGrubu: "Oyuncak Genel", vade: 21, komisyon: 19.00 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "Oyuncak", urunGrubu: "Oyuncak Ah\u015Fap", vade: 21, komisyon: 19.00 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "Oyuncak", urunGrubu: "E\u011Fitici Oyuncak", vade: 21, komisyon: 19.00 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "\u00C7ocuk Gere\u00E7leri", urunGrubu: "\u00C7ocuk Gere\u00E7leri", vade: 21, komisyon: 19.00 },
    { kategori: "\u00C7ocuk & Bebek", altKategori: "Oto Koltu\u011Fu", urunGrubu: "Oto Koltu\u011Fu", vade: 3, komisyon: 19.00 },
    { kategori: "Giyim", altKategori: "Kad\u0131n Giyim", urunGrubu: "Elbise", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Kad\u0131n Giyim", urunGrubu: "Bluz", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Kad\u0131n Giyim", urunGrubu: "G\u00F6mlek", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Kad\u0131n Giyim", urunGrubu: "Ti\u015F\u00F6rt", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Kad\u0131n Giyim", urunGrubu: "Pantolon", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Kad\u0131n Giyim", urunGrubu: "Jean", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Kad\u0131n Giyim", urunGrubu: "Etek", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Kad\u0131n Giyim", urunGrubu: "Ceket", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Kad\u0131n Giyim", urunGrubu: "Mont", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Kad\u0131n Giyim", urunGrubu: "Kazak", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Kad\u0131n Giyim", urunGrubu: "Sweatshirt", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Erkek Giyim", urunGrubu: "Ti\u015F\u00F6rt", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Erkek Giyim", urunGrubu: "G\u00F6mlek", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Erkek Giyim", urunGrubu: "Pantolon", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Erkek Giyim", urunGrubu: "Mont", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Erkek Giyim", urunGrubu: "Sweatshirt", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Spor Giyim", urunGrubu: "Spor Giyim", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "\u0130\u00E7 Giyim", urunGrubu: "Kad\u0131n \u0130\u00E7 Giyim", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "\u0130\u00E7 Giyim", urunGrubu: "Erkek \u0130\u00E7 Giyim", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "\u0130\u00E7 Giyim", urunGrubu: "Pijama", vade: 21, komisyon: 21.50 },
    { kategori: "Giyim", altKategori: "Mayo & Bikini", urunGrubu: "Mayo & Bikini", vade: 21, komisyon: 21.50 },
    { kategori: "Ev & Mobilya", altKategori: "Mobilya", urunGrubu: "Salon Mobilyas\u0131", vade: 28, komisyon: 23.00 },
    { kategori: "Ev & Mobilya", altKategori: "Mobilya", urunGrubu: "Yatak Odas\u0131", vade: 28, komisyon: 23.00 },
    { kategori: "Ev & Mobilya", altKategori: "Mobilya", urunGrubu: "Ofis Mobilyas\u0131", vade: 28, komisyon: 23.00 },
    { kategori: "Ev & Mobilya", altKategori: "Ev Tekstili", urunGrubu: "Nevresim", vade: 21, komisyon: 21.50 },
    { kategori: "Ev & Mobilya", altKategori: "Ev Tekstili", urunGrubu: "Hal\u0131", vade: 21, komisyon: 21.50 },
    { kategori: "Ev & Mobilya", altKategori: "Mutfak", urunGrubu: "Tencere & Tava", vade: 21, komisyon: 21.50 },
    { kategori: "Ev & Mobilya", altKategori: "Mutfak", urunGrubu: "Mutfak Gere\u00E7leri", vade: 21, komisyon: 21.50 },
    { kategori: "Ev & Mobilya", altKategori: "Dekorasyon", urunGrubu: "Duvar Dekorasyon", vade: 21, komisyon: 21.50 },
    { kategori: "Ev & Mobilya", altKategori: "Ayd\u0131nlatma", urunGrubu: "Avize", vade: 21, komisyon: 20.50 },
    { kategori: "Ev & Mobilya", altKategori: "Banyo", urunGrubu: "Banyo Aksesuarlar\u0131", vade: 21, komisyon: 21.50 },
    { kategori: "Kozmetik & Ki\u015Fisel Bak\u0131m", altKategori: "Makyaj", urunGrubu: "Ruj", vade: 21, komisyon: 21.50 },
    { kategori: "Kozmetik & Ki\u015Fisel Bak\u0131m", altKategori: "Cilt Bak\u0131m\u0131", urunGrubu: "Nemlendirici", vade: 21, komisyon: 21.50 },
    { kategori: "Kozmetik & Ki\u015Fisel Bak\u0131m", altKategori: "Sa\u00E7 Bak\u0131m\u0131", urunGrubu: "\u015Eampuan", vade: 21, komisyon: 21.50 },
    { kategori: "Kozmetik & Ki\u015Fisel Bak\u0131m", altKategori: "Parf\u00FCm", urunGrubu: "Kad\u0131n Parf\u00FCm", vade: 21, komisyon: 21.50 },
    { kategori: "Kozmetik & Ki\u015Fisel Bak\u0131m", altKategori: "Ki\u015Fisel Bak\u0131m", urunGrubu: "Di\u015F Bak\u0131m\u0131", vade: 21, komisyon: 20.00 },
    { kategori: "Spor & Outdoor", altKategori: "Spor Giyim", urunGrubu: "Spor Ti\u015F\u00F6rt", vade: 21, komisyon: 21.50 },
    { kategori: "Spor & Outdoor", altKategori: "Spor Ekipman", urunGrubu: "Fitness Ekipman\u0131", vade: 21, komisyon: 19.00 },
    { kategori: "Spor & Outdoor", altKategori: "Spor Ekipman", urunGrubu: "Bisiklet", vade: 21, komisyon: 16.00 },
    { kategori: "Spor & Outdoor", altKategori: "Spor Ekipman", urunGrubu: "Kamp Malzemeleri", vade: 21, komisyon: 19.00 },
    { kategori: "Spor & Outdoor", altKategori: "Spor Ayakkab\u0131", urunGrubu: "Ko\u015Fu Ayakkab\u0131s\u0131", vade: 21, komisyon: 21.50 },
    { kategori: "S\u00FCpermarket", altKategori: "G\u0131da", urunGrubu: "At\u0131\u015Ft\u0131rmal\u0131k", vade: 21, komisyon: 18.00 },
    { kategori: "S\u00FCpermarket", altKategori: "G\u0131da", urunGrubu: "Bakliyat & Makarna", vade: 21, komisyon: 13.00 },
    { kategori: "S\u00FCpermarket", altKategori: "\u0130\u00E7ecek", urunGrubu: "\u00C7ay", vade: 21, komisyon: 18.00 },
    { kategori: "S\u00FCpermarket", altKategori: "Temizlik", urunGrubu: "\u00C7ama\u015F\u0131r Deterjan\u0131", vade: 21, komisyon: 20.00 },
    { kategori: "S\u00FCpermarket", altKategori: "Bebek Bak\u0131m", urunGrubu: "Bebek Bezi", vade: 21, komisyon: 14.00 },
    { kategori: "S\u00FCpermarket", altKategori: "Pet Shop", urunGrubu: "Kedi Mamas\u0131", vade: 21, komisyon: 16.50 },
    { kategori: "Otomotiv", altKategori: "Ara\u00E7 Aksesuarlar\u0131", urunGrubu: "Ara\u00E7 \u0130\u00E7i Aksesuar", vade: 21, komisyon: 19.50 },
    { kategori: "Otomotiv", altKategori: "Ara\u00E7 Bak\u0131m", urunGrubu: "Ara\u00E7 Bak\u0131m \u00DCr\u00FCnleri", vade: 21, komisyon: 19.50 },
    { kategori: "Otomotiv", altKategori: "Yedek Par\u00E7a", urunGrubu: "Yedek Par\u00E7a", vade: 21, komisyon: 16.00 },
    { kategori: "Otomotiv", altKategori: "Lastik & Jant", urunGrubu: "Lastik", vade: 21, komisyon: 11.00 },
    { kategori: "Otomotiv", altKategori: "Lastik & Jant", urunGrubu: "Jant", vade: 21, komisyon: 16.00 },
    { kategori: "K\u0131rtasiye & Ofis", altKategori: "K\u0131rtasiye", urunGrubu: "Kalem", vade: 21, komisyon: 21.50 },
    { kategori: "K\u0131rtasiye & Ofis", altKategori: "Ofis", urunGrubu: "Ofis Malzemeleri", vade: 21, komisyon: 21.50 },
    { kategori: "Kitap & Hobi", altKategori: "Kitap", urunGrubu: "Roman", vade: 21, komisyon: 21.50 },
    { kategori: "Kitap & Hobi", altKategori: "Hobi", urunGrubu: "Puzzle", vade: 21, komisyon: 19.00 },
    { kategori: "Sa\u011Fl\u0131k", altKategori: "Sa\u011Fl\u0131k", urunGrubu: "Takviye Edici G\u0131da", vade: 21, komisyon: 20.00 },
    { kategori: "Sa\u011Fl\u0131k", altKategori: "Sa\u011Fl\u0131k", urunGrubu: "Medikal \u00DCr\u00FCnler", vade: 21, komisyon: 20.00 },
    { kategori: "Sa\u011Fl\u0131k", altKategori: "Eczane", urunGrubu: "Eczane \u00DCr\u00FCnleri", vade: 21, komisyon: 20.00 }
  ];

  var DEFAULT_COMMISSION = 21.50;
  var DEFAULT_VADE = 21;

  // ===== VERI: Kargo =====
  var BAREM_SHIPPING = [
    { minPrice: 0, maxPrice: 149.99, label: "0-149,99 TL", carriers: { "TEX-PTT": 34.16, "ARAS": 42.91, "S\u00DCRAT": 48.74, "KOLAY GELS\u0130N": 51.24, "DHLeCommerce": 82.91, "YK": 104.58 } },
    { minPrice: 150, maxPrice: 299.99, label: "150-299,99 TL", carriers: { "TEX-PTT": 65.83, "ARAS": 73.74, "S\u00DCRAT": 79.58, "KOLAY GELS\u0130N": 82.08, "DHLeCommerce": 82.91, "YK": 104.58 } },
    { minPrice: 300, maxPrice: Infinity, label: "300+ TL (\u00DCcretsiz)", carriers: { "TEX-PTT": 0, "ARAS": 0, "S\u00DCRAT": 0, "KOLAY GELS\u0130N": 0, "DHLeCommerce": 0, "YK": 0 } }
  ];

  var KDV_RATE = 0.20;
  var HIZMET_BEDELI = 10.20;
  var CARRIER = "TEX-PTT";
  var SALES_MULTIPLIER = 5;

  // ===== Yardimci Fonksiyonlar =====
  function parsePrice(str) {
    if (!str) return 0;
    if (typeof str === 'number') return str;
    str = String(str).trim();
    // TL ile biten ilk fiyat bloğunu bul (birleşik metin koruması)
    var tlBlocks = str.match(/[\d.,\s]+TL/gi);
    if (tlBlocks && tlBlocks.length > 0) {
      var block = tlBlocks[0].replace(/TL/i, "").trim();
      var kuruslu = block.match(/^([\d.]+),(\d{2})$/);
      if (kuruslu) {
        var v = parseFloat(kuruslu[1].replace(/\./g, "") + "." + kuruslu[2]);
        if (v > 0) return v;
      }
      var tam = block.match(/^[\d.]+$/);
      if (tam) {
        var v2 = parseFloat(block.replace(/\./g, ""));
        if (v2 > 0) return v2;
      }
    }
    // Kuruşlu fiyat fallback
    var km = str.match(/([\d.]+),(\d{2})/);
    if (km) {
      var v3 = parseFloat(km[1].replace(/\./g, "") + "." + km[2]);
      if (v3 > 0) return v3;
    }
    // Son fallback
    var cleaned = str.replace(/[^\d.,]/g, "").replace(/\./g, "").replace(",", ".");
    var v4 = parseFloat(cleaned);
    return (v4 > 0 && v4 < 1000000) ? v4 : 0;
  }

  function formatPrice(n) {
    if (n == null || isNaN(n)) return "-";
    return n.toLocaleString("tr-TR", { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + " TL";
  }

  function formatPercent(n) {
    if (n == null || isNaN(n)) return "-";
    return "%" + n.toLocaleString("tr-TR", { minimumFractionDigits: 1, maximumFractionDigits: 1 });
  }

  // ===== Komisyon Eslestirme =====
  function findCommission(breadcrumbs) {
    if (!breadcrumbs || breadcrumbs.length === 0) {
      return { komisyon: DEFAULT_COMMISSION, vade: DEFAULT_VADE, match: "Varsay\u0131lan" };
    }
    var searchTerms = breadcrumbs.map(function(b) { return b.toLowerCase().trim(); }).filter(function(b) { return b && b !== "trendyol"; });
    var bestMatch = null, bestScore = 0;

    for (var i = 0; i < COMMS.length; i++) {
      var item = COMMS[i];
      var itemTerms = [item.kategori.toLowerCase(), item.altKategori.toLowerCase(), item.urunGrubu.toLowerCase()];
      var score = 0;

      for (var s = 0; s < searchTerms.length; s++) {
        for (var t = 0; t < itemTerms.length; t++) {
          if (searchTerms[s] === itemTerms[t]) score += 10;
          else if (itemTerms[t].indexOf(searchTerms[s]) >= 0 || searchTerms[s].indexOf(itemTerms[t]) >= 0) score += 5;
          else {
            var sw = searchTerms[s].split(/\s+/);
            var iw = itemTerms[t].split(/\s+/);
            for (var a = 0; a < sw.length; a++) {
              for (var b = 0; b < iw.length; b++) {
                if (sw[a] === iw[b] && sw[a].length > 2) score += 3;
                else if ((sw[a].indexOf(iw[b]) >= 0 || iw[b].indexOf(sw[a]) >= 0) && Math.min(sw[a].length, iw[b].length) > 2) score += 1;
              }
            }
          }
        }
      }
      if (score > bestScore) { bestScore = score; bestMatch = item; }
    }

    if (bestMatch && bestScore > 0) {
      return { komisyon: bestMatch.komisyon, vade: bestMatch.vade, match: bestMatch.kategori + " > " + bestMatch.altKategori + " > " + bestMatch.urunGrubu };
    }
    return { komisyon: DEFAULT_COMMISSION, vade: DEFAULT_VADE, match: "Varsay\u0131lan" };
  }

  // ===== Kargo =====
  function getShippingCost(salePrice) {
    for (var i = 0; i < BAREM_SHIPPING.length; i++) {
      var tier = BAREM_SHIPPING[i];
      if (salePrice >= tier.minPrice && salePrice <= tier.maxPrice) {
        var keys = Object.keys(tier.carriers);
        var matchedKey = keys.find(function(k) { return k === CARRIER; }) || keys[0];
        var basePrice = tier.carriers[matchedKey];
        return { total: basePrice * (1 + KDV_RATE), tier: tier.label };
      }
    }
    return { total: 0, tier: "\u00DCcretsiz" };
  }

  // ===== Breadcrumbs =====
  function getBreadcrumbs() {
    var bcEls = document.querySelectorAll(".breadcrumb a, .product-detail-breadcrumbs-item a");
    var bc = [];
    bcEls.forEach(function(el) {
      var t = el.textContent.trim();
      if (t && t !== "Trendyol") bc.push(t);
    });
    if (bc.length > 0) return bc;

    var parts = window.location.pathname.split("/").filter(function(p) { return p && p !== "sr"; });
    return parts.map(function(p) {
      return p.replace(/-x-.*$/, "").replace(/-c\d+$/, "").replace(/-/g, " ");
    });
  }

  // ===== Urunleri Tara (Yeni Trendyol yapisi: a.product-card) =====
  function scrapeProducts() {
    // Hem yeni hem eski yapiyi destekle
    var cards = document.querySelectorAll("a.product-card");
    if (cards.length === 0) {
      cards = document.querySelectorAll(".p-card-wrppr, [class*='p-card']");
    }

    var products = [];
    var breadcrumbs = getBreadcrumbs();

    cards.forEach(function(card, idx) {
      // Fiyat: yeni yapi (.sale-price) veya eski yapi (.prc-box-dscntd)
      var priceEl = card.querySelector(".sale-price") ||
                    card.querySelector(".prc-box-dscntd") ||
                    card.querySelector(".prc-box-sllng") ||
                    card.querySelector("[class*='prc-box']");
      if (!priceEl) return;
      var price = parsePrice(priceEl.textContent);
      if (!price) return;

      // Marka + Isim: yeni yapi (.product-brand, .product-name) veya eski
      var brandEl = card.querySelector(".product-brand") || card.querySelector(".prdct-desc-cntnr-ttl");
      var nameEl = card.querySelector(".product-name") || card.querySelector(".prdct-desc-cntnr-name");
      var brand = brandEl ? brandEl.textContent.trim() : "";
      var name = nameEl ? nameEl.textContent.trim() : "";
      var fullName = (brand ? brand + " " : "") + (name || "\u00DCr\u00FCn " + (idx + 1));

      // Yorum sayisi: yeni yapi (.total-count) veya eski
      var rcEl = card.querySelector(".total-count") || card.querySelector("[class*='ratingCount']");
      var reviewCount = 0;
      if (rcEl) {
        var m = rcEl.textContent.match(/(\d+)/);
        reviewCount = m ? parseInt(m[0]) : 0;
      }

      // Puan: yeni yapi (.average-rating) veya eski
      var ratEl = card.querySelector(".average-rating") || card.querySelector("[class*='ratingScore']");
      var rating = ratEl ? parseFloat(ratEl.textContent) : 0;

      // Link
      var link = card.href || "";
      if (!link) {
        var linkEl = card.querySelector("a[href]");
        link = linkEl ? linkEl.href : "";
      }

      // Hesapla
      var commInfo = findCommission(breadcrumbs);
      var commRate = commInfo.komisyon;
      var commAmount = price * (commRate / 100);
      var shipping = getShippingCost(price);
      var estSales = reviewCount * SALES_MULTIPLIER;
      var kalanKDVDahil = price - commAmount - HIZMET_BEDELI - shipping.total;
      var kdvAmount = kalanKDVDahil - (kalanKDVDahil / (1 + KDV_RATE));
      var kalanKDVHaric = kalanKDVDahil - kdvAmount;
      var autoEstCost = kalanKDVHaric / 1.20;
      var profit = kalanKDVHaric - autoEstCost;
      var monthlyProfit = profit * estSales;

      products.push({
        idx: idx + 1,
        name: fullName.substring(0, 65),
        price: price,
        rating: rating,
        reviewCount: reviewCount,
        link: link,
        commRate: commRate,
        commAmount: commAmount,
        commMatch: commInfo.match,
        shipTotal: shipping.total,
        shipTier: shipping.tier,
        kalanKDVDahil: kalanKDVDahil,
        kdvAmount: kdvAmount,
        kalanKDVHaric: kalanKDVHaric,
        autoEstCost: autoEstCost,
        profit: profit,
        estSales: estSales,
        monthlyProfit: monthlyProfit
      });
    });

    return products;
  }

  // ===== Tablo Goster =====
  function showTable(products) {
    var overlay = document.createElement("div");
    overlay.id = "tk-overlay";
    overlay.style.cssText = "position:fixed;top:0;left:0;right:0;bottom:0;z-index:999999;background:rgba(0,0,0,0.93);overflow-y:auto;font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif;";

    var html = '<div style="max-width:1500px;margin:20px auto;padding:0 16px;">' +
      '<div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:16px;flex-wrap:wrap;gap:12px;">' +
      '<div><h2 style="color:#f27a1a;font-size:22px;margin:0 0 4px 0;">TK Karl\u0131l\u0131k Hesaplay\u0131c\u0131</h2>' +
      '<span style="font-size:13px;color:#aaa;">' + products.length + ' \u00FCr\u00FCn bulundu | Kargo: ' + CARRIER + ' | Kategori: ' + products[0].commMatch + '</span></div>' +
      '<div style="display:flex;gap:8px;align-items:center;">' +
      '<select id="tk-carrier-sel" style="padding:8px 12px;border-radius:6px;border:1px solid #555;background:#333;color:#fff;font-size:13px;">' +
      '<option value="TEX-PTT"' + (CARRIER === "TEX-PTT" ? " selected" : "") + '>TEX-PTT (En Ucuz)</option>' +
      '<option value="ARAS"' + (CARRIER === "ARAS" ? " selected" : "") + '>Aras Kargo</option>' +
      '<option value="S\u00DCRAT"' + (CARRIER === "S\u00DCRAT" ? " selected" : "") + '>S\u00FCrat Kargo</option>' +
      '<option value="KOLAY GELS\u0130N"' + (CARRIER === "KOLAY GELS\u0130N" ? " selected" : "") + '>Kolay Gelsin</option>' +
      '<option value="DHLeCommerce"' + (CARRIER === "DHLeCommerce" ? " selected" : "") + '>DHL eCommerce</option>' +
      '<option value="YK"' + (CARRIER === "YK" ? " selected" : "") + '>Yurti\u00E7i Kargo</option>' +
      '</select>' +
      '<button id="tk-close-btn" style="padding:10px 28px;border-radius:8px;border:none;background:#e53e3e;color:#fff;font-weight:700;cursor:pointer;font-size:15px;">Kapat (ESC)</button>' +
      '</div></div>' +
      '<div style="overflow-x:auto;border-radius:10px;box-shadow:0 4px 20px rgba(0,0,0,0.3);">' +
      '<table id="tk-table" style="width:100%;border-collapse:collapse;font-size:12px;background:#fff;min-width:1200px;">' +
      '<thead><tr style="background:linear-gradient(135deg,#f27a1a,#ff6000);color:#fff;">' +
      '<th style="padding:12px 8px;font-size:11px;text-align:center;white-space:nowrap;">#</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:left;min-width:220px;">\u00DCr\u00FCn Ad\u0131</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:right;white-space:nowrap;cursor:pointer;" data-sort="price">Fiyat \u2195</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:center;white-space:nowrap;">Puan</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:center;white-space:nowrap;">Yorum</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:right;white-space:nowrap;">Komisyon</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:right;white-space:nowrap;">Kargo</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:right;white-space:nowrap;">Hizmet</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:right;white-space:nowrap;">Kalan(KDV D.)</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:right;white-space:nowrap;">KDV</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:right;white-space:nowrap;background:rgba(255,255,255,0.15);cursor:pointer;" data-sort="kalanKDVHaric">Kalan(KDV H.) \u2195</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:right;white-space:nowrap;">Maks.Maliyet</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:right;white-space:nowrap;background:rgba(255,255,255,0.15);cursor:pointer;" data-sort="profit">K\u00E2r/Sat\u0131\u015F \u2195</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:center;white-space:nowrap;">Ay.Sat\u0131\u015F</th>' +
      '<th style="padding:12px 8px;font-size:11px;text-align:right;white-space:nowrap;background:rgba(255,255,255,0.15);cursor:pointer;" data-sort="monthlyProfit">Ayl\u0131k K\u00E2r \u2195</th>' +
      '</tr></thead><tbody id="tk-tbody"></tbody></table></div></div>';

    overlay.innerHTML = html;
    document.body.appendChild(overlay);
    renderRows(products);

    // Kapat butonu
    document.getElementById("tk-close-btn").addEventListener("click", function() { overlay.remove(); });

    // Kargo degistir
    document.getElementById("tk-carrier-sel").addEventListener("change", function() {
      CARRIER = this.value;
      var newProducts = scrapeProducts();
      renderRows(newProducts);
    });

    // Siralama
    var sortDir = {};
    document.getElementById("tk-table").querySelectorAll("th[data-sort]").forEach(function(th) {
      th.addEventListener("click", function() {
        var key = th.getAttribute("data-sort");
        sortDir[key] = sortDir[key] === "asc" ? "desc" : "asc";
        var dir = sortDir[key];
        products.sort(function(a, b) {
          var va = a[key];
          var vb = b[key];
          return dir === "asc" ? va - vb : vb - va;
        });
        renderRows(products);
      });
    });

    // ESC ile kapat
    document.addEventListener("keydown", function escHandler(e) {
      if (e.key === "Escape") {
        overlay.remove();
        document.removeEventListener("keydown", escHandler);
      }
    });
  }

  function renderRows(products) {
    var tbody = document.getElementById("tk-tbody");
    if (!tbody) return;
    var html = "";
    for (var i = 0; i < products.length; i++) {
      var p = products[i];
      var bgColor = i % 2 === 0 ? "#fff" : "#f9f9f9";
      var profitColor = p.profit >= 0 ? "#00b365" : "#e53e3e";
      var kalanColor = p.kalanKDVHaric >= 0 ? "#00b365" : "#e53e3e";
      var monthlyColor = p.monthlyProfit >= 0 ? "#00b365" : "#e53e3e";

      html += '<tr style="background:' + bgColor + ';border-bottom:1px solid #eee;">' +
        '<td style="padding:8px;color:#999;text-align:center;font-size:11px;">' + p.idx + '</td>' +
        '<td style="padding:8px;">' + (p.link ? '<a href="' + p.link + '" target="_blank" style="color:#333;text-decoration:none;">' : '') +
        '<div style="font-weight:600;max-width:220px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap;" title="' + p.name + '">' + p.name + '</div>' + (p.link ? '</a>' : '') + '</td>' +
        '<td style="padding:8px;text-align:right;font-weight:700;color:#f27a1a;white-space:nowrap;">' + formatPrice(p.price) + '</td>' +
        '<td style="padding:8px;text-align:center;color:#666;">' + (p.rating > 0 ? p.rating.toFixed(1) : "-") + '</td>' +
        '<td style="padding:8px;text-align:center;color:#666;">' + (p.reviewCount || "-") + '</td>' +
        '<td style="padding:8px;text-align:right;color:#e53e3e;white-space:nowrap;font-size:11px;">' + formatPercent(p.commRate) + '<br><span style="font-size:10px;">-' + formatPrice(p.commAmount) + '</span></td>' +
        '<td style="padding:8px;text-align:right;color:#e53e3e;white-space:nowrap;font-size:11px;">-' + formatPrice(p.shipTotal) + '<br><span style="font-size:10px;color:#999;">' + p.shipTier + '</span></td>' +
        '<td style="padding:8px;text-align:right;color:#e53e3e;white-space:nowrap;font-size:11px;">-' + formatPrice(HIZMET_BEDELI) + '</td>' +
        '<td style="padding:8px;text-align:right;white-space:nowrap;font-size:11px;">' + formatPrice(p.kalanKDVDahil) + '</td>' +
        '<td style="padding:8px;text-align:right;color:#e53e3e;white-space:nowrap;font-size:11px;">-' + formatPrice(p.kdvAmount) + '</td>' +
        '<td style="padding:8px;text-align:right;font-weight:700;color:' + kalanColor + ';white-space:nowrap;">' + formatPrice(p.kalanKDVHaric) + '</td>' +
        '<td style="padding:8px;text-align:right;color:#666;white-space:nowrap;font-size:11px;">' + formatPrice(p.autoEstCost) + '</td>' +
        '<td style="padding:8px;text-align:right;font-weight:700;color:' + profitColor + ';white-space:nowrap;">' + formatPrice(p.profit) + '</td>' +
        '<td style="padding:8px;text-align:center;color:#666;">' + p.estSales + '</td>' +
        '<td style="padding:8px;text-align:right;font-weight:700;font-size:13px;color:' + monthlyColor + ';white-space:nowrap;">' + formatPrice(p.monthlyProfit) + '</td></tr>';
    }
    tbody.innerHTML = html;
  }

  // ===== Baslatma: Listing sayfasiysa otomatik button goster =====
  if (isListingPage()) {
    createFloatingButton();
  }

  // SPA navigasyonlari icin URL degisikliklerini izle
  var lastUrl = window.location.href;
  var urlObserver = new MutationObserver(function() {
    if (window.location.href !== lastUrl) {
      lastUrl = window.location.href;
      var existingBtn = document.getElementById("tk-float-btn");
      var existingOverlay = document.getElementById("tk-overlay");
      if (isListingPage()) {
        if (!existingBtn) createFloatingButton();
      } else {
        if (existingBtn) existingBtn.remove();
        if (existingOverlay) existingOverlay.remove();
      }
    }
  });
  urlObserver.observe(document.body, { childList: true, subtree: true });

})();
